 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File:	fraction.tcl
#
# Description:
#	Widget representing a fraction of something, for example
#	a portion of a task.
#
# Global variables:
#c	fraction_priv(af,$w)
#		Active foreground color of window $w
#c	fraction_priv(ao,$w)
#		Active outline color of window $w
#c	fraction_priv(as,$w)
#		Active stipple pattern of window $w
#c	fraction_priv(aw,$w)
#		Active outline width of window $w
#c	fraction_priv(orient,$w)
#		Orientation of window $w
#c	fraction_priv(variable,$w)
#		Variable on display in window $w
#
# Transient procedures:
#c	fraction:update$w
#		Procedure called whenever the value of the variable on
#		display in $w changes its value.

 # fraction.tcl,v 1.1.1.1 1994/12/07 10:17:29 donald Exp
 # /homes/drs1004/repository/tkaux/fraction.tcl,v
 # fraction.tcl,v
# Revision 1.1.1.1  1994/12/07  10:17:29  donald
# First import after some small chanegs.
#
 # Revision 1.11  1994/10/27  18:29:42  kennykb
 # Release 2.0 -- 10-27-94.  To be uploaded to archive sites.
 #
 # Revision 1.10  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.9  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.8  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.7  1993/10/20  18:46:26  kennykb
 # Repaired copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.6  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.5  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.4  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.3  1993/07/20  19:17:12  kennykb
 # Improved structured comments.
 # Changed modules through `g' in the alphabet to follow `:' and `_' naming
 # conventions.
 #
 # Revision 1.2  1993/07/16  15:58:00  kennykb
 # Renamed all commands that start with `wiget.' to either `widget_' or
 # `widget:'.  Added the code for creating composite widgets.
 #
 # Revision 1.1  1993/06/03  15:29:04  kennykb
 # Initial revision
 #

# Procedure:	fraction
#
# Synopsis:
#	Widget indicating a fraction of something, for instance, the portion
#	of a task that has been completed.
#
# Usage:
#c	fraction pathName ?option value?...
#
# Parameters:
#c	pathName
#		Path name of the fraction widget.
#
# Options:
#	Name: 			orientation
#	Class:			Orientation
#	Command-line switch:	-o, -orient, -orientation
#	Default:		`horizontal'
#		One of the two keywords, `horizontal' or `vertical',
#		specifying the orientation of the fraction bar.
#	Name:			variable
#	Class:			Variable
#	Command-line switch:	-v, -var, -variable
#	Default:		Same as pathName
#		Specifies the name of a variable holding the fraction.
#	Name: 			activeForeground
#	Class:			Background
#	Command-line switch:	-activeforeground, -af
#		Refer to the `options' man page
#	Name:			activeOutline
#	Class:			Background
#	Command-line switch:	-ao, -activeoutline
#	Default:		{}
#		Specifies the outline color of the `completion' bar.
#	Name:			activeStipple
#	Class:			Bitmap
#	Command-line switch:	-as, -activestipple
#	Default:		{}
#		Specifies the stipple pattern of the `completion' bar.
#	Name:			activeWidth
#	Class:			LineWidth
#	Command-line switch:	-aw, -activewidth
#	Default:		1.0
#		Specifies the outline width of the `completion bar.
#
#	Other options as for the `canvas' widget.
#
# Description:
#
#	The `fraction' command creates a widget that shows a rectangular
#	bar.  The bar is initially filled with the background color.  
#
#	By assigning a value between 0 and 1 to the variable specified by
#	the `-variable' option, the user can cause the bar to be partially
#	filled with color, and optionally stippled according to the
#	`-activeStipple' parameter.
#
# Bugs:
#	`fraction' is not a first-class widget.  It ought to have a widget
#	command, and respond to `config' requests.

 # Default look and feel:

option add *Fraction*orientation horizontal widgetDefault
option add *Fraction*activeForeground RoyalBlue widgetDefault
option add *Fraction*activeOutline {} widgetDefault
option add *Fraction*activeStipple {} widgetDefault
option add *Fraction*activeWidth 1.0 widgetDefault
option add *Fraction.c.width 160 widgetDefault
option add *Fraction.c.height 16 widgetDefault
option add *Fraction.borderWidth 2 widgetDefault
option add *Fraction.relief sunken widgetDefault

proc fraction {w args} {
	global fraction_priv	

	frame $w -class Fraction

	set variable $w
	set orient [option get $w orientation Orientation]
	set af [option get $w activeForeground Foreground]
	set ao [option get $w activeOutline Background]
	set as [option get $w activeStipple Bitmap]
	set aw [option get $w activeWidth LineWidth]
	set ca {}

	# Parse command line

	while {[llength $args] > 0} {
		set string [lindex $args 0]
		set args [lrange $args 1 end]
		case $string in {
			{-v -var -variable} {
				set variable [lindex $args 0]
				set args [lrange $args 1 end]
			}
			{-af -activeforeground} {
				set af [lindex $args 0]
				set args [lrange $args 1 end]
			}
			{-ao -activeoutline} {
				set ao [lindex $args 0]
				set args [lrange $args 1 end]
			}
			{-as -activestipple} {
				set as [lindex $args 0]
				set args [lrange $args 1 end]
			}
			{-aw -activewidth} {
				set aw [lindex $args 0]
				set args [lrange $args 1 end]
			}
			{-orientation -orient -o} {
				set orient [lindex $args 0]
				set args [lrange $args 1 end]
			}
			default {
				lappend ca $string
				lappend ca [lindex $args 0]
				set args [lrange $args 1 end]
			}
		}
	}

	upvar #0 $variable content
	global $variable
	if ![info exists content] {set content 0.0}

	eval canvas $w.c $ca

	pack append $w $w.c {expand frame center}

	set fraction_priv(variable,$w.c) $variable
	set fraction_priv(orient,$w.c) $orient
	set fraction_priv(af,$w.c) $af
	set fraction_priv(ao,$w.c) $ao
	set fraction_priv(as,$w.c) $as
	set fraction_priv(aw,$w.c) $aw

	proc fraction:update$w args "
		fraction:update $w
	"

	trace variable $variable w fraction:update$w

	fraction:update$w

	widget_addBinding $w Destroy "fraction:destroy $w"

	return $w
}

# Procedure:	fraction:destroy
#
# Synopsis:
#	Internal procedure to clean up when a `fraction' widget is destroyed.
#
# Usage:
#c	fraction:destroy pathName
#
# Parameters:
#c	pathName
#		Name of a `fraction' widget
#
# Return value:
#	None.
#
# Description:
#	`fraction:destroy' is called whenever a `fraction' widget is destroyed.
#	It removes the traces from the variable on display in the widget,
#	and unsets all the global variables that relate to the widget.  It
#	also deletes the transient trace procedure.

proc fraction:destroy w {
	global fraction_priv
	set variable $fraction_priv(variable,$w.c)
	global $variable
	trace vdelete $variable w fraction:update$w
	rename fraction:update$w {}
	foreach item {variable orient af ao as aw} {
		unset fraction_priv($item,$w.c)
	}
}

# Procedure:	fraction:update
#
# Synopsis:
#	Internal procedure to change the display of a fraction widget in
#	response to a change in the associated variable.
#
# Usage:
#c	fraction:update pathName
#
# Parameters:
#c	pathName
#		Path name of a `fraction' widget
#
# Return value:
#	None
#
# Description:
#	`fraction:update' is called whenever the value of the variable on
#	display in a `fraction' window changes its value.  It changes the
#	bar in the window to reflect the new value.
#
# Possible improvement:
#	Make changes so that the events can get bunched.  Basically, this
#	means doing the `coords' operation in an `after' handler.  Possibly
#	a major performance win.

proc fraction:update w {
	global fraction_priv
	upvar #0 $fraction_priv(variable,$w.c) content

	case $fraction_priv(orient,$w.c) in {
		{vertical vert v} {
			set x [lindex [$w.c config -width] 4]
			set y0 [lindex [$w.c config -height] 4]
			set y1 [int $y0*(1.0-$content)]
		}
		default {
			set x [int [lindex [$w.c config -width] 4]*$content]
			set y0 0
			set y1 [lindex [$w.c config -height] 4]
		}
	}

	if {[$w.c find withtag complete] != ""} {
		$w.c coords complete 0 $y0 $x $y1
	} else {
		$w.c create rectangle 0 $y0 $x $y1 \
			-tag complete \
			-fill $fraction_priv(af,$w.c) \
			-outline $fraction_priv(ao,$w.c) \
			-stipple $fraction_priv(as,$w.c) \
			-width $fraction_priv(aw,$w.c)
	}
	update
}
