import sys
import pickle
import time
from PyQt4 import QtGui, QtCore, Qt
import RPi.GPIO as GPIO
import turing

#GPIO Pins initialisation
RunPin = turing.Interface.config.RunPause	#Pin number for Run/Pause button
ContPin = turing.Interface.config.Continue
ResetPin = turing.Interface.config.Reset

WritePin = turing.Interface.config.WriteErase
MoveLeftPin = turing.Interface.config.MoveTapeLeft
MoveRightPin = turing.Interface.config.MoveTapeRight

ProgRedPin = turing.Interface.config.ProgramLEDRed
ProgGreenPin = turing.Interface.config.ProgramLEDGreen

GPIO.setmode(GPIO.BOARD)
GPIO.setup(RunPin, GPIO.IN)		#Run/Pause
GPIO.setup(ContPin, GPIO.IN)		#Continue
GPIO.setup(ResetPin, GPIO.IN)		#Reset

GPIO.setup(WritePin, GPIO.IN)		#Write/Erase
GPIO.setup(MoveLeftPin, GPIO.IN)	#Move tape left
GPIO.setup(MoveRightPin, GPIO.IN)	#Move tape right

GPIO.setup(ProgRedPin, GPIO.OUT)	#Green LED signifying that program is running
GPIO.setup(ProgGreenPin, GPIO.OUT)	#Red LED signifying that simulation has ended
GPIO.output(ProgRedPin, True)		#Initialise LEDs to off
GPIO.output(ProgGreenPin, True)

SwitchLoopOn = True	#Boolean to control while loop for monitoring switches

RunState = 0 		#0 = program not running, 1 = program still running, 2 = program stopped

class SwitchThread(QtCore.QThread):	#Separate thread from GUI to monitor switches

	def __init__(self):

		QtCore.QThread.__init__(self)

	def run(self):

		RunSwitch = False	#True if switch is being pressed, False otherwise
		ContSwitch = False	
		ResetSwitch = False

		WriteSwitch = False
		LeftSwitch = False
		RightSwitch = False

		global RunState
		global SwitchLoopOn

		while SwitchLoopOn:
			
			time.sleep(0.005)
			if SwitchLoopOn == False:
				break
			else:
				
				if RunSwitch == False:		#if switch is not pressed initially
					if GPIO.input(RunPin):	#and it is then pressed
						if RunState in (0, 2):	
							self.emit(QtCore.SIGNAL('runClicked'))		#Run program
						elif RunState == 1:	
							self.emit(QtCore.SIGNAL('pauseClicked'))	#Pause program when it's running
						RunSwitch = True	#Register that switch is now being pressed

				elif RunSwitch:			#if switch is being pressed initially
					if GPIO.input(RunPin) == False:	#and it is then released
						RunSwitch = False	#Register that switch is no longer pressed

				if RunState in (0, 2):		#if program is not currently running

					if ContSwitch == False:
						if GPIO.input(ContPin):
							ContSwitch = True
							self.emit(QtCore.SIGNAL('contClicked'))

					elif ContSwitch:
						if GPIO.input(ContPin) == False:
							ContSwitch = False
		
					if ResetSwitch == False:
						if GPIO.input(ResetPin):
							ResetSwitch = True
							self.emit(QtCore.SIGNAL('resetClicked'))
					
					elif ResetSwitch:
						if GPIO.input(ResetPin) == False:
							ResetSwitch = False
				
					if WriteSwitch == False:
						if GPIO.input(WritePin):
							if turing.tape[turing.head] == 0:	#write '0' if initially 'blank'
								turing.write(1)
							elif turing.tape[turing.head] == 1:	#write '1' if initially '0'
								turing.write(2)
							elif turing.tape[turing.head] == 2:	#write 'blank' if initially '1'
								turing.write(0)
							WriteSwitch = True

					elif WriteSwitch:
						if GPIO.input(WritePin) == False:
							WriteSwitch = False

					if LeftSwitch == False:
						if GPIO.input(MoveLeftPin):
							LeftSwitch = True
							turing.movetapeleft()
				
					elif LeftSwitch:
						if GPIO.input(MoveLeftPin) == False:
							LeftSwitch = False
			
					if RightSwitch == False:
						if GPIO.input(MoveRightPin):
							RightSwitch = True
							turing.movetaperight()

					elif RightSwitch:
						if GPIO.input(MoveRightPin) == False:
							RightSwitch = False

		return

class LeftButton(QtGui.QAbstractButton):
	def __init__(self, parent = None):
		super(LeftButton, self).__init__(parent)
		self.pixmap = QtGui.QPixmap('Left Button Gray.gif')
		self.colour = 0		#current colour, 0: Gray, 1: Blue
		self.setFixedSize(self.sizeHint())

	def paintEvent(self, event):
		painter = QtGui.QPainter(self)
		painter.drawPixmap(event.rect(), self.pixmap)

	def paintBlue(self):
		self.pixmap = QtGui.QPixmap('Left Button Blue.gif')
		self.update()
		self.colour = 1

	def paintGray(self):
		self.pixmap = QtGui.QPixmap('Left Button Gray.gif')
		self.update()
		self.colour = 0
	
	def mousePressEvent(self, event):
		if self.colour == 1:
			self.pixmap = QtGui.QPixmap('Left Button Gray Pressed.gif')
		elif self.colour == 0:
			self.pixmap = QtGui.QPixmap('Left Button Blue Pressed.gif')

		self.update()

	def mouseReleaseEvent(self, event):
		if self.colour == 1:
			self.paintGray()
		elif self.colour == 0:
			self.paintBlue()
		self.emit(QtCore.SIGNAL('LeftInsSignal'))	

	def sizeHint(self):
		return QtCore.QSize(25, 25)

class RightButton(QtGui.QAbstractButton):
	def __init__(self, parent = None):
		super(RightButton, self).__init__(parent)
		self.pixmap = QtGui.QPixmap('Right Button Gray.gif')
		self.colour = 0		#current colour, 0: Gray, 1: Blue
		self.setFixedSize(self.sizeHint())

	def paintEvent(self, event):
		painter = QtGui.QPainter(self)
		painter.drawPixmap(event.rect(), self.pixmap)

	def paintBlue(self):
		self.pixmap = QtGui.QPixmap('Right Button Blue.gif')
		self.update()
		self.colour = 1

	def paintGray(self):
		self.pixmap = QtGui.QPixmap('Right Button Gray.gif')
		self.update()
		self.colour = 0
	
	def mousePressEvent(self, event):
		if self.colour == 1:
			self.pixmap = QtGui.QPixmap('Right Button Gray Pressed.gif')
		elif self.colour == 0:
			self.pixmap = QtGui.QPixmap('Right Button Blue Pressed.gif')

		self.update()

	def mouseReleaseEvent(self, event):
		if self.colour == 1:
			self.paintBlue()
		elif self.colour == 0:
			self.paintGray()
		self.emit(QtCore.SIGNAL('RightInsSignal'))	

	def sizeHint(self):
		return QtCore.QSize(25, 25)

class LEDButton(QtGui.QAbstractButton):
	def __init__(self, parent = None):
		super(LEDButton, self).__init__(parent)
		self.pixmap = QtGui.QPixmap('LED Gray.gif')
		self.colour = 0		#current colour, 0: Gray, 1: Red, 2: Green

	def paintEvent(self, event):
		painter = QtGui.QPainter(self)
		painter.drawPixmap(event.rect(), self.pixmap)

	def paintGray(self):
		self.pixmap = QtGui.QPixmap('LED Gray.gif')
		self.update()
		self.colour = 0

	def paintRed(self):
		self.pixmap = QtGui.QPixmap('LED Red.gif')
		self.update()
		self.colour = 1

	def paintGreen(self):
		self.pixmap = QtGui.QPixmap('LED Green.gif')
		self.update()
		self.colour = 2
	
	def mousePressEvent(self, event):
		if self.colour == 0:
			self.pixmap = QtGui.QPixmap('LED Gray Pressed.gif')
		elif self.colour == 1:
			self.pixmap = QtGui.QPixmap('LED Red Pressed.gif')
		elif self.colour == 2:
			self.pixmap = QtGui.QPixmap('LED Green Pressed.gif')
		
		self.update()

	def mouseReleaseEvent(self, event):
		if self.colour == 0:
			self.paintRed()
		elif self.colour == 1:
			self.paintGreen()
		elif self.colour == 2:
			self.paintGray()
		self.emit(QtCore.SIGNAL('LEDButtonSignal'))	

	def sizeHint(self):
		return QtCore.QSize(30, 30)

class MainWindow(QtGui.QMainWindow):

	def __init__(self):
		super(MainWindow, self).__init__()

		self.initUI()

	def initUI(self):
		
		global switchThread
		global RunState
		global SwitchLoopOn			

		#Status Bar
		self.statusBar().showMessage('Ready')
		self.statusBar().setStyleSheet("background-color: rgb(200, 200, 225);")

		#Central widget
		self.centralWidget = QtGui.QWidget(self)
		self.centralWidget.setStyleSheet("background-color: rgb(200, 200, 225);")
		
		self.Stop = False		#Determines if operation should be terminated
		self.RunStep = 0		#Counts the number of steps run


		#INITIALISATION BUTTONS
		self.PinBox = QtGui.QHBoxLayout()
		self.InitLabel = QtGui.QLabel('INITIALISATION:', self.centralWidget)
		
		self.PinBox.addStretch(1)

		for x in range(turing.reallength):
			vars(self)['PinButton'+str(x)] = LEDButton()		#button representing LED
			vars(self)['PinButton'+str(x)].setToolTip("Toggle to change LED initialisation state")
			vars(self)['PinButton'+str(x)].position = x		#position of button
			vars(self)['PinButton'+str(x)].out = 0			#output of button
			self.connect(vars(self)['PinButton'+str(x)], QtCore.SIGNAL('LEDButtonSignal'), self.PinButtonClicked)
			self.PinBox.addWidget(vars(self)['PinButton'+str(x)])

		self.PinBox.addStretch(1)	

		self.InitButton = QtGui.QPushButton('Initialise Tape')	#Initialise LEDs to defined states
		self.InitButton.clicked.connect(self.InitClicked)
		
		self.ClearButton = QtGui.QPushButton('Clear Tape')	#Turns off all LEDs
		self.ClearButton.clicked.connect(self.ClearClicked)

		self.ResetButton = QtGui.QPushButton('Reset Selection')	#Resets initialisation states of all buttons to 'off'
		self.ResetButton.clicked.connect(self.PinReset)

		self.InitBox = QtGui.QHBoxLayout()
		self.InitBox.addStretch(1)			
		self.InitBox.addWidget(self.InitButton)
		self.InitBox.addWidget(self.ClearButton)
		self.InitBox.addWidget(self.ResetButton)			
		self.InitBox.addStretch(1)	
	
	
		#TABLE OF INSTRUCTIONS/STATE TABLE

		#Table Title
		self.TableTitle = QtGui.QLabel('STATE TABLE:')

		#Creates Table Headings (View 1)			
		self.tabletop = QtGui.QGridLayout()
		self.tabletop.setSpacing(10)

		self.StateColumn = 0

		self.StateTitle = QtGui.QLabel('STATE')
		self.StateSymbolTitle = QtGui.QLabel('STATE      \      SYMBOL')
		self.tabletop.addWidget(self.StateSymbolTitle, 0, self.StateColumn)
		
		for x in range(turing.symbolno):				
			vars(self)['Symbol' + turing.symbols[x] + 'Column'] = 1 + x 
			vars(self)['SymbolTitle' + turing.symbols[x]] = QtGui.QLabel(turing.symbols[x])
			vars(self)['SymbolTitle' + turing.symbols[x]].setAlignment(QtCore.Qt.AlignCenter)
			self.tabletop.addWidget(vars(self)['SymbolTitle' + turing.symbols[x]], 0, 1 + x)

		#Creates Table (View 1)
		self.table = QtGui.QGridLayout()			#Makes actual Table layout
		self.table.setSpacing(10)		

		#Creates Table Headings (View 2)
		self.tabletop1 = QtGui.QGridLayout()
		self.tabletop1.setSpacing(10)

		self.StateTitle1 = QtGui.QLabel('State')
		self.SymbolTitle1 = QtGui.QLabel('Symbol')
		self.WriteTitle1 = QtGui.QLabel('Write Symbol')
		self.MoveTitle1 = QtGui.QLabel('Move Tape')
		self.NextTitle1 = QtGui.QLabel('Next State')
			
		self.StateColumn1 = 0		#State column in table
		self.SymbolColumn1 = 1		#Symbol column in table
		self.WriteColumn1 = 2		#Write/Erase column in table
		self.MoveColumn1 = 3		#Move left/right column in table
		self.NextColumn1 = 4		#Next state column in table

		self.tabletop1.addWidget(self.StateTitle1, 0, self.StateColumn1)
		self.tabletop1.addWidget(self.SymbolTitle1, 0, self.SymbolColumn1)
		self.tabletop1.addWidget(self.WriteTitle1, 0, self.WriteColumn1)
		self.tabletop1.addWidget(self.MoveTitle1, 0, self.MoveColumn1)
		self.tabletop1.addWidget(self.NextTitle1, 0, self.NextColumn1)

		self.Write = 0			#Defines order of Write, Move, Next
		self.Move = self.Write + 1
		self.NextState = self.Move + 1

		#Creates Table (View 2)		
		self.table1 = QtGui.QGridLayout()			#Actual Table layout
		self.table1.setSpacing(10)

		#Initialises tables (both views)
		self.ResetTable()	#Initialises table
		self.statusBar().showMessage(' ')	#Clears status bar

		#Once tables are initialised, frame them as widgets in scroll areas
		#View 1
		self.scrollArea = QtGui.QScrollArea()			#Makes scroll area for table
		self.scrollArea.setWidgetResizable(True)

		self.tableWidget = QtGui.QWidget()			#Makes Table Widget within scroll area	
		self.tableWidget.setStyleSheet("background-color: rgb(235, 235, 235);")

		self.tablebox = QtGui.QVBoxLayout(self.tableWidget)	#Widget VBox layout

		self.tablebox.addLayout(self.table)	#Adds table to Table Widget
		self.tablebox.addStretch(1)		#Fills remaining space of Table Widget
		self.scrollArea.setWidget(self.tableWidget)	#Adds Table Widget to scroll area

		#View 2
		self.scrollArea1 = QtGui.QScrollArea()			#Makes scroll area for table
		self.scrollArea1.setWidgetResizable(True)

		self.tableWidget1 = QtGui.QWidget()			#Makes Table Widget within scroll area
		self.tableWidget1.setStyleSheet("background-color: rgb(235, 235, 235);")
		
		self.tablebox1 = QtGui.QVBoxLayout(self.tableWidget1)	#Widget VBox layout

		self.tablebox1.addLayout(self.table1)	#Adds table to Table Widget
		self.tablebox1.addStretch(1)		#Fills remaining space of Table Widget	
		self.scrollArea1.setWidget(self.tableWidget1)	#Adds Table Widget to scroll area


		#Tab for View 1
		self.Tab1 = QtGui.QWidget()
		self.Tab1Layout = QtGui.QVBoxLayout(self.Tab1)
		self.Tab1Layout.addLayout(self.tabletop)
		self.Tab1Layout.addWidget(self.scrollArea)	

		#Tab for View 2
		self.Tab2 = QtGui.QWidget()
		self.Tab2Layout = QtGui.QVBoxLayout(self.Tab2)
		self.Tab2Layout.addLayout(self.tabletop1)
		self.Tab2Layout.addWidget(self.scrollArea1)	
		
		#Adds Tabs to a tab widget
		self.TabWidget = QtGui.QTabWidget()
		self.TabWidget.addTab(self.Tab1, 'View 1')
		self.TabWidget.addTab(self.Tab2, 'View 2')
	

		#TABLE BUTTONS
		self.TableButtons = QtGui.QHBoxLayout()
		self.TableAdd = QtGui.QPushButton('Add State')
		self.TableRemove = QtGui.QPushButton('Remove State')
		self.TableSave = QtGui.QPushButton('Save Program')
		self.TableLoad = QtGui.QPushButton('Load Program')
		self.TableReset = QtGui.QPushButton('Reset Program')
		self.TableClear = QtGui.QPushButton('Clear Program')

		self.TableAdd.clicked.connect(self.AddStateClicked)
		self.TableRemove.clicked.connect(self.RemoveStateClicked)
		self.TableSave.clicked.connect(self.SaveTable)
		self.TableLoad.clicked.connect(self.LoadTable)
		self.TableReset.clicked.connect(self.ResetTable)
		self.TableClear.clicked.connect(self.ClearTable)

		self.TableButtons.addStretch(1)
		self.TableButtons.addWidget(self.TableAdd)
		self.TableButtons.addWidget(self.TableRemove)
		self.TableButtons.addWidget(self.TableSave)
		self.TableButtons.addWidget(self.TableLoad)
		self.TableButtons.addWidget(self.TableClear)
		self.TableButtons.addWidget(self.TableReset)
		

		#ADDS THE DIFFERENT SECTIONS TO THE CENTRAL WIDGET	
		self.vbox = QtGui.QVBoxLayout(self.centralWidget)
		self.vbox.addWidget(self.InitLabel)
		self.vbox.addLayout(self.PinBox)
		self.vbox.addLayout(self.InitBox)
		self.vbox.addSpacing(20)
		self.vbox.addWidget(self.TableTitle)
		self.vbox.addWidget(self.TabWidget)
		self.vbox.addLayout(self.TableButtons)
		
		self.centralWidget.setLayout(self.vbox)
		self.setCentralWidget(self.centralWidget)
	
		#Customizes window
		self.setGeometry(25, 50, 1100, 800)
		self.setMinimumWidth(1100)
		self.setWindowTitle('Turing Machine Program')

		self.show()
		
		#Links signals from switch-monitoring thread to functions
		self.connect(switchThread, QtCore.SIGNAL('runClicked'), self.RunClicked)
		self.connect(switchThread, QtCore.SIGNAL('pauseClicked'), self.StopClicked)
		self.connect(switchThread, QtCore.SIGNAL('contClicked'), self.ContClicked)
		self.connect(switchThread, QtCore.SIGNAL('resetClicked'), self.ResetClicked)

	def TableInit(self):
	#Initialise Nested List which stores State Table information			 

		self.TableInfo = [self.stateno, turing.symbolno]	#First 2 elements are no. of states and no. of symbols
		
		for x in range(self.stateno):			
			self.TableInfo.append([])		#Create a list for each state

			for y in range(turing.symbolno):	
				self.TableInfo[2+x].append([0, 0, 0])	#Create a list for for each symbol (Write/Move/Next State)
		#print 'Table Initialised as ', self.TableInfo

	def PinButtonClicked(self):
	#Changes initial tape configuration (in turing.py) when LED buttons are toggled

		sender = self.sender()
		if sender.out == 0:
			sender.out = 1 
			turing.inittape[sender.position] = 1
		elif sender.out == 1:
			sender.out = 2 
			turing.inittape[sender.position] = 2
		elif sender.out == 2:	
			sender.out = 0 
			turing.inittape[sender.position] = 0

	def InitClicked(self):
	#Calls turing.py to display LEDs as specified by the configured initial tape

		turing.initialise()
		self.statusBar().showMessage('Tape initialised.')		
		self.RunStep = 0

	def ClearClicked(self):
	#Calls turing.py to clear the tape and display the LEDs (off)

		turing.tapeclear()
		turing.printtape()
		self.statusBar().showMessage('Tape cleared.')
		self.RunStep = 0

	def PinReset(self):
	#Resets all the initialisation buttons (gray) and resets initial tape

		for x in range(turing.reallength):
			vars(self)['PinButton'+str(x)].paintGray()
			vars(self)['PinButton'+str(x)].out = 0
			turing.inittape[vars(self)['PinButton'+str(x)].position] = 0

	def WriteChanged(self):
	#Action when any write instruction in state table is changed

		if self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] == 0:
			vars(self)[str(self.currentState)+'WriteInstruction'+str(self.currentSymbol)].setStyleSheet("background-color: lightGray;")

		elif self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] == 1:
			vars(self)[str(self.currentState)+'WriteInstruction'+str(self.currentSymbol)].setStyleSheet("background-color: rgb(200, 20, 20); color: rgb(235, 235, 235)")

		elif self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] == 2:
			vars(self)[str(self.currentState)+'WriteInstruction'+str(self.currentSymbol)].setStyleSheet("background-color: rgb(70, 150, 70); color: rgb(235, 235, 235)" )

		if self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] == 0:
			vars(self)[str(self.currentState)+'WriteInstruction'+str(self.currentSymbol)].setText('Write B')
		else:
			vars(self)[str(self.currentState)+'WriteInstruction'+str(self.currentSymbol)].setText('Write ' + str(turing.symbols[self.TableInfo[2+self.currentState][self.currentSymbol][self.Write]] ))

		vars(self)[str(self.currentState)+'WriteOption'+str(self.currentSymbol)].setCurrentIndex(self.TableInfo[2+self.currentState][self.currentSymbol][self.Write])

	def MoveChanged(self):
	#Action when any move instruction in state table is changed

		if self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] == 0:
			vars(self)[str(self.currentState)+'MoveLeftInstruction'+str(self.currentSymbol)].paintGray()
			vars(self)[str(self.currentState)+'MoveRightInstruction'+str(self.currentSymbol)].paintGray()
		elif self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] == 1:
			vars(self)[str(self.currentState)+'MoveLeftInstruction'+str(self.currentSymbol)].paintBlue()
			vars(self)[str(self.currentState)+'MoveRightInstruction'+str(self.currentSymbol)].paintGray()
		elif self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] == 2:		
			vars(self)[str(self.currentState)+'MoveLeftInstruction'+str(self.currentSymbol)].paintGray()
			vars(self)[str(self.currentState)+'MoveRightInstruction'+str(self.currentSymbol)].paintBlue()

		vars(self)[str(self.currentState)+'MoveOption'+str(self.currentSymbol)].setCurrentIndex(self.TableInfo[2+self.currentState][self.currentSymbol][self.Move])

	def NextStateChanged(self):
	#Action when any next state instruction in state table is changed

		vars(self)[str(self.currentState)+'NextStateOption'+str(self.currentSymbol)].setCurrentIndex(self.TableInfo[2+self.currentState][self.currentSymbol][self.NextState])
		vars(self)[str(self.currentState)+'NextStateInstruction'+str(self.currentSymbol)].setCurrentIndex(self.TableInfo[2+self.currentState][self.currentSymbol][self.NextState])	

	def WriteSelected(self):
	#Action when any write instruction is changed in view 2

		sender = self.sender()
		sender.Option = sender.currentIndex()
		self.currentState = sender.State
		self.currentSymbol = sender.Symbol
		self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] = sender.Option
		
		self.WriteChanged()

	def MoveSelected(self):
	#Action when any move instruction is changed in view 2

		sender = self.sender()
		sender.Option = sender.currentIndex()
		self.currentState = sender.State
		self.currentSymbol = sender.Symbol
		self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] = sender.Option
		
		self.MoveChanged()
	
	def NextStateSelected(self):
	#Action when any next state instruction is changed in view 2

		sender = self.sender()
		sender.Option = sender.currentIndex()
		self.currentState = sender.State
		self.currentSymbol = sender.Symbol
		self.TableInfo[2+sender.State][sender.Symbol][self.NextState]= sender.Option

		self.NextStateChanged()

	def AddState(self):
	#Changes made to GUI displayed when a state is added

		#View 1
		if self.stateno == 0:
			self.StateLabel = QtGui.QLabel('State ' + str(self.stateno) + ' (Start)')
		else:
			self.StateLabel = QtGui.QLabel('State ' + str(self.stateno))
		self.table.addWidget(self.StateLabel, self.stateno, self.StateColumn)

		for y in range(turing.symbolno):
			vars(self)[str(self.stateno)+'InstructionWidget'+str(y)] = QtGui.QWidget()
			vars(self)[str(self.stateno)+'InstructionBox'+str(y)] = QtGui.QHBoxLayout(vars(self)[str(self.stateno)+'InstructionWidget'+str(y)])
	
			vars(self)[str(self.stateno)+'MoveLeftInstruction'+str(y)] = LeftButton()
			vars(self)[str(self.stateno)+'MoveLeftInstruction'+str(y)].setToolTip("Move tape left.")
			vars(self)[str(self.stateno)+'MoveLeftInstruction'+str(y)].State = self.stateno
			vars(self)[str(self.stateno)+'MoveLeftInstruction'+str(y)].Symbol = y
			self.connect(vars(self)[str(self.stateno)+'MoveLeftInstruction'+str(y)], QtCore.SIGNAL('LeftInsSignal'), self.LeftInsClicked)

			vars(self)[str(self.stateno)+'WriteInstruction'+str(y)] = QtGui.QPushButton('Write B')
			vars(self)[str(self.stateno)+'WriteInstruction'+str(y)].setStyleSheet("background-color: lightGray;")
			vars(self)[str(self.stateno)+'WriteInstruction'+str(y)].setToolTip("Write a different symbol.")
			vars(self)[str(self.stateno)+'WriteInstruction'+str(y)].State = self.stateno
			vars(self)[str(self.stateno)+'WriteInstruction'+str(y)].Symbol = y
			vars(self)[str(self.stateno)+'WriteInstruction'+str(y)].clicked.connect(self.WriteInsClicked)

			vars(self)[str(self.stateno)+'MoveRightInstruction'+str(y)] = RightButton()
			vars(self)[str(self.stateno)+'MoveRightInstruction'+str(y)].setToolTip("Move tape right.")
			vars(self)[str(self.stateno)+'MoveRightInstruction'+str(y)].State = self.stateno
			vars(self)[str(self.stateno)+'MoveRightInstruction'+str(y)].Symbol = y
			self.connect(vars(self)[str(self.stateno)+'MoveRightInstruction'+str(y)], QtCore.SIGNAL('RightInsSignal'), self.RightInsClicked)

			vars(self)[str(self.stateno)+'GoToLabel'+str(y)] = QtGui.QLabel('Go to')
			
			vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)] = QtGui.QComboBox()
			vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)].setStyleSheet("background-color: lightGray;")
			vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)].setToolTip("Choose next state.")

			for c in range(self.stateno + 1):
				vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)].addItem(str(c))
			vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)].addItem('Stop')	
			vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)].State = self.stateno
			vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)].Symbol = y
			vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)].Option = 0
			vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)].activated[str].connect(self.NextStateSelected)
			
			vars(self)[str(self.stateno)+'InstructionBox'+str(y)].addWidget(vars(self)[str(self.stateno)+'MoveLeftInstruction'+str(y)])
			vars(self)[str(self.stateno)+'InstructionBox'+str(y)].addWidget(vars(self)[str(self.stateno)+'WriteInstruction'+str(y)])
			vars(self)[str(self.stateno)+'InstructionBox'+str(y)].addWidget(vars(self)[str(self.stateno)+'MoveRightInstruction'+str(y)])
			vars(self)[str(self.stateno)+'InstructionBox'+str(y)].addWidget(vars(self)[str(self.stateno)+'GoToLabel'+str(y)])
			vars(self)[str(self.stateno)+'InstructionBox'+str(y)].addWidget(vars(self)[str(self.stateno)+'NextStateInstruction'+str(y)])

			self.table.addWidget(vars(self)[str(self.stateno)+'InstructionWidget'+str(y)], self.stateno, 1 + y)
		
		#View 2
		if self.stateno == 0:
			self.StateLabel1 = QtGui.QLabel('State ' + str(self.stateno) + ' (Start)')
		else:
			self.StateLabel1 = QtGui.QLabel('State ' + str(self.stateno))
		self.table1.addWidget(self.StateLabel1, (self.stateno)*turing.symbolno, self.StateColumn1)
	
		for y in range(turing.symbolno):
			self.SymbolLabel1 = QtGui.QLabel(turing.symbols[y])
			
			vars(self)[str(self.stateno)+'WriteOption'+str(y)] = QtGui.QComboBox()
	
			for w in range(turing.symbolno):
				vars(self)[str(self.stateno)+'WriteOption'+str(y)].addItem('Write ' + turing.symbols[w])
			vars(self)[str(self.stateno)+'WriteOption'+str(y)].State = (self.stateno)
			vars(self)[str(self.stateno)+'WriteOption'+str(y)].Symbol = y
			vars(self)[str(self.stateno)+'WriteOption'+str(y)].Option = 0
			vars(self)[str(self.stateno)+'WriteOption'+str(y)].activated[str].connect(self.WriteSelected)

			vars(self)[str(self.stateno)+'MoveOption'+str(y)] = QtGui.QComboBox()
			vars(self)[str(self.stateno)+'MoveOption'+str(y)].addItem("Don't move tape")
			vars(self)[str(self.stateno)+'MoveOption'+str(y)].addItem('Move tape left')
			vars(self)[str(self.stateno)+'MoveOption'+str(y)].addItem('Move tape right')
			vars(self)[str(self.stateno)+'MoveOption'+str(y)].State = (self.stateno)
			vars(self)[str(self.stateno)+'MoveOption'+str(y)].Symbol = y
			vars(self)[str(self.stateno)+'MoveOption'+str(y)].Option = 0
			vars(self)[str(self.stateno)+'MoveOption'+str(y)].activated[str].connect(self.MoveSelected)

			vars(self)[str(self.stateno)+'NextStateOption'+str(y)] = QtGui.QComboBox()	

			for c in range(self.stateno + 1):
				vars(self)[str(self.stateno)+'NextStateOption'+str(y)].addItem(str(c))
			vars(self)[str(self.stateno)+'NextStateOption'+str(y)].addItem('Stop')	
			vars(self)[str(self.stateno)+'NextStateOption'+str(y)].State = self.stateno
			vars(self)[str(self.stateno)+'NextStateOption'+str(y)].Symbol = y
			vars(self)[str(self.stateno)+'NextStateOption'+str(y)].Option = 0
			vars(self)[str(self.stateno)+'NextStateOption'+str(y)].activated[str].connect(self.NextStateSelected)

			self.table1.addWidget(self.SymbolLabel1, (self.stateno)*turing.symbolno + y, self.SymbolColumn1)
			self.table1.addWidget(vars(self)[str(self.stateno)+'WriteOption'+str(y)], self.stateno*turing.symbolno + y, self.WriteColumn1)
			self.table1.addWidget(vars(self)[str(self.stateno)+'MoveOption'+str(y)], self.stateno*turing.symbolno + y, self.MoveColumn1)
			self.table1.addWidget(vars(self)[str(self.stateno)+'NextStateOption'+str(y)], self.stateno*turing.symbolno + y, self.NextColumn1)			

		#Increment stateno				
		self.stateno = self.stateno + 1

		#Adds additional state to existing Next State dropdown menus (combobox)	
		for a in range(self.stateno - 1):
			for b in range(turing.symbolno):
				vars(self)[str(a)+'NextStateOption'+str(b)].insertItem(self.stateno - 1, str(self.stateno - 1))
				vars(self)[str(a)+'NextStateInstruction'+str(b)].insertItem(self.stateno - 1, str(self.stateno - 1))	
		
	def AddStateClicked(self):
	#Adds a row to GUI and includes a new list in state table information

		#Edits GUI displayed
		self.AddState()
		for a in range(self.stateno - 1):
			for b in range(turing.symbolno):
				if self.TableInfo[2 + a][b][self.NextState] == self.stateno - 1:
					self.TableInfo[2 + a][b][self.NextState] = self.TableInfo[2 + a][b][self.NextState] + 1					

		#Edits state table information		
		self.TableInfo.append([])
		for y in range(turing.symbolno):	#List for for each symbol (Write/Move/Next State)
			self.TableInfo[1+self.stateno].append([0, 0, 0])
		self.TableInfo[0] = self.stateno

		self.statusBar().showMessage('State added.')

	def RemoveStateClicked(self):
	#Removes a row from GUI and updates the state table information

		if self.stateno > 1:	#If there is currently more than 1 state
			self.stateno = self.stateno - 1		#Reduces number of states
			
			for x in range(13):	#Deletes widgets from View 1
				self.table1.itemAt(self.table1.count() - 1 - x).widget().deleteLater()

			for x in range(4):	#Deletes widgets from View 2
				self.table.itemAt(self.table.count() - 1 - x).widget().deleteLater()

			for a in range(self.stateno):		#Removes state in dropdown menus
				for b in range(turing.symbolno):
					vars(self)[str(a)+'NextStateOption'+str(b)].removeItem(self.stateno)
					vars(self)[str(a)+'NextStateInstruction'+str(b)].removeItem(self.stateno)

			#Edits state table information
			self.TableInfo[0] = self.stateno	
			for a in range(self.stateno):
				for b in range(turing.symbolno):
					if self.TableInfo[2 + a][b][self.NextState] == self.stateno + 1:
						self.TableInfo[2 + a][b][self.NextState] = self.TableInfo[2 + a][b][self.NextState] - 1		#Changes index for Stop State

			self.TableInfo.pop()	#Removes last list of instructions from state table information

	def ResetTable(self):
	#Resets table to have only 1 state

		self.stateno = 0	#Resets no. of states to 0
		
		self.TableInfo = [self.stateno, turing.symbolno]	#Creates nested list which stores state table information
		
		for x in range(self.table.count() - 1):	#remove all widgets from view 1
			self.table.itemAt(x + 1).widget().deleteLater()
		
		for x in range(self.table1.count()):	#remove all widgets from view 2
			self.table1.itemAt(x).widget().deleteLater()
		
		self.AddStateClicked()		#Adds 1 state

		self.statusBar().showMessage('Program reset.')

	def SaveTable(self):
	#Saves table to a program file

		fname = QtGui.QFileDialog.getSaveFileName(self, 'Save file', '/home/pi/TM')
		f = open(fname, 'w')
		with f:
			pickle.dump(self.TableInfo, f)
		f.close()

		self.statusBar().showMessage('Program saved.')

	def LoadTable(self):
	#Loads table from a program file

		fname = QtGui.QFileDialog.getOpenFileName(self, 'Open file', '/home/pi/TM')
		f = open(fname, 'r')
		with f:
			self.ResetTable()
			self.statusBar().showMessage('Loading program...')
			self.TableInfo = pickle.load(f)
		f.close()		

		for z in range(self.TableInfo[0]-1):
			self.AddState()
			
		for x in range(self.stateno):
			for y in range(turing.symbolno):
				self.currentState = x
				self.currentSymbol = y
				self.WriteChanged()
				self.MoveChanged()
				self.NextStateChanged()

		self.statusBar().showMessage('Program loaded.')

	def ClearTable(self):
	#Restores all table information to default values

		self.TableInit()
		for x in range(self.stateno):
			for y in range(turing.symbolno):
				self.TableInfo[2+x][y] = [0, 0, 0]
				self.currentState = x
				self.currentSymbol = y
				self.WriteChanged()
				self.MoveChanged()
				self.NextStateChanged()

		self.statusBar().showMessage('Program cleared.')

	def LeftInsClicked(self):
	#Updates state table information and GUI when move tape left option in view 1 is selected

		sender = self.sender()
		self.currentState = sender.State
		self.currentSymbol = sender.Symbol
		if self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] in (0, 2):
			self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] = 1			
		else:
			self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] = 0

		self.MoveChanged()

	def RightInsClicked(self):
	#Updates state table information and GUI when move tape right option in view 1 is selected

		sender = self.sender()
		self.currentState = sender.State
		self.currentSymbol = sender.Symbol
		if self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] in (0, 1):
			self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] = 2			
		else:
			self.TableInfo[2+self.currentState][self.currentSymbol][self.Move] = 0

		self.MoveChanged()

	def WriteInsClicked(self):
	#Updates state table information and GUI when any write instruction is changed in view 1

		sender = self.sender()
		self.currentState = sender.State
		self.currentSymbol = sender.Symbol

		if self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] in (0, 1):
			self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] = self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] + 1
			
		elif self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] == 2:
			self.TableInfo[2+self.currentState][self.currentSymbol][self.Write] = 0

		self.WriteChanged()

	def NextStep(self):
	#Runs the next step as instructed by the program

		global RunState
		self.RunStep = self.RunStep + 1

		CurrentState = turing.state
		CurrentHead = turing.tape[turing.head]
		turing.write(self.TableInfo[2+CurrentState][CurrentHead][self.Write])	#writes new symbol
		time.sleep(self.lag)
		turing.move(self.TableInfo[2+CurrentState][CurrentHead][self.Move])	#moves tape
		time.sleep(self.lag)
		turing.state = self.TableInfo[2+CurrentState][CurrentHead][self.NextState]	#changes current state

		self.statusBar().showMessage('Running simulation step ' + str(self.RunStep) + ". Current state is '" + str(turing.state) + "'.")
		RunState = 1
		self.ProgramLED(RunState)	#Turns Program LED green 

	def RunClicked(self):
	#Runs program

		global RunState
		self.lag = turing.Interface.config.RunDelay		#sets delay in between Turing machine operations (write/move)

		while turing.state < self.stateno:
			QtGui.QApplication.processEvents()
			if self.Stop:
				break
			self.NextStep()
		
		if self.Stop:
			self.Stop = False

		else:
			self.statusBar().showMessage('Simulation ended. Total number of simulation steps: ' + str(self.RunStep) + '.')
			self.RunStep = 0
			turing.state = 0

			RunState = 2
			self.ProgramLED(RunState)			

	def ContClicked(self):
	#Runs the next step of the program

		global RunState
		self.lag = 0.1

		self.NextStep()
		if turing.state == self.stateno:
			self.statusBar().showMessage('Simulation ended. Total number of simulation steps: ' + str(self.RunStep) + '.')	
			RunState = 2		
			self.ProgramLED(RunState)
			self.RunStep = 0
			turing.state = 0
		else:
			RunState = 0
			
	def StopClicked(self):
	#Pauses the running of the program

		global RunState
		if RunState == 1:
			self.Stop = True
			self.statusBar().showMessage('Simulation paused at simulation step ' + str(self.RunStep) + '.')
			RunState = 0

	def ResetClicked(self):
	#Resets simulation

		global RunState
		turing.tapeclear()
		turing.printtape()
		self.statusBar().showMessage('Simulation reset.')
		self.RunStep = 0
		self.InitClicked()
		self.ProgramLED(0)		

	def ProgramLED(self, value = 0):
	#sets state of Program LED

		if value == 0:
			GPIO.output(ProgRedPin, True)
			GPIO.output(ProgGreenPin, True)

		elif value == 1:
			GPIO.output(ProgRedPin, False)
			GPIO.output(ProgGreenPin, True)

		elif value == 2:
			GPIO.output(ProgRedPin, True)
			GPIO.output(ProgGreenPin, False)

	def closeEvent(self, event):
		self.ClearClicked()
		self.ProgramLED(0)

switchThread = SwitchThread()

def main():

	global SwitchLoopOn
	app = QtGui.QApplication(sys.argv)
	switchThread.start()
	MW = MainWindow()
	app.exec_()
	SwitchLoopOn = False
	sys.exit()

if __name__ == '__main__':
	main()