# Copyright (c) 2012 Matthew Kirk
# See LICENSE file for details

import sys
import os
from PyQt4.QtGui import *
from PyQt4.QtCore import *

class TabbedLauncher(QTabWidget):

	def __init__(self,logger_function):
		super(TabbedLauncher, self).__init__()
		
		self._logger_function = logger_function

		self.plugin_tab = QWidget(self)
		self.config_tab = QWidget(self)

		self.launch_layout = QVBoxLayout(self.plugin_tab)
		self.live_plot_layout = QFormLayout()
		self.launch_button_layout = QHBoxLayout()
		self.config_layout = QFormLayout(self.config_tab)
		
		self._create_config_options()
		self._create_live_plot_option()
		self._create_plugin_list()
		self._create_buttons()

		self._setup_config_options()
		self._setup_config_form()
		self._setup_live_plot_form()
		self._setup_buttons()

		self._setup_config_tab()
		self._setup_plugin_tab()
		self._setup_tabs()

	def _create_config_options(self):
		self.SAMPLE_LABEL = "Measurement wait:"
		self.TIME_LABEL = "Sample time:"
		self.FILE_TIMESTAMP_LABEL = "File timestamp format:"
		self.DATA_TIMESTAMP_LABEL = "Data Timestamp Format:"
		self.FILENAME_LABEL = "Filename beginning:"
		self.EXTENSION_LABEL = "File extension:"
		self.WINDOWS_LOCATION_LABEL = "Windows Readable Location:"

		self.FILE_TIMESTAMP_FORMAT = "%Y-%m-%d-%H-%M-%S"
		self.DATA_TIMESTAMP_FORMAT = "%Y-%m-%d-%H:%M:%S"
		self.FILENAME_FORMAT = "data-"
		self.EXTENSION_FORMAT = ".log"
		
		self.sample_spin = QDoubleSpinBox()
		self.time_spin = QSpinBox()
		self.file_timestamp_line = QLineEdit(self.FILE_TIMESTAMP_FORMAT)
		self.data_timestamp_line = QLineEdit(self.DATA_TIMESTAMP_FORMAT)
		self.filename_line = QLineEdit(self.FILENAME_FORMAT)
		self.extension_line = QLineEdit(self.EXTENSION_FORMAT)
		self.windows_checkbox = QCheckBox("")

	def _create_live_plot_option(self):
		self.LIVE_PLOT_LABEL = "Plot data as it arrives:"
		self.live_plot_checkbox = QCheckBox("")
		tooltip = ("Plot the data using Gnuplot as it is measured, as well "
					"as logging it to a file.\nIf a tuple of data is being "
					"recived, only the first data point will be plotted.")
		self.live_plot_checkbox.setToolTip(tooltip)

	def _create_plugin_list(self):
		self.plugin_list_label = QLabel("Available Plugins:")

		self.plugin_folder = QDir(sys.path[0] + "/plugins")

		self.plugin_folder_model = QFileSystemModel(self.plugin_tab)
		self.plugin_folder_model.setNameFilters(["[^_]*.py"])
		self.plugin_folder_model.setNameFilterDisables(False)
		self.plugin_folder_model_index = self.plugin_folder_model.setRootPath(self.plugin_folder.path())

		self.plugin_folder_view = QListView()
		self.plugin_folder_view.setModel(self.plugin_folder_model)
		self.plugin_folder_view.setRootIndex(self.plugin_folder_model_index)

		self.plugin_folder_model.dataChanged.connect(self.plugin_folder_view.update)

	def _create_buttons(self):
		self.launch_button = QPushButton("&Start Logging")
		self.quit_button = QPushButton("&Quit")

	def _setup_config_options(self):
		self.sample_spin.setRange(0,300)
		self.sample_spin.setDecimals(1)
		self.sample_spin.setSingleStep(0.1)
		self.sample_spin.setSuffix(" s")
		self.sample_spin.setValue(1)
		self.sample_spin.setSpecialValueText("Minimum")

		self.time_spin.setRange(0,700000)
		self.time_spin.setSuffix(" s")
		self.time_spin.setValue(15)
		self.time_spin.setSpecialValueText("Forever")

		self.sample_spin.setToolTip("How long between measurements")
		self.time_spin.setToolTip("How long to measure for")
		self.file_timestamp_line.setToolTip("The timestamp format for the "
											"name of the file")
		self.data_timestamp_line.setToolTip("The timestamp format for the "
											"data in the file")
		self.filename_line.setToolTip("The filename before the timestamp")
		self.extension_line.setToolTip("The extension of the data file")
		self.windows_checkbox.setToolTip("Put the data file on the /boot "
										"partition, so it will be visible "
										"when the\nSD card is in a Windows "
										"machine - must be root to use this setting.")

	def _setup_config_form(self):
		self.config_layout.addRow(self.SAMPLE_LABEL,self.sample_spin)
		self.config_layout.addRow(self.TIME_LABEL,self.time_spin)
		self.config_layout.addRow(self.FILE_TIMESTAMP_LABEL,self.file_timestamp_line)
		self.config_layout.addRow(self.DATA_TIMESTAMP_LABEL,self.data_timestamp_line)
		self.config_layout.addRow(self.FILENAME_LABEL,self.filename_line)
		self.config_layout.addRow(self.EXTENSION_LABEL,self.extension_line)
		self.config_layout.addRow(self.WINDOWS_LOCATION_LABEL,self.windows_checkbox)

	def _setup_live_plot_form(self):
		self.live_plot_layout.addRow(self.LIVE_PLOT_LABEL, self.live_plot_checkbox)

	def _setup_buttons(self):
		self.launch_button_layout.addWidget(self.quit_button)
		self.launch_button_layout.addWidget(self.launch_button)
		
		self.quit_button.clicked.connect(qApp.quit)
		self.launch_button.clicked.connect(self._prelaunch_check)

	def _setup_config_tab(self):
		self.config_tab.setLayout(self.config_layout)

	def _setup_plugin_tab(self):
		self.launch_layout.addWidget(self.plugin_list_label)
		self.launch_layout.addWidget(self.plugin_folder_view)

		self.launch_layout.addLayout(self.live_plot_layout)
		self.launch_layout.addLayout(self.launch_button_layout)

		self.plugin_tab.setLayout(self.launch_layout)

	def _setup_tabs(self):
		self.addTab(self.plugin_tab, "&Plugins")
		self.addTab(self.config_tab, "&Configuration")

	def _get_selected_plugin(self):
		current_selection_model = self.plugin_folder_view.selectionModel()
		selected_plugin_index = current_selection_model.currentIndex()
		current_plugin_file = self.plugin_folder_model.data(selected_plugin_index)
		current_plugin_name = str(current_plugin_file.toString())[:-3]
		return current_plugin_name

	def _prelaunch_check(self):
		plugin_name = self._get_selected_plugin()
		settings = self._get_config_settings()
		if not plugin_name:
			QMessageBox.warning(self, "Error", "No plugin was selected.")
			return
		elif os.getuid() != 0 and (settings["windows_readable_location"] or 
				plugin_name == "temperature_sensor"):
			message = ("Need to be root to use this plugin or setting, try "
						"re-running the program using sudo")
			QMessageBox.critical(self, "Problem", message)
			return
		try:
			self._logger_function(plugin_name, settings)
		except NotImplementedError as nie:
			message = "This plugin doesn't implement the {} method."
			message = message.format(nie.message)
			QMessageBox.critical(self, "Error", message)

	def _print_config_settings(self):
		print self.SAMPLE_LABEL + str(self.sample_spin.value())
		print self.TIME_LABEL + str(self.time_spin.value())
		print self.FILE_TIMESTAMP_LABEL + self.file_timestamp_line.text()
		print self.DATA_TIMESTAMP_LABEL + self.data_timestamp_line.text()
		print self.FILENAME_LABEL + self.filename_line.text()
		print self.EXTENSION_LABEL + self.extension_line.text()
		print self.WINDOWS_LOCATION_LABEL + str(self.windows_checkbox.isChecked())
		print self.LIVE_PLOT_LABEL + str(self.live_plot_checkbox.isChecked())

	def _get_config_settings(self):
		config_options = {"measurement_wait": self.sample_spin.value(),
						"sampling_time": self.time_spin.value(),
						"file_timestamp_format": str(self.file_timestamp_line.text()),
						"data_timestamp_format": str(self.data_timestamp_line.text()),
						"file_label": str(self.filename_line.text()),
						"extension": str(self.extension_line.text()),
						"windows_readable_location": self.windows_checkbox.isChecked(),
						"live_plot": self.live_plot_checkbox.isChecked()}
		return config_options
