# Copyright (c) 2012 Matthew Kirk
# See LICENSE file for details

class Sensor(object):
	
	# If can_quit is True, the logger will check for a quit input during
	# the measurement period. This can lead to the timings of the
	# measurments being a bit inaccurate, especially for smaller (< 1 s)
	# waits between samples.
	can_quit = False

	# If the get_data function returns more than one data value, this 
	# variable needs changing to match how many values are returned.
	# This can either be done in the same place as here in your plugin, 
	# or in the setup function.
	no_of_measurements = 1
	
	def setup(self):
		"""Set up anything that needs doing before measuring starts.
		
		This could be declaring some variabels that are used later, or 
		turning off a GPIO pin, or finding all the sensors that will 
		be used and changing no_of_measurements appropriately."""
		pass

	def wait_for_start(self):
		"""Wait until the measurement should begin.
		
		Either wait for some input that indicates that logging should 
		start now, or do nothing if no input is needed.
		"""
		pass
		
	def get_data(self):
		"""Get data from the measuring device, and return the data.
		
		For a single piece of data, just return the value, but when 
		multiple values are measured, return a list/tuple of all values.
		"""
		raise NotImplementedError("get_data")
		
	def stop_measuring(self):
		"""Check for an input to stop measuring.
		
		By default, can_quit is False, so the logger will not check for 
		a quit input. If your sensor can provided this sort of input, 
		set can_quit to True, and implement this function so that it 
		returns True when a quit signal has been recieved, and False 
		otherwise. 
		"""
		if self.can_quit:
			# If the can_quit variable is True, this method should have
			# been overridden by the child class.
			raise NotImplementedError("stop_measuring")
		else:
			return False
		
	def finished(self):
		"""Do anything that should happen after measurement is finished."""
		pass
