# Copyright (c) 2012 Matthew Kirk
# See LICENSE file for details

import subprocess
from time import strftime, time, sleep
from os import listdir, getcwd, path
from importlib import import_module

def time_up(start_time, max_time):
	"""Check whether total logging time has passed yet."""
	if max_time == 0:
		# Sample time was given as forever.
		return False
	else:
		return ((time() - start_time) > max_time)

def logger(plugin_choice, config_dict):
	"""Log data from chosen plugin, according to configuration dictionary.

	plugin choice is name of script file, without .py extension
	config_dict is a dictionary with the following configuration options:

	"measurement_wait" - how long between measurements
	"sampling_time" - how long to sample for,
	"file_timestamp_format" - format of the timestamp used in log file name,
	"data_timestamp_format" - format of the timestamp written to the file,
	"file_label" - the start of the log file name,
	"extension" - what extension to give the log file,
	"windows_readable_location" - whether to write log file in /boot, to
	"make it visible from a Windows computer,
	"live_plot" - whether to plot data as it is measured, with Gnuplot.
	"""

	plugin_module = import_module("plugins." + plugin_choice)
	plugin_class = "".join([w.capitalize() for w in plugin_choice.split("_")])
	plugin = eval("".join(["plugin_module.", plugin_class, "()"]))

	if config_dict["windows_readable_location"]:
		logfile_directory = "/boot"
	else:
		logfile_directory = ""

	plugin.setup()
	measurements = plugin.no_of_measurements

	if config_dict["live_plot"]:
		# Open up pipe to Gnuplot as a subprocess, and configure plottin.
		live_plot_process = subprocess.Popen(["gnuplot"], stdin=subprocess.PIPE)
		live_plot_input = live_plot_process.stdin
		live_plot_input.write("set datafile separator \",\"\n")
		live_plot_input.write("set xlabel \"Time / s\"\n")
		live_plot_input.write("set ylabel \"" + plugin_choice + "\"\n")
		live_plot_input.write("unset key\n")
		# Plot data points using "+" symbols (pt 1).
		plot_data = ["plot" + ",".join([" \"-\" with points pt 1"] * measurements)]
		data_list = []
		for i in range(measurements):
			data_list.append([])
		end = ["e\n"]

	plugin.wait_for_start()

	log_file_name = "".join([config_dict["file_label"],
							strftime(config_dict["file_timestamp_format"]),
							config_dict["extension"]])
	exit_flag = False
	t1 = 0
	start_time = time()

	with open(path.join(logfile_directory, log_file_name), "w", 1) as log_file:
		while not (plugin.stop_measuring() or
					time_up(start_time, config_dict["sampling_time"])):
			t2 = time()
			if (t2 - t1) < config_dict["measurement_wait"]:
				# Need to wait a bit before next measurement.
				if plugin.can_quit:
					# Break up the sleep into smaller chunks, so it is
					# possible to stop measuring quickly.
					time_difference = config_dict["measurement_wait"] - (t2 - t1)
					no_of_sleeps = int(round(time_difference / 0.01))
					# Divide the wait into 0.01 second bits, and round to the
					# nearest tenth of a second.
					for i in range(no_of_sleeps):
						sleep(0.01)
						if (plugin.stop_measuring() or
								time_up(start_time, config_dict["sampling_time"])):
							# Set exit_flag to allow us to break out of the
							# outer while loop.
							exit_flag = True
							break # Break out of this sleeping loop.
				else:
					sleep(config_dict["measurement_wait"] - (t2 - t1))
			if exit_flag:
				# Breaking out of outer while loop here.
				break
			t1 = time()
			data = plugin.get_data()
			data_time = strftime(config_dict["data_timestamp_format"])
			delta_time = str(time() - start_time)
			if measurements == 1:
				# Only a single data value returned, (hopefully float or int).
				log_data = ",".join([data_time, delta_time, str(data)])
			else:
				# More than 1 data value being returned, so iterate over them.
				log_data = ",".join([data_time, delta_time] + [str(x) for x in data])
			log_file.write(log_data)
			log_file.write("\n")
			if config_dict["live_plot"]:
				if measurements == 1:
					# Only a single data value being returned.
					data_list[0].append(",".join([delta_time, str(data)]))
				else:
					# A tuple of data is being recorded.
					for i in range(measurements):
						data_list[i].append(",".join([delta_time, str(data[i])]))
				command_list = list(plot_data)
				for i in range(measurements):
					command_list += data_list[i]
					command_list += end
				command_list += ["\n"]
				data = "\n".join(command_list)
				live_plot_input.write(data)

	plugin.finished()
