﻿// $Id: NearestNeighborUpsampler.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Stereo.GpGpuLib;

namespace VideoLib.Stereo.GpGpu.StereoNodes
{
	/// <summary>
	/// Upsamples images using nearest-neighbour interpolation.
	/// </summary>
	class NearestNeighborUpsampler : UpsamplerStereoNode, IDisposable
	{
		private DepthMap result = new DepthMap();

		/// <summary>
		/// Initializes a new instance of the <see cref="NearestNeighborUpsampler"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public NearestNeighborUpsampler(ExecutionStream stream) : base(stream) { }

		/// <summary>
		/// Applies trivial nearest-neighbour upsampling.
		/// </summary>
		/// <param name="depthMap">The low resolution input depth map</param>
		/// <param name="originalImage">The original image</param>
		/// <param name="depthMapOut">The up-sampled output depth map</param>
		public override void UpsampleDepthMap(DepthMap depthMap, int inputDisparities, InputImage leftImage, InputImage rightImage, out DepthMap depthMapOut, int outputDisparities)
		{
			if ((leftImage.Width == depthMap.Width) && (leftImage.Height == depthMap.Height))
				depthMapOut = depthMap;
			else
			{
				ManagedUpsamplers.NnUpsample(depthMap, leftImage, result);
				depthMapOut = result;
			}
		}

		public void Dispose()
		{
			result.Dispose();
			result = null;
		}
	}

	/// <summary>
	/// Factory that produces a <see cref="NearestNeighborUpsampler"/>.
	/// </summary>
	class NearestNeighborUpsamplerFactory : StereoNodeFactory<UpsamplerStereoNode>
	{
		/// <summary>
		/// Creates a <see cref="NearestNeighborUpsampler"/> to run on the specified execution stream.
		/// </summary>
		/// <param name="stream">The execution stream.</param>
		/// <returns>
		/// An instance of <see cref="UpsamplerStereoNode"/>.
		/// </returns>
		public override UpsamplerStereoNode Create(ExecutionStream stream)
		{
			return new NearestNeighborUpsampler(stream);
		}

		/// <summary>
		/// Checks if the <see cref="NullUpSampler"/> is compatible with the given datatypes.
		/// </summary>
		/// <param name="imageType">Image datatype.</param>
		/// <param name="gridType">Cost space datatype.</param>
		/// <param name="mapType">Depth map datatype.</param>
		public override void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType)
		{
			isValid = (mapType == DepthMapType.Single);
		}

		/// <summary>
		/// Gets the name of this node.
		/// </summary>
		public override string Name { get { return "Nearest neighbour"; } }

		/// <summary>
		/// Gets the sort key. Should put this node first.
		/// </summary>
		public override string SortKey { get { return "_1_" + Name; } }
	}
}
