﻿// $Id: JointBilateralUpsampler.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VideoLib.Parameters;
using Stereo.GpGpuLib;

namespace VideoLib.Stereo.GpGpu.StereoNodes
{
	/// <summary>
	/// Upsamples images using joint-bilateral upsampling (Kopf et al.).
	/// </summary>
	class JointBilateralUpsampler : UpsamplerStereoNode, IDisposable
	{
		#region Parameters

		[Parameter(Minimum = 0.5, Maximum = 2, Default = 1, FriendlyName = "Spatial sigma")]
		public float SigmaS { get; set; }

		[Parameter(Minimum = 0.01, Maximum = 1, Default = 0.1, FriendlyName = "Colour sigma")]
		public float SigmaC { get; set; }

		#endregion

		private DepthMap result = new DepthMap();

		/// <summary>
		/// Initializes a new instance of the <see cref="JointBilateralUpsampler"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public JointBilateralUpsampler(ExecutionStream theStream) : base(theStream) { }

		/// <summary>
		/// Up-samples the depth map.
		/// </summary>
		/// <param name="depthMap">The low resolution input depth map.</param>
		/// <param name="originalImage">The original image.</param>
		/// <param name="depthMapOut">The up-sampled output depth map.</param>
		public override void UpsampleDepthMap(DepthMap depthMap, int inputDisparities, InputImage leftImage, InputImage rightImage, out DepthMap depthMapOut, int outputDisparities)
		{
			if ((leftImage.Width == depthMap.Width) && (leftImage.Height == depthMap.Height))
				depthMapOut = depthMap;
			else
			{
				ManagedUpsamplers.JbUpsample(depthMap, leftImage, SigmaS, SigmaC, result);
				depthMapOut = result;
			}
		}

		public void Dispose()
		{
			result.Dispose();
			result = null;
		}
	}

	/// <summary>
	/// Factory that produces a <see cref="JointBilateralUpsampler"/>.
	/// </summary>
	class JointBilateralUpsamplerFactory : StereoNodeFactory<UpsamplerStereoNode>
	{
		/// <summary>
		/// Creates a <see cref="JointBilateralUpsampler"/> to run on the specified execution stream.
		/// </summary>
		/// <param name="stream">The execution stream.</param>
		/// <returns>
		/// An instance of <see cref="UpsamplerStereoNode"/>.
		/// </returns>
		public override UpsamplerStereoNode Create(ExecutionStream stream)
		{
			return new JointBilateralUpsampler(stream);
		}

		/// <summary>
		/// Checks if the <see cref="JointBilateralUpsampler"/> is compatible with the given datatypes.
		/// </summary>
		/// <param name="imageType">Image datatype.</param>
		/// <param name="gridType">Cost space datatype.</param>
		/// <param name="mapType">Depth map datatype.</param>
		public override void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType)
		{
			isValid = (mapType == DepthMapType.Single);
		}

		/// <summary>
		/// Gets the name of this node.
		/// </summary>
		public override string Name { get { return "Joint-bilateral upsampling"; } }
	}
}
