﻿// $Id: ResidualPreProcessor.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Stereo.GpGpuLib;
using VideoLib.Parameters;

namespace VideoLib.Stereo.GpGpu.StereoNodes
{
	/// <summary>
	/// Residual pre-processing subtracts the local average from each pixel, to reduce radiometric differences
	/// between input images and to emphasise details over constant image areas.
	/// </summary>
	class ResidualPreProcessor : PreProcessorStereoNode, IDisposable
	{
		#region Parameters

		/// <summary>
		/// Gets or sets the standard deviation of the Gaussian blur.
		/// </summary>
		[Parameter(Minimum = 0.5, Maximum = 30.0, Default = 5.0)]
		public float Sigma { get; set; }

		/// <summary>
		/// Gets or sets the amount of the original image to mix with the residual
		/// </summary>
		[Parameter(Minimum = 0, Maximum = 1, Default = 0, FriendlyName = "Image re-mixing")]
		public float ReMixingFactor { get; set; }

		#endregion

		/// <summary>
		/// Initializes a new instance of the <see cref="ResidualPreProcessor"/> class.
		/// </summary>
		public ResidualPreProcessor() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="ResidualPreProcessor"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public ResidualPreProcessor(ExecutionStream stream) : base(stream) { }

		private InputImage temp1 = new InputImage();
		private InputImage temp2 = new InputImage();

		/// <summary>
		/// Pre-processes the given stereo image pair.
		/// </summary>
		public override void PreProcessImages(InputImage leftImage, InputImage rightImage)
		{
			ManagedPreProcessors.RunBlurKernel(leftImage, temp1, temp2, Sigma);
			ManagedPreProcessors.RunResidualKernel(leftImage, temp2, ReMixingFactor);

			ManagedPreProcessors.RunBlurKernel(rightImage, temp1, temp2, Sigma);
			ManagedPreProcessors.RunResidualKernel(rightImage, temp2, ReMixingFactor);
		}

		/// <summary>
		/// Frees temporary images.
		/// </summary>
		public void Dispose()
		{
			temp1.Dispose();
			temp2.Dispose();
		}
	}

	/// <summary>
	/// Factory that produces a <see cref="ResidualPreProcessor"/>.
	/// </summary>
	class ResidualPreProcessorFactory : StereoNodeFactory<PreProcessorStereoNode>
	{
		/// <summary>
		/// Creates a <see cref="PreProcessorStereoNode"/> to run on the specified execution stream.
		/// </summary>
		/// <param name="stream">The execution stream.</param>
		/// <returns>
		/// An instance of <see cref="PreProcessorStereoNode"/>.
		/// </returns>
		public override PreProcessorStereoNode Create(ExecutionStream stream)
		{
			return new ResidualPreProcessor(stream);
		}

		/// <summary>
		/// Checks if the <see cref="ResidualPreProcessor"/> is compatible with the given datatypes.
		/// </summary>
		public override void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType)
		{
			isValid = (imageType == InputImageType.Xrgb_uint);
		}

		/// <summary>
		/// Gets the name of this node.
		/// </summary>
		public override string Name { get { return "Residual pre-processing"; } }
	}
}
