﻿// $Id: BilSubPreProcessor.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Stereo.GpGpuLib;
using VideoLib.Parameters;

namespace VideoLib.Stereo.GpGpu.StereoNodes
{
	class BilSubPreProcessor : PreProcessorStereoNode, IDisposable
	{
		#region Parameters

		[Parameter(Minimum = 1, Maximum = 20, Default = 7, FriendlyName = "Kernel radius")]
		public int Radius { get; set; }

		[Parameter(Minimum = 0.5, Maximum = 20, Default = 3, FriendlyName = "Spatial Sigma")]
		public float SigmaS { get; set; }

		[Parameter(Minimum = 0.5, Maximum = 50, Default = 20, FriendlyName = "Range Sigma")]
		public float SigmaR { get; set; }

		#endregion

		/// <summary>
		/// Initializes a new instance of the <see cref="ResidualPreProcessor"/> class.
		/// </summary>
		public BilSubPreProcessor() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="ResidualPreProcessor"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public BilSubPreProcessor(ExecutionStream stream) : base(stream) { }

		private InputImage tempImage = new InputImage();

		/// <summary>
		/// Pre-processes the given stereo image pair.
		/// </summary>
		public override void PreProcessImages(InputImage leftImage, InputImage rightImage)
		{
			ManagedPreProcessors.RunBilSubKernel(leftImage,  tempImage, Radius, SigmaS, SigmaR);
			ManagedPreProcessors.RunBilSubKernel(rightImage, tempImage, Radius, SigmaS, SigmaR);
		}

		/// <summary>
		/// Frees temporary images.
		/// </summary>
		public void Dispose()
		{
			if (tempImage != null)
			{
				tempImage.Dispose();
				tempImage = null;
			}
		}
	}

	/// <summary>
	/// Factory that produces a <see cref="BilSubPreProcessor"/>.
	/// </summary>
	class BilSubPreProcessorFactory : StereoNodeFactory<PreProcessorStereoNode>
	{
		/// <summary>
		/// Creates a <see cref="PreProcessorStereoNode"/> to run on the specified execution stream.
		/// </summary>
		/// <param name="stream">The execution stream.</param>
		/// <returns>
		/// An instance of <see cref="PreProcessorStereoNode"/>.
		/// </returns>
		public override PreProcessorStereoNode Create(ExecutionStream stream)
		{
			return new BilSubPreProcessor(stream);
		}

		/// <summary>
		/// Checks if the <see cref="ResidualPreProcessor"/> is compatible with the given datatypes.
		/// </summary>
		public override void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType)
		{
			isValid = (imageType == InputImageType.Xrgb_uint);
		}

		/// <summary>
		/// Gets the name of this node.
		/// </summary>
		public override string Name { get { return "Background subtraction"; } }
	}
}
