﻿// $Id: ImageDownsampler.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VideoLib.Parameters;
using Stereo.GpGpuLib;

namespace VideoLib.Stereo.GpGpu.StereoNodes
{
	/// <summary>
	/// Downsamples the input images, to speed up the stereo computation process.
	/// </summary>
	class ImageDownsampler : DownsamplerStereoNode
	{
		#region Parameters

		/// <summary>Gets or sets the anti-aliasing blur to apply before down-sampling</summary>
		[Parameter(Minimum = 0.01, Maximum = 2, Default = 0.25)]
		public float Sigma { get; set; }

		#endregion

		private InputImage tempImage1 = new InputImage();
		private InputImage tempImage2 = new InputImage();
		private InputImage resultL = new InputImage();
		private InputImage resultR = new InputImage();

		/// <summary>
		/// Initializes a new instance of the <see cref="ImageDownsampler"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public ImageDownsampler(ExecutionStream stream) : base(stream) { }

		/// <summary>
		/// Down-samples the pair of input images
		/// </summary>
		/// <param name="leftImage">The left image.</param>
		/// <param name="rightImage">The right image.</param>
		/// <param name="dsFactor">Is set to the down-sampling factor actually used</param>
		/// <param name="sampledLeftImage">The down-sampled left image output</param>
		/// <param name="sampledRightImage">The down-sampled right image output</param>
		public override void DownsampleImages(InputImage leftImage, InputImage rightImage, out int dsFactor,
			out InputImage sampledLeftImage, out InputImage sampledRightImage)
		{
			dsFactor = DownsamplingFactor;

			if (dsFactor <= 1)
			{
				sampledLeftImage = leftImage;
				sampledRightImage = rightImage;
			}
			else
			{
				// Blur (for anti-aliasing purposes) and downsample the left input image
				ManagedPreProcessors.RunBlurKernel(leftImage, tempImage1, tempImage2, Sigma * (float)dsFactor);
				ManagedDownsamplers.DownsampleImage(tempImage2, dsFactor, resultL);

				// Blur (for anti-aliasing purposes) and downsample the right input image
				ManagedPreProcessors.RunBlurKernel(rightImage, tempImage1, tempImage2, Sigma * (float)dsFactor);
				ManagedDownsamplers.DownsampleImage(tempImage2, dsFactor, resultR);

				sampledLeftImage = resultL;
				sampledRightImage = resultR;
			}
		}

		public void Dispose()
		{
			tempImage1.Dispose();
			tempImage2.Dispose();
			resultL.Dispose();
			resultR.Dispose();

			resultL = resultR = tempImage1 = tempImage2 = null;
		}
	}

	/// <summary>
	/// Factory that produces an <see cref="ImageDownsampler"/>.
	/// </summary>
	class ImageDownsamplerFactory : StereoNodeFactory<DownsamplerStereoNode>
	{
		/// <summary>
		/// Creates a <see cref="ImageDownsampler"/> to run on the specified execution stream.
		/// </summary>
		/// <param name="stream">The execution stream.</param>
		/// <returns>
		/// An instance of <see cref="DownsamplerStereoNode"/>.
		/// </returns>
		public override DownsamplerStereoNode Create(ExecutionStream stream)
		{
			return new ImageDownsampler(stream);
		}

		/// <summary>
		/// Checks if the <see cref="ImageDownsampler"/> is compatible with the given datatypes.
		/// </summary>
		/// <param name="imageType">Image datatype.</param>
		/// <param name="gridType">Cost space datatype.</param>
		/// <param name="mapType">Depth map datatype.</param>
		public override void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType)
		{
			isValid = (imageType == InputImageType.Xrgb_uint); // always valid
		}

		/// <summary>
		/// Gets the name of this node.
		/// </summary>
		public override string Name { get { return "Downsampling"; } }
	}
}
