﻿// $Id: RegisteredCollection.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace VideoLib.Stereo.GpGpu
{
	/// <summary>
	/// Represents a limited-functionality list which generates events when items are added or removed
	/// </summary>
	/// <typeparam name="T">The type of objects to be stored in the list</typeparam>
	public class RegisteredCollection<T>
	{
		private List<T> inner;

		/// <summary>
		/// Represents an event handler, used by (<see cref="ItemAdded"/>, <see cref="ItemRemoved"/>).
		/// </summary>
		/// <param name="sender">The collection that has been modified</param>
		/// <param name="item">The item that was changed</param>
		public delegate void ItemModificationHandler(RegisteredCollection<T> sender, T item);
		/// <summary>
		/// Occurs when an item is added to the collection
		/// </summary>
		public event ItemModificationHandler ItemAdded;
		/// <summary>
		/// Occurs when an item is removed from the collection
		/// </summary>
		public event ItemModificationHandler ItemRemoved;

		/// <summary>
		/// Creates an empty instance of RegisteredCollection
		/// </summary>
		public RegisteredCollection() { inner = new List<T>(); }
		/// <summary>
		/// Creates an instance of RegisteredCollection, which has been given an initial capacity for items
		/// </summary>
		/// <param name="capacity">The initial capacity of the list</param>
		public RegisteredCollection(int capacity) { inner = new List<T>(capacity); }
		/// <summary>
		/// Creates an instance of RegisteredCollection, initialized with <paramref name="collection"/>
		/// </summary>
		/// <param name="collection">The initial collection of items to hold in the list</param>
		public RegisteredCollection(IEnumerable<T> collection) { inner = new List<T>(collection); }

		/// <summary>
		/// Registers a pair of event handlers with the collection, and optionally sends <see cref="ItemAdded"/> for
		/// each item currently in the list 
		/// </summary>
		/// <param name="addedHandler">The handler to invoke when an item is added</param>
		/// <param name="removedHandler">The handler to invoke when an item is removed</param>
		/// <param name="sendInitialUpdate">
		/// If true, <paramref name="addedHandler"/> is invoked for every item currently in the collection
		/// </param>
		public void RegisterHandlers(ItemModificationHandler addedHandler, ItemModificationHandler removedHandler, bool sendInitialUpdate)
		{
			ItemAdded += addedHandler;
			ItemRemoved += removedHandler;

			if (sendInitialUpdate)
				inner.ForEach(x => OnItemAdded(x));
		}

		// Fires ItemAdded
		private void OnItemAdded(T item)
		{
			if (ItemAdded != null)
				ItemAdded(this, item);
		}
		// Fires ItemRemoved
		private void OnItemRemoved(T item)
		{
			if (ItemRemoved != null)
				ItemRemoved(this, item);
		}

		/// <summary>
		/// Adds a new item to the list
		/// </summary>
		/// <param name="item">The item to add</param>
		public void Add(T item)
		{
			inner.Add(item);
			OnItemAdded(item);
		}
		/// <summary>
		/// Adds a collection of new items to the list
		/// </summary>
		/// <param name="collection">The item to add</param>
		public void AddRange(IEnumerable<T> collection)
		{
			inner.AddRange(collection);

			foreach (T item in collection)
				OnItemAdded(item);
		}
		/// <summary>
		/// Removes the first matching item from the list
		/// </summary>
		/// <param name="item">The item to remove</param>
		public void Remove(T item)
		{
			if (item != null)
			{
				T toRemove = inner.First(x => x.Equals(item));
				if (inner.Remove(item))
					OnItemRemoved(toRemove);
			}
		}
		/// <summary>
		/// Removes all matching items from the list
		/// </summary>
		/// <param name="item">The item to remove</param>
		public void RemoveAll(T item)
		{
			IEnumerable<T> toRemove = inner.Where(x => x.Equals(item));
			inner.RemoveAll(x => x.Equals(item));

			foreach(T remItem in toRemove)
				OnItemRemoved(remItem);
		}

		/// <summary>
		/// Gets the item in the list at <paramref name="index"/>.
		/// </summary>
		/// <param name="index">The index to retrieve</param>
		/// <returns>The item found at <paramref name="index"/></returns>
		public T this[int index]
		{
			get { return inner[index]; }
		}

		/// <summary>
		/// Finds an item that matches a specified predicate
		/// </summary>
		/// <param name="match">The matching criteria</param>
		/// <returns>An item from the collection which satisfies the predicate</returns>
		public T Find(Predicate<T> match)
		{
			return inner.Find(match);
		}

		/// <summary>
		/// Finds all items that match a specified predicate
		/// </summary>
		/// <param name="match">The matching criteria</param>
		/// <returns>An item from the collection which satisfies the predicate</returns>
		public List<T> FindAll(Predicate<T> match)
		{
			return inner.FindAll(match);
		}

		/// <summary>
		/// Gets the current number of items in the list
		/// </summary>
		public int Count
		{
			get { return inner.Count; }
		}
	}
}
