﻿// $Id: LeftRightValidationNode.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VideoLib;
using VideoLib.Frames;
using System.Collections;

namespace CrossChecker
{
	public class LeftRightValidationNode : Node
	{
		public float Threshold { get; set; }

		public LeftRightValidationNode()
			: base()
		{
			Threshold = 1.0f;

			AddInputPins("inL", "inR");
			AddOutputPins("outL");
		}

		public override void Process()
		{
			// check input frames
			var frames = GetInputFrames("inL", "inR");
			if (PassOnEndOfStreamFrame(frames)) return;
			var bitmapFrames = CastFramesTo<BitmapFrame>(frames);
			AssertFrameProperties(bitmapFrames, FrameProps.Size | FrameProps.Pixelformat | FrameProps.Sequencenumber);
			AssertPixelFormat(bitmapFrames, PixelFormat.Float);

			// 0. Start with raw left disparity map
			BitmapFrame outputFrame = bitmapFrames["inL"];

			// 1. Apply left-right consistency check
			outputFrame = LeftRightCheck(bitmapFrames["inL"], bitmapFrames["inR"]);

			// 2. clean up some outliers
			Console.WriteLine("Inpainted {0} pixels using median (maxInvalid=4).", MedianInpainting(outputFrame, 4));

			// 3. Fill invalid regions
			Console.WriteLine("Filled in {0} pixels.", BlackSegmentFilling(outputFrame));

			// 4. Fix some obvious errors
			Console.WriteLine("Fixed {0} pixels (threshold=-1).", MedianFixing(outputFrame, -1.00f));

			PushFrame("outL", outputFrame);
		}


		private BitmapFrame LeftRightCheck(BitmapFrame left, BitmapFrame right)
		{
			BitmapFrame result = new BitmapFrame(left.Width, left.Height, left.Format) { SequenceNumber = left.SequenceNumber };

			unsafe
			{
				// fix all pointers
				fixed (byte* pOutByte = result.Data, pLeftByte = left.Data, pRightByte = right.Data)
				{
					float* pOut = (float*)pOutByte;
					float* pLeft = (float*)pLeftByte;
					float* pRight = (float*)pRightByte;

					for (int y = 0; y < result.Height; y++)
					{
						for (int x = 0; x < result.Width; x++)
						{
							float leftDisp = pLeft[y * result.Width + x];
							int rightX = (int)(0.5f + x - leftDisp);

							if (rightX < 0 || rightX >= result.Width)
							{
								// outside the image
								pOut[y * result.Width + x] = 0;
							}
							else
							{
								float rightDisp = pRight[y * result.Width + rightX];
								if(Math.Abs(leftDisp - rightDisp) > Threshold)
									pOut[y * result.Width + x] = -1; // invalid disparity
								else
									pOut[y * result.Width + x] = leftDisp;
							}
							//float rightDisp = 
							//pOut[y * result.Width + (int)(0.5f + Math.Max(Math.Min(x + pRight[y * result.Width + x], result.Width - 1), 0))] = pRight[y * result.Width + x];
						}
					}
				}
			}

			return result;
		}

		private unsafe int MedianInpainting(BitmapFrame frame, int maxInvalid)
		{
			// count number of inpainted pixels
			int numPixels = 0;

			// make a copy of the frame
			byte[] inputData = new byte[frame.Data.Length];
			Array.Copy(frame.Data, inputData, frame.Data.Length);

			// fix all pointers
			fixed (byte* pOutByte = frame.Data, pInByte = inputData)
			{
				float* pOut = (float*)pOutByte;
				float* pIn = (float*)pInByte;

				for (int y = 0; y < frame.Height; y++)
				{
					for (int x = 0; x < frame.Width; x++)
					{
						float val = pIn[y * frame.Width + x];

						if (val < 0)
						{
							List<float> values = new List<float>(9);

							int numValid = 0;
							int numInvalid = 0;

							for (int dy = Math.Max(0, y - 1); dy <= Math.Min(y + 1, frame.Height - 1); dy++)
							{
								for (int dx = Math.Max(0, x - 1); dx <= Math.Min(x + 1, frame.Width - 1); dx++)
								{
									float pixVal = pIn[dy * frame.Width + dx];
									if (pixVal < 0)
									{
										numInvalid++;
									}
									else
									{
										values.Add(pixVal);
										numValid++;
									}
								}
							}
							values.Sort();

							if (numInvalid <= maxInvalid && numValid > 0) // including central pixel
							{
								pOut[y * frame.Width + x] = values[numValid / 2];
								numPixels++;
							}
						}
					}
				}
			}

			return numPixels;
		}

		private unsafe int MedianFixing(BitmapFrame frame, float threshold)
		{
			// count number of fixed pixels
			int numPixels = 0;

			// make a copy of the frame
			byte[] inputData = new byte[frame.Data.Length];
			Array.Copy(frame.Data, inputData, frame.Data.Length);

			// fix all pointers
			fixed (byte* pOutByte = frame.Data, pInByte = inputData)
			{
				float* pOut = (float*)pOutByte;
				float* pIn = (float*)pInByte;

				for (int y = 0; y < frame.Height; y++)
				{
					for (int x = 0; x < frame.Width; x++)
					{
						float val = pIn[y * frame.Width + x];

						if (val >= 0) // only consider valid pixels
						{
							List<float> values = new List<float>(9);
							int numValid = 0;

							for (int dy = Math.Max(0, y - 1); dy <= Math.Min(y + 1, frame.Height - 1); dy++)
							{
								for (int dx = Math.Max(0, x - 1); dx <= Math.Min(x + 1, frame.Width - 1); dx++)
								{
									float pixVal = pIn[dy * frame.Width + dx];
									values.Add(pixVal);
									if (pixVal >= 0)
										numValid++;
								}
							}
							values.Sort();

							if (Math.Abs(values[values.Count / 2] - val) > threshold)
							{
								if (values[values.Count / 2] > 0 || numValid <= 2)
								{
									pOut[y * frame.Width + x] = values[values.Count / 2];
									numPixels++;
								}
							}
						}
					}
				}
			}

			return numPixels;
		}

		private unsafe int BlackSegmentFilling(BitmapFrame frame)
		{
			// count number of inpainted pixels
			int numPixels = 0;

			fixed (byte* pByte = frame.Data)
			{
				float* pFloat = (float*)pByte;
				
				// process each scanline independently
				for (int y = 0; y < frame.Height; y++)
				{
					//// start and end of runs of black pixels
					//int x1 = -1; float x1val = -1;
					//int x2 = -1; float x2val = -1;


					float pval = -1; // value of previous pixel
					int x = 0; // current pixel

					while (x < frame.Width)
					{
						float xval = pFloat[y * frame.Width + x];

						if (xval < 0) // found start of a run of black pixels
						{
							int x1 = x; // first black pixel
							int x2 = x; // last black pixel
							float x1val = pval; // value of pixel before first black pixel
							float x2val = -1;   // value of pixel after last black pixel

							// look for end of run
							x++;
							while (x < frame.Width)
							{
								xval = pFloat[y * frame.Width + x];

								if (xval < 0) // continue run of black pixels
								{
									x2 = x;
									x++;
								}
								else
								{
									x2val = xval;
									break;
								}
							}

							// TODO: fill in
							float fillValue = -1;
							if(x1val < 0) fillValue = x2val;
							else if(x2val < 0) fillValue = x1val;
							else fillValue = Math.Min(x1val, x2val);
							for (int i = x1; i <= x2; i++)
							{
								pFloat[y * frame.Width + i] = fillValue;
								numPixels++;
							}
						}
						else // go to next pixel
						{
							x++;
						}
						pval = xval;
					}
				}
			}

			return numPixels;
		}
	}
}
