﻿// $Id: CrossCheckerProgram.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using VideoLib;
using VideoLib.Scheduling;
using VideoLib.Sources;
using VideoLib.LibAv;
using VideoLib.Frames;
using VideoLib.Sinks;
using LeftRightStereo;
using System.IO;

namespace CrossChecker
{
	class CrossCheckerProgram
	{
		static void Main(string[] args)
		{
			if (args.Length < 4)
			{
				Console.WriteLine("Usage: CrossChecker input_dir num_disparities disparity_scale technique_name");
				return;
			}

			// inputs
			string dataPath = args[0]; // "../../../data/teddy/";
			int numDisparities = Int32.Parse(args[1]); // 64
			int dispScale = Int32.Parse(args[2]); // 4
			string techniqueName = args[3];

			// derived inputs
			string outputPath = Path.Combine(dataPath, techniqueName);
			string leftImageFilename = Path.Combine(dataPath, "left.png");
			string rightImageFilename = Path.Combine(dataPath, "right.png");
			string maskAllFilename = Path.Combine(dataPath, "mask-all.png");
			string maskNonoccFilename = Path.Combine(dataPath, "mask-nonocc.png");
			string maskDiscFilename = Path.Combine(dataPath, "mask-disc.png");
			string groundTruthFilename = Path.Combine(dataPath, "true.png");

			// run left-right & right-left stereo
			string leftDispFilename = Path.Combine(outputPath, "disp-left.png");
			string rightDispFilename = Path.Combine(outputPath, "disp-right.png");
			Directory.CreateDirectory(outputPath);
			LeftRightStereoProgram.Main(new string[] { leftImageFilename, rightImageFilename, numDisparities.ToString(), leftDispFilename, rightDispFilename, techniqueName });

			// set up topology and scheduler ------------------------------------------------------
			Topology topology = Topology.MainInstance;
			Scheduler scheduler = new SimpleDequeScheduler(topology);

			// set up nodes -----------------------------------------------------------------------
			var leftSource = new ImageSequenceSource(leftDispFilename) { LastNumber = 0 };
			var rightSource = new ImageSequenceSource(rightDispFilename) { LastNumber = 0 };

			// convert to float map
			var leftMap = new GreyToFloatNode() { ScaleFactor = numDisparities / 255.0f };
			var rightMap = new GreyToFloatNode() { ScaleFactor = numDisparities / 255.0f };
			topology.Connect(leftSource, new LibAvConvert(PixelFormat.RGB24), leftMap);
			topology.Connect(rightSource, new LibAvConvert(PixelFormat.RGB24), rightMap);

			// left-right consistency check
			var checker = new LeftRightValidationNode();
			topology.Connect(leftMap, checker["inL"]);
			topology.Connect(rightMap, checker["inR"]);

			// convert back to image
			topology.Connect(checker["outL"], new FloatToGreyNode() { ScaleFactor = dispScale }, new ImageSequenceSink(Path.Combine(outputPath, "disp-left-cleaned.gif")));

			// evaluation
			var eval = new MiddleburyEvaluationNode() { MaskAllFilename = maskAllFilename, MaskDiscFilename = maskDiscFilename, MaskNonoccFilename = maskNonoccFilename };
			topology.Connect(checker["outL"], eval["in_disparities"]);
			topology.Connect(
				new ImageSequenceSource(groundTruthFilename) { LastNumber = 0 },
				new GreyToFloatNode() { ScaleFactor = 1.0f / dispScale },
				eval["in_groundtruth"]);

			// start the scheduler ----------------------------------------------------------------
			scheduler.Setup();
			scheduler.StartWorkers(1);
			scheduler.Join();
		}
	}
}
