﻿// $Id: MainFlexForm.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Threading;
using System.Windows.Forms;
using VideoLib;
using VideoLib.Filters;
using VideoLib.LibAv;
using VideoLib.Scheduling;
using VideoLib.Sinks;
using VideoLib.Stereo.GpGpu;
using System.IO;

namespace RealTimeStereoTestViewer
{
    /// <summary>
    /// A different interface, designed around the basis of a flexible dynamic configuration of stereo processes.
    /// </summary>
    public partial class MainFlexForm : Form
    {
        private const int PositionOffset = 40;

        private delegate void TopologyInit(Topology topology);
        private readonly TopologyInit setupCapture;

        private FlexConfigForm flexForm;
        private DynamicParametersForm auxForm;

        private DisparityEstimationNode disparitiesNode;

        public MainFlexForm()
        {
            InitializeComponent();

            // Position the current (main) form a small distance from the top-left corner
            this.StartPosition = FormStartPosition.Manual;
            var bounds = Screen.PrimaryScreen.WorkingArea;
            this.Location = new Point(bounds.X + PositionOffset, bounds.Y + PositionOffset);

            // This delegate gets triggered when the topology needs to be built
            setupCapture = new TopologyInit(SetupCapture);

            this.FormClosing += new FormClosingEventHandler(MainFlexForm_FormClosing);
        }

        private void MainFlexForm_Load(object sender, EventArgs e)
        {
            this.BeginInvoke(new Action(delegate
            {
                PopulateVideoSourceComboBox();
                PopulateVideoResolutionComboBox();
            }));
        }

        #region Form closing handlers (which cause all other forms to close)
        private bool closing = false;
        private void MainFlexForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            closing = true;

            if (flexForm != null)
                flexForm.BeginInvoke(new Action(delegate { flexForm.Close(); }));
            if (auxForm != null)
                auxForm.BeginInvoke(new Action(delegate { auxForm.Close(); }));
        }
        private void auxForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            auxForm = null;
            if(!closing)
                BeginInvoke(new Action(delegate { Close(); }));
        }
        private void flexForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            flexForm = null;
            if (!closing)
                BeginInvoke(new Action(delegate { Close(); }));
        }
        #endregion

        private void disparitiesNode_Initialized(DisparityEstimationNode sender)
        {
            this.BeginInvoke(new Action(delegate
            {
                // Lazy creation of the flex form - the first time this is called
                if (flexForm == null)
                {
                    flexForm = new FlexConfigForm();

                    // Position the parameters form a small distance from the top right corner of the screen
                    flexForm.StartPosition = FormStartPosition.Manual;
                    var bounds = Screen.PrimaryScreen.WorkingArea;
                    flexForm.Location = new Point(bounds.X + bounds.Width - 500 - PositionOffset, bounds.Y + PositionOffset);

                    flexForm.FormClosing += new FormClosingEventHandler(flexForm_FormClosing);
                    flexForm.Show();
                }

                flexForm.DisparityNode = sender;
            }));
        }

        #region Video source selection

		static private FileVideoSource fileSource = new FileVideoSource()    { Name = "Stereo Video File" };
        static private ImageVideoSource imageSource = new ImageVideoSource() { Name = "Stereo Image" };

        private VideoSourceBase _videoSource;
        private VideoSourceBase VideoSource
        {
            get { return _videoSource; }
            set
            {
                if (value != _videoSource) // actually changed?
                {
                    Debug.WriteLine(String.Format("VideoSource set to \"{0}\".", value));
                    _videoSource = value;
                    videoSourceComboBox.SelectedItem = value;
                }
            }
        }
        
        private VideoSize _videoResolution;
        private VideoSize VideoResolution
        {
            get { return _videoResolution; }
            set
            {
				if (value != null)
				{
					Debug.WriteLine(String.Format("VideoResolution set to \"{0}\".", value));
					_videoResolution = value;

					// select in combo box, add first if necessary
					if (!videoResolutionComboBox.Items.Contains(value))
						videoResolutionComboBox.Items.Add(value);
					videoResolutionComboBox.SelectedItem = value;
				}
            }
        }
        
        private void PopulateVideoSourceComboBox()
        {
            videoSourceComboBox.Items.Clear();
            videoSourceComboBox.Items.Add(fileSource);
            videoSourceComboBox.Items.Add(imageSource);
			PopulateImageSources();
            videoSourceComboBox.SelectedIndex = 0;
        }

		private void PopulateImageSources()
		{
			var dataDir = new DirectoryInfo("../../../../Data/input");
			if (!dataDir.Exists) return;

			// populate stereo videos
			foreach (var dir in dataDir.GetDirectories())
			{
				var leftFiles = dir.GetFiles("L0001.*");
				if (leftFiles.Length == 0) continue;

				var rightFiles = dir.GetFiles("R0001.*");
				if (rightFiles.Length == 0) continue;

				var leftFile = leftFiles[0];
				var rightFile = rightFiles[0];
				var leftImage = Path.Combine(leftFile.DirectoryName, "L{0:0000}" + leftFile.Extension);
				var rightImage = Path.Combine(rightFile.DirectoryName, "R{0:0000}" + rightFile.Extension);

				videoSourceComboBox.Items.Add(new ImageVideoSource()
				{
					LeftFilename = leftImage,
					RightFilename = rightImage,
					Name = String.Format("Stereo Video: {0} ({1} frames)", dir.Name, dir.GetFiles("L????.*").Length)
				});
			}

			// populate stereo stills
			foreach (var dir in dataDir.GetDirectories())
			{
				var leftFiles = dir.GetFiles("left.*");
				if (leftFiles.Length == 0) continue;

				var rightFiles = dir.GetFiles("right.*");
				if (rightFiles.Length == 0) continue;

				var leftImage = leftFiles[0];
				var rightImage = rightFiles[0];

				videoSourceComboBox.Items.Add(new ImageVideoSource()
				{
					LeftFilename = leftImage.FullName,
					RightFilename = rightImage.FullName,
					Name = String.Format("Stereo Image: {0}", dir.Name)
				});
			}
		}

        private void UpdateFormElements()
        {
            startButton.Enabled = VideoSource.EnableStartButton();
        }

        private void PopulateVideoResolutionComboBox()
        {
            videoResolutionComboBox.Items.Clear();
            videoResolutionComboBox.Items.Add(new VideoSize() { TargetSize = new Size(240, 180) });
            videoResolutionComboBox.Items.Add(new VideoSize() { TargetSize = new Size(320, 240) });
            videoResolutionComboBox.Items.Add(new VideoSize() { TargetSize = new Size(480, 360) });
            videoResolutionComboBox.Items.Add(new VideoSize() { TargetSize = new Size(640, 480) });
            videoResolutionComboBox.Items.Add(new VideoSize() { TargetSize = new Size(752, 480) });
            videoResolutionComboBox.SelectedIndex = 1;
        }

        private void videoSourceComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            VideoSource = videoSourceComboBox.SelectedItem as VideoSourceBase;
            VideoSource.SelectSource();
			videoFilenameTextBox.Text = VideoSource.StatusText;
			VideoResolution = VideoSource.PreferredSize;
            UpdateFormElements();
        }

        private void videoResolutionComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            VideoResolution = videoResolutionComboBox.SelectedItem as VideoSize;
        }

        #endregion

        #region Start/Stop video playback

        private void startButton_Click(object sender, EventArgs e)
        {
            // enable/disable controls
            videoSourceComboBox.Enabled = false;
            videoResolutionComboBox.Enabled = false;
            videoFilenameTextBox.Enabled = false;
            startButton.Enabled = false;
            stopButton.Enabled = true;

            new Thread(StartVideo).Start();
        }

        private void stopButton_Click(object sender, EventArgs e)
        {
            new Thread(StopVideo).Start();
        }

        private Topology topology = null;
        private Scheduler scheduler = null;

        // NB: blocks
        private void StartVideo()
        {
            try
            {
                // Set up the topology and scheduler
                topology = Topology.MainInstance;
                topology.Clear();
                scheduler = new SimpleDequeScheduler(topology);

                // Set up stereo source
                Invoke(setupCapture, topology);

                // Start the scheduler
                scheduler.Setup();
                scheduler.StartWorkers(4);
                scheduler.Join();
            }
            catch (Exception e)
            {
                MessageBox.Show(
                    String.Format("{0}: {1}\n\n{2}\n\n(MainFlexForm.StartVideo)", e.GetType().Name, e.Message, e),
                    String.Format("Error: {0}", e.GetType()), MessageBoxButtons.OK, MessageBoxIcon.Error);
                Application.Exit();
                return;
            }

            // invoke stop
            if(Created && !Disposing)
                this.Invoke(new Action(delegate() { InvokeOnClick(stopButton, new EventArgs()); }));
        }

        private void StopVideo()
        {
            scheduler.Stop();
            topology.Clear();

            // TODO: close flex forms and clean up
            this.BeginInvoke(new Action(delegate()
                {
                    // enable/disable controls
                    videoSourceComboBox.Enabled = true;
                    videoResolutionComboBox.Enabled = true;
                    startButton.Enabled = true;
                    stopButton.Enabled = false;
                    UpdateFormElements();
                }));
        }

        private void SetupCapture(Topology topology)
        {
            // create the video source
            OutputPin videoSourceL = null;
            OutputPin videoSourceR = null;
            VideoSource.SetupSource(topology, out videoSourceL, out videoSourceR);

            /*var lImageSink = new ImageSequenceSink("Sample/left_{0}.png");
            var rImageSink = new ImageSequenceSink("Sample/right_{0}.png");
            topology.Connect(videoSourceL, lImageSink);
            topology.Connect(videoSourceR, rImageSink);*/

            // resample the input videos
            var resampleL = new LibAvConvert(VideoResolution.TargetSize) { Mode = VideoLib.LibAv.ResamplingMode.Bilinear };
            var resampleR = new LibAvConvert(VideoResolution.TargetSize) { Mode = VideoLib.LibAv.ResamplingMode.Bilinear };

            //SobelFilter tempSobelL = new SobelFilter();
            //SobelFilter tempSobelR = new SobelFilter();

            topology.Connect(videoSourceL, resampleL);
            topology.Connect(videoSourceR, resampleR);

            // set up anaglyph node
            var shiftNode = new ImageShiftNode();
            topology.Connect(resampleR, shiftNode);
            var anaglyphFilter = new AnaglyphFilter(AnaglyphMethod.Colour);
            topology.Connect(resampleL, anaglyphFilter["inL"]);
            topology.Connect(shiftNode, anaglyphFilter["inR"]);

            // set up disparity estimation node
            disparitiesNode = new DisparityEstimationNode();
            topology.Connect(resampleL, disparitiesNode["inL"]);
			topology.Connect(shiftNode, disparitiesNode["inR"]);
			//topology.Connect(
			//    disparitiesNode,
			//    new FloatToGreyNode() { ScaleFactor = 255 },
			//    new ImageSequenceSink("TDCB-Disparities-{0:D4}.png"));

            // synchronise the outputs
            var sync = new SyncFilter(4);
            topology.Connect(resampleL, sync["in0"]);
            topology.Connect(resampleR, sync["in1"]);
			topology.Connect(disparitiesNode, new FloatToGreyNode() { ScaleFactor = 255 }, sync["in2"]);
            topology.Connect(anaglyphFilter, sync["in3"]);

			//// save result
			//var greyDisp = new FloatToGreyNode() { ScaleFactor = 60 * 4 };
			//topology.Connect(disparitiesNode, greyDisp, new ImageSequenceSink("DisparityOut.png"));
			//topology.Connect(greyDisp, new ImageSequenceSink("DisparityOut.gif"));

            // connect outputs to displays
            topology.Connect(sync["out0"], new VideoDisplay(videoControl1, "Left"));
            topology.Connect(sync["out1"], new VideoDisplay(videoControl2, "Right"));
            topology.Connect(sync["out2"], new VideoDisplay(videoControl3, "Depth Map"));
            topology.Connect(sync["out3"], new VideoDisplay(videoControl4, "Anaglyph"));

            disparitiesNode.Initialized += new DisparityEstimationNode.InitializedHandler(disparitiesNode_Initialized);

            List<List<object>> paramTargets = new List<List<object>>();

            paramTargets.Add(new List<object>() { shiftNode });
            //paramTargets.Add(new List<object>() { depthBlurFilter });
            //paramTargets.Add(new List<object>() { tempSobelL, tempSobelR });

            // Lazy creation of the auxilliary form the first time this is called
            if (auxForm == null)
            {
                auxForm = new DynamicParametersForm(paramTargets);

                // Position the auxilliary form a small distance from the lower right corner of the screen
                auxForm.StartPosition = FormStartPosition.Manual;
                var bounds = Screen.PrimaryScreen.WorkingArea;
                auxForm.Location = new Point(bounds.X + bounds.Width - auxForm.Width - PositionOffset,
                    bounds.Y + bounds.Height - auxForm.Height - PositionOffset);

                auxForm.FormClosing += new FormClosingEventHandler(auxForm_FormClosing);
                auxForm.Show();
            }
            else auxForm.Renew(paramTargets);
        }

        #endregion

        private void MainFlexForm_KeyDown(object sender, KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.MediaPlayPause:
                case Keys.Space: // toggle playback
                    if (startButton.Enabled)
                    {
                        startButton_Click(sender, null);
                        e.Handled = true;
                    }
                    else if (stopButton.Enabled)
                    {
                        stopButton_Click(sender, null);
                        e.Handled = true;
                    }
                    break;

                case Keys.D1: // decrease resolution
                    if (startButton.Enabled)
                    {
                        if (videoResolutionComboBox.SelectedIndex > 0)
                            videoResolutionComboBox.SelectedIndex--;
                        e.Handled = true;
                    }
                    break;

                case Keys.D2: // increase resolution
                    if (startButton.Enabled)
                    {
                        if (videoResolutionComboBox.SelectedIndex < videoResolutionComboBox.Items.Count - 1)
                            videoResolutionComboBox.SelectedIndex++;
                        e.Handled = true;
                    }
                    break;
            }
        }
    }
}
