﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Windows.Forms;
using VideoLib.Parameters;
using VideoLib.Stereo.GpGpu;

namespace RealTimeStereoTestViewer
{
    /// <summary>
    /// A basic, flexible, disparity map generation configuration form
    /// </summary>
    public partial class FlexConfigForm : Form
    {
        private const string SettingsPath = "flex_settings.dat";

        // The disparity node that this form is attached to
        private DisparityEstimationNode disparityNode;

        // These are populated from the controls
        private Dictionary<TechniqueType, TechniqueConfigurationControl> techniqueControls = null;
        private Array techniqueTypeVals;

        // The persistent settings of parameters
        private ParameterValueDictionary valueDictionary;

        // This is true when the form is ready to handle itemAdded and/or itemRemoved events
        private bool readyForEvents = false;
        private RegisteredCollection<IStereoNode>.ItemModificationHandler itemAdded;
        private RegisteredCollection<IStereoNode>.ItemModificationHandler itemRemoved;

        public FlexConfigForm()
        {
            InitializeComponent();

            // Retrieve the values of the techniques for later reference
            techniqueTypeVals = Enum.GetValues(typeof(TechniqueType));
            techniqueControls = new Dictionary<TechniqueType, TechniqueConfigurationControl>();

            itemAdded = new RegisteredCollection<IStereoNode>.ItemModificationHandler(ActiveItemAdded);
            itemRemoved = new RegisteredCollection<IStereoNode>.ItemModificationHandler(ActiveItemRemoved);

            Load += new EventHandler(FlexConfigForm_Load);
            FormClosing += new FormClosingEventHandler(FlexConfigForm_FormClosing);

            DeserializeSettings(SettingsPath);
        }

        #region UI event handlers

        private void FlexConfigForm_Load(object sender, EventArgs e)
        {
            if (disparityNode != null)
                RefreshAll();

            readyForEvents = true;
        }
        private void FlexConfigForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            readyForEvents = false;

            if (disparityNode != null && disparityNode.ActiveParamObjects != null)
            {
                disparityNode.ActiveParamObjects.ItemAdded -= itemAdded;
                disparityNode.ActiveParamObjects.ItemRemoved -= itemRemoved;
            }

            disparityNode = null;

            SerializeSettings(SettingsPath);
        }
        void FlexConfigForm_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.S && e.Control)
            {
                // Create a save file dialog for serializing the settings
                var sfd = new SaveFileDialog();
                sfd.InitialDirectory = Environment.CurrentDirectory;
                sfd.Filter = "Data files (*.dat)|*.dat";

                if (sfd.ShowDialog() == DialogResult.OK)
                    SerializeSettings(sfd.FileName);

                e.Handled = true;
            }
            if (e.KeyCode == Keys.O && e.Control)
            {
                // Create an open file dialog for de-serializing the settings
                var ofd = new OpenFileDialog();
                ofd.InitialDirectory = Environment.CurrentDirectory;
                ofd.Filter = "Data files (*.dat)|*.dat";

                if (ofd.ShowDialog() == DialogResult.OK)
                {
                    DeserializeSettings(ofd.FileName);

                    // Now apply the settings to the active StereoNodes
                    foreach (TechniqueConfigurationControl control in techniqueControls.Values)
                    {
                        var paramControls = control.GetParameterControls();
                        foreach (DynamicParametersControl paramControl in paramControls)
                        {
                            // Apply the settings to the primary target of the control
                            valueDictionary.ApplySettings(paramControl.Target);

                            // Apply the settings to any followers of the primary target
                            if (paramControl.Followers != null)
                                paramControl.Followers.ForEach(obj => valueDictionary.ApplySettings(obj));
                                    
                            paramControl.RefreshValues();
                        }
                    }
                }
                e.Handled = true;
            }
        }

        private void SerializeSettings(string path)
        {
            using (FileStream fs = File.Open(path, FileMode.Create))
            {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize(fs, valueDictionary);
                fs.Close();
            }
        }
        private void DeserializeSettings(string path)
        {
            if (File.Exists(SettingsPath))
            {
                bool failed = false;
                using (FileStream fs = File.Open(path, FileMode.Open))
                {
                    BinaryFormatter bf = new BinaryFormatter();
                    try
                    {
                        valueDictionary = (ParameterValueDictionary)bf.Deserialize(fs);
                    }
                    catch (TypeLoadException) { failed = true; }
                    catch (SerializationException) { failed = true; }
                    finally { fs.Close(); }
                }

                if (failed)
                    valueDictionary = new ParameterValueDictionary();
            }
            else valueDictionary = new ParameterValueDictionary();
        }

        #endregion

        private void ActiveItemAdded(RegisteredCollection<IStereoNode> sender, IStereoNode target)
        {
            // Apply current settings to the target and followers
            valueDictionary.ApplySettings(target);
            if (target.FollowerNodes != null)
                target.FollowerNodes.ForEach(obj => valueDictionary.ApplySettings(obj));

            // Search for the target, and fill the relevant tab page
            BeginInvoke(new Action(delegate
            {
                if (techniqueControls.Keys.Contains(target.Type))
                {
                    techniqueControls[target.Type].ShowParameters(target);
                }
                else
                {
                    // other techniques
                    for (int i = 0; i < techniqueTypeVals.Length; ++i)
                    {
                        if ((TechniqueType)techniqueTypeVals.GetValue(i) == target.Type)
                        {
                            //List<TechniqueDescription> descs = disparityNode.GetTechniques(target.Type);

                            var control = new DynamicParametersControl(target, target.FollowerNodes, true);
                            control.AutoSize = true;
                            control.AutoSizeMode = AutoSizeMode.GrowAndShrink;
                            mainLayoutPanel.Controls.Add(control);
                            break;
                        }
                    }
                }
            }));
        }
        private void ActiveItemRemoved(RegisteredCollection<IStereoNode> sender, IStereoNode target)
        {
            // Store the current settings of the target for later reference
            valueDictionary.StoreSettings(target);

            BeginInvoke(new Action(delegate
            {
                if (techniqueControls.Keys.Contains(target.Type))
                {
                    techniqueControls[target.Type].HideParameters(target);
                }
                else
                {
                    // Search for the target, and clear the relevant tab page
                    for (int i = 0; i < techniqueTypeVals.Length; ++i)
                    {
                        if ((TechniqueType)techniqueTypeVals.GetValue(i) == target.Type)
                        {
                            // Populate a list of matching controls to remove
                            List<Control> toRemove = new List<Control>();
                            foreach (Control control in mainLayoutPanel.Controls)
                                if (control is DynamicParametersControl)
                                    if (((DynamicParametersControl)control).Target == target)
                                        toRemove.Add(control);

                            // Remove all of the matching controls
                            foreach (Control control in toRemove)
                                mainLayoutPanel.Controls.Remove(control);

                            break;
                        }
                    }
                }
            }));
        }


        private void RefreshAll()
        {
            if (disparityNode != null)
            {
                // suspend layout to speed up adding lots of controls
                mainLayoutPanel.SuspendLayout();
                this.SuspendLayout();
                
                // clear everything
                mainLayoutPanel.Controls.Clear();
                techniqueControls.Clear();

				for (int i = 0; i < techniqueTypeVals.Length - 1; ++i)
                {
                    TechniqueType techniqueType = (TechniqueType)techniqueTypeVals.GetValue(i);

                    var techBox = new TechniqueConfigurationControl(disparityNode);
                    techBox.Title = techniqueType.ToString();
                    techBox.Techniques = disparityNode.GetTechniques(techniqueType);

                    mainLayoutPanel.Controls.Add(techBox);
                    techniqueControls[techniqueType] = techBox;
                }

                // Register for node change updates
                if(disparityNode.ActiveParamObjects != null)
                    disparityNode.ActiveParamObjects.RegisterHandlers(itemAdded, itemRemoved, true);

                // resume layout
                mainLayoutPanel.ResumeLayout(true);
                this.ResumeLayout(true);
            }
        }

        /// <summary>
        /// The disparity node targetted by the form
        /// </summary>
        public DisparityEstimationNode DisparityNode
        {
            get { return disparityNode; }
            set 
            {
                if (readyForEvents && disparityNode != value)
                {
                    // Remove handlers from current (outgoing) disparity estimation node)
                    if (disparityNode != null && disparityNode.ActiveParamObjects != null)
                    {
                        disparityNode.ActiveParamObjects.ItemAdded -= itemAdded;
                        disparityNode.ActiveParamObjects.ItemRemoved -= itemRemoved;
                    }

                    // Set the disparity node reference before refreshing the whole list of techniques
                    disparityNode = value;
                    RefreshAll();
                }
                else disparityNode = value;
            }
        }
    }
}
