﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace RealTimeStereoTestViewer
{
    /// <summary>
    /// A quite simple collection of <see cref="DynamicParametersControl"/> controls, which refer to one or more
    /// targets or target-follower collections.
    /// </summary>
    public partial class DynamicParametersForm : Form
    {
        /// <summary>
        /// Creates a new <see cref="DynamicParametersForm"/>, with parameter controls to target each element of <paramref name="owners"/>.
        /// </summary>
        /// <param name="owners">A list of parameter targets to control</param>
        public DynamicParametersForm(List<object> owners)
        {
            InitializeComponent();
            Renew(owners);

            this.AutoSize = true;
            this.AutoSizeMode = AutoSizeMode.GrowAndShrink;
        }
        /// <summary>
        /// Creates a new <see cref="DynamicParametersForm"/>, with parameter controls to target each owner and set of followers in
        /// <paramref name="ownersWithFollowers"/>.
        /// </summary>
        /// <param name="ownersWithFollowers">A list of parameter targets and followers to control: 
        /// The first element of each sub-list is a target, and the remainder are followers</param>
        public DynamicParametersForm(List<List<object>> ownersWithFollowers)
        {
            InitializeComponent();
            Renew(ownersWithFollowers);

            this.AutoSize = true;
            this.AutoSizeMode = AutoSizeMode.GrowAndShrink;
        }

        /// <summary>
        /// Renews the contents of the control to refer to a new set of targets, given by <paramref name="owners"/>
        /// </summary>
        /// <param name="owners">The new list of targets to refer to</param>
        public void Renew(List<object> owners)
        {
            mainLayoutPanel.Controls.Clear();
            foreach (object obj in owners)
            {
                var control = new DynamicParametersControl(obj);
                control.AutoSize = true;
                control.AutoSizeMode = AutoSizeMode.GrowAndShrink;
                mainLayoutPanel.Controls.Add(control);
            }

            // Build a string identifying the objects that are targetted
            StringBuilder sb = new StringBuilder("[");
            for (int i = 0; i < owners.Count; ++i)
            {
                sb.Append(owners[i].ToString());

                if (i < owners.Count - 1)
                    sb.Append(", ");
            }
            sb.Append(" - Parameters]");
            this.Text = sb.ToString();
        }

        /// <summary>
        /// Renews the contents of the control to refer to a new set of targets, given by <paramref name="ownersWithFollowers"/>
        /// </summary>
        /// <param name="ownersWithFollowers">The new list of parameter targets and followers to control: 
        /// The first element of each sub-list is a target, and the remainder are followers</param>
        public void Renew(List<List<object>> ownersWithFollowers)
        {
            mainLayoutPanel.Controls.Clear();
            foreach (List<object> list in ownersWithFollowers)
            {
                List<object> followers = new List<object>(list.Where((o, idx) => idx > 0));

                var control = new DynamicParametersControl(list[0], followers, true);
                control.AutoSize = true;
                control.AutoSizeMode = AutoSizeMode.GrowAndShrink;
                mainLayoutPanel.Controls.Add(control);
            }

            // Build a string identifying the objects that are targetted
            StringBuilder sb = new StringBuilder("[");
            for (int i = 0; i < ownersWithFollowers.Count; ++i)
            {
                sb.Append(ownersWithFollowers[i][0].ToString());

                if (i < ownersWithFollowers.Count - 1)
                    sb.Append(", ");
            }
            sb.Append(" - Parameters]");
            this.Text = sb.ToString();
        }
    }
}
