﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using VideoLib.Parameters;

namespace RealTimeStereoTestViewer
{
    /// <summary>
    /// A dynamically built parameters control, which groups together controls to modify each parameter
    /// found for a given target object, optionally modifying 'follower objects' in-sync
    /// </summary>
    public partial class DynamicParametersControl : UserControl
    {
        private readonly object theTarget;
        private readonly List<object> followers;
        private readonly bool group;
        private FlowLayoutPanel layoutPanel;

        /// <summary>
        /// Creates a new instance of <see cref="DynamicParametersControl"/>, referring to the given target object,
        /// with no followers
        /// </summary>
        /// <param name="obj">The object to control the parameters of</param>
        public DynamicParametersControl(object obj) : this(obj, null, true) { }

        /// <summary>
        /// Creates a new instance of <see cref="DynamicParametersControl"/>, referring to the given target object,
        /// with each element of <paramref name="followerObjects"/> modified in-sync with <paramref name="obj"/>
        /// </summary>
        /// <param name="obj">The object to control the parameters of</param>
        /// <param name="followerObjects">The set of objects to modify in-sync with <paramref name="obj"/></param>
        /// <param name="group">If <c>true</c>, the inner controls are tightly packed, for use when grouping the whole into another control.
        /// Otherwise the margins are normal</param>
        public DynamicParametersControl(object obj, List<object> followerObjects, bool group)
        {
            InitializeComponent();

            theTarget = obj;
            followers = followerObjects;
            this.group = group;

            // Check that all objects in 'followerObjects' are of the same type
            Type objType = obj.GetType();
            if (followerObjects != null)
            {
                bool invalid = false;
                foreach (object follower in followerObjects)
                    if (!follower.GetType().Equals(objType))
                    {
                        invalid = true;
                        break;
                    }
                if (invalid)
                    throw new FormatException("The types of 'followObjects' should all be the same as 'obj'");
            }

            List<ParameterInfo> paramsInfo = ParameterReflector.EnumerateParameters(objType);
            layoutPanel = new FlowLayoutPanel();

            // Only add anything to the control if the number of parameters is non-zero
            if (paramsInfo.Count > 0)
            {
                layoutPanel.FlowDirection = FlowDirection.TopDown;
                layoutPanel.Dock = DockStyle.Fill;
                layoutPanel.Font = new Font(layoutPanel.Font.FontFamily, 8.0f);
                //layoutPanel.BackColor = Color.White;
                layoutPanel.AutoSize = true;
                layoutPanel.AutoSizeMode = AutoSizeMode.GrowAndShrink;

                if (group)
                {
                    GroupBox groupPanel = new GroupBox();
                    groupPanel.Text = GetId(obj);
                    groupPanel.Font = new Font(groupPanel.Font.FontFamily, 8.0f, FontStyle.Bold);
                    groupPanel.AutoSize = true;
                    groupPanel.AutoSizeMode = AutoSizeMode.GrowAndShrink;
                    groupPanel.Controls.Add(layoutPanel);
                    Controls.Add(groupPanel);
                }
                else
                {
                    Margin = new Padding(0);
                    layoutPanel.Margin = new Padding(0);
                    Controls.Add(layoutPanel);
                }
                
                // Add the controls to the container
                foreach (ParameterInfo pInfo in paramsInfo)
                    AddParamControl(obj, followerObjects, pInfo);
            }
        }

        private void AddParamControl(object target, List<object> followers, ParameterInfo pInfo)
        {
            Control control = null;

            if (pInfo.ParamType == typeof(float))
                control = new FloatParameterControl(pInfo, target, followers);
            else if (pInfo.ParamType == typeof(int))
                control = new IntegerParameterControl(pInfo, target, followers);

            if(!this.group) control.Margin = new Padding(0);
            layoutPanel.Controls.Add(control);
        }
        private string GetId(object obj)
        {
            string[] names = obj.GetType().ToString().Split('.');

            // Do some helpful pascal-case reformatting (split the words on capital letter)
            StringBuilder sb = new StringBuilder();
            for (int index = 0; index < names[names.Length - 1].Length; ++index)
            {
                char next = names[names.Length - 1][index];

                if (Char.IsUpper(next))
                    sb.Append(' ');

                sb.Append(next);
            }

            return sb.ToString();
        }

        /// <summary>
        /// Gets the object that is the primary target of the parameter controls inside this control
        /// </summary>
        public object Target { get { return theTarget; } }

        /// <summary>
        /// Gets the objects that follow the parameters which are set inside this control
        /// </summary>
        public List<object> Followers { get { return followers; } }

        /// <summary>
        /// Refreshes the values of the derived parameter controls
        /// </summary>
        public void RefreshValues()
        {
            foreach (Control c in layoutPanel.Controls)
            {
                if (c is IParameterControl)
                {
                    ((IParameterControl)c).RefreshValue();
                }
            }
        }
    }
}
