// $Id: ManagedAggregators.h 65 2010-03-18 17:06:22Z cr333 $
#pragma once

#include "CostSpace.h"
#include "ExecutionStream.h"
#include "InputImage.h"
#include "TiledGrids.h"

namespace Stereo { namespace GpGpuLib {

/// <summary>
/// A set of methods for aggregating (locally optimizing) cost spaces
/// </summary>
public ref class ManagedAggregators
{
public:

	/// <summary>
	/// Runs a box aggregation filter on <paramref name="costSpace"/>, storing intermediate results in <paramref name="temp"/> as necessary, and storing
	/// the final result in <paramref name="costSpace"/>
	/// </summary>
	static void RunBoxKernel(CostSpace^ costSpace, int radius, CostSpace^ temp, ExecutionStream^ stream);

	/// <summary>
	/// Runs a shiftable window aggregation filter on <paramref name="costSpace"/>, storing intermediate results in <paramref name="temp"/> as necessary, 
	/// and storing the final result in <paramref name="costSpace"/>. The shiftable window is defined by <paramref name="boxRadius"/> and <paramref name="minRadius"/>
	/// </summary>
	static void RunShiftKernel(CostSpace^ costSpace, int boxRadius, int minRadius, CostSpace^ temp, ExecutionStream^ stream);

	/// <summary>
	/// Runs the naive implementation of DCB aggregation on <paramref name="costSpace"/>, and storing the result
	/// back in <paramref name="costSpace"/>. The filter is defined by <paramref name="sigmaS"/> and <paramref name="sigmaC"/>.
	/// </summary>
	static void RunNaiveDcbKernel(CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR, unsigned int radius, float sigmaS, float sigmaC, ExecutionStream^ stream);

	/// <summary>
	/// Runs a DCB grid aggregation filter on <paramref name="costSpace"/>, using <paramref name="gridTexture"/> as intermediate storage, and storing the result
	/// back in <paramref name="costSpace"/>. The filter is defined by <paramref name="sigmaS"/> and <paramref name="sigmaC"/>.
	/// </summary>
	static void RunDcbGridKernel(CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR, InputImage^ gridTexture, float sigmaS, float sigmaC, ExecutionStream^ stream);

	// with temporal smoothing
	static void RunTdcbGridKernel(CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR, array<InputImage^>^ gridTextures,
		float sigmaS, float sigmaC, int weighting, float wa, float wb, ExecutionStream^ stream);

	// with partial colour support
	static void RunDcbGrid2Kernel(CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR, TiledGrids^ grids, float sigmaS, float sigmaC1, float sigmaC2, ExecutionStream^ stream);

	/// <summary>
	/// Runs a colour-weighted correlation aggregation filter on <paramref name="costSpace"/>, storing the result
	/// back in <paramref name="costSpace"/>. The filter is defined by <paramref name="gammaP"/> and <paramref name="gammaC"/>.
	/// </summary>
	/// <remarks>This method is very slow. For an approximation which is much faster, try <see cref="RunDcbGridKernel"/></remarks>
	static void RunYoonKweonKernel(CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR, int radius, float gammaP, float gammaC, ExecutionStream^ stream);
};

} }