#pragma once

#include <builtin_types.h>
#include "GpuExecutionStream.h"

enum GpuCostSpaceType
{
	COST_SPACE_TYPE_ANY,		// The cost space type is not specified
	COST_SPACE_TYPE_SINGLE		// Single float costs, packed up in x-y-depth order into a 3D cudaPitchedPtr
};

// A class encapsulating a couple of different formats of cost space grid
class GpuCostSpace
{
private:
	cudaPitchedPtr gpuGrid;
	cudaExtent extent;
	int w, h, d;

	GpuCostSpaceType gridType;
	
	void CopyDeviceSpaceToHost(const cudaPitchedPtr & device, float* hostPtr);

public:
	GpuCostSpace() : w(0), h(0), d(0) { gpuGrid.ptr = 0; }
	~GpuCostSpace() { Destroy(); }

	void Create(GpuCostSpaceType type, int width, int height, int depth);
	void Destroy();

	// This method allows nodes to operate without requiring in-place modification, swapping the resultant data
	// and responsibility to the output grid
	void SwapData(GpuCostSpace & other);

	// Sizes to match and copies the data from another cost space into this one
	void AsyncCopyFrom(const GpuCostSpace* const other, const GpuExecutionStream & stream);

	// Sizes the cost space to match another
	void SizeToMatch(const GpuCostSpace & other);

	cudaPitchedPtr Get() const { return gpuGrid; }
	GpuCostSpaceType GetType() const { return gridType; }

	int GetWidth() const { return w; }
	int GetHeight() const { return h; }
	int GetDepth() const { return d; }

	// Downsamples the cost space given by the given factor, storing the result in this cost space
	void DownsampleFrom(const GpuCostSpace* const other, int downsampleFactor);

	// Copies the complete cost space into host memory given, which should be big enough
	void CopyDataOut(float* result);
};