// $Id: CostSpace.h 65 2010-03-18 17:06:22Z cr333 $
#pragma once

#include "GpuCostSpace.h"
#include "ExecutionStream.h"

namespace Stereo {
namespace GpGpuLib {

using namespace System;
using namespace System::Runtime::InteropServices;

/// <summary>
/// The format of the cost space.
/// </summary>
public enum class CostSpaceType
{
	/// <summary>Not yet determined</summary>
	Any = COST_SPACE_TYPE_ANY,

	/// <summary>Costs stored as single-precision floats</summary>
	Single = COST_SPACE_TYPE_SINGLE
};

/// <summary>
/// A simple wrapper around the unmanaged cost space type for use in managed code.
/// </summary>
public ref class CostSpace
{
private:
	GpuCostSpace* costSpace;

internal:
	cudaPitchedPtr GetPointer() { return costSpace->Get(); }

public:
	CostSpace()
	{
		costSpace = new GpuCostSpace();
	}

	~CostSpace()
	{
		if(costSpace != 0)
		{
			delete costSpace;
		}
		costSpace = 0;
	}

	GpuCostSpace* Get() { return costSpace; }
	
	property int Width { int get() { return costSpace->GetWidth(); } }
	property int Height { int get() { return costSpace->GetHeight(); } }
	property int Depth { int get() { return costSpace->GetDepth(); } }
	property CostSpaceType Type { CostSpaceType get() { return (CostSpaceType)(int)costSpace->GetType(); } }

	void SizeCostSpace(CostSpaceType type, int w, int h, int d)
	{
		costSpace->Create((GpuCostSpaceType)((int)type), w, h, d);
	}

	int GetWidth() { return costSpace->GetWidth(); }
	int GetHeight() { return costSpace->GetHeight(); }
	int GetDepth() { return costSpace->GetDepth(); }

	/// <summary>
	/// Copies the contents of <paramref name="source"/> into the cost space, resizing it as necessary
	/// </summary>
	void AsyncCopyFrom(CostSpace^ source, ExecutionStream^ stream)
	{
		costSpace->AsyncCopyFrom(source->Get(), *stream->Get());
	}

	/// <summary>
	/// Copies out the cost space into a (large!) managed array (intended for debugging, not performance code)
	/// <para>Warning: The implementation includes per-call allocation and freeing of memory</para>
	/// </summary>
	void CopyDataOut([Out] array<float>^% data)
	{
		data = gcnew array<float>(costSpace->GetWidth() * costSpace->GetHeight() * costSpace->GetDepth());

		pin_ptr<float> pinData = &data[0];
		costSpace->CopyDataOut(pinData);
	}
};

} } // Stereo::GpGpuLib