﻿// $Id: TechniqueDescription.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace VideoLib.Stereo.GpGpu
{
	/// <summary>
	/// Describes an available technique (created internally to VideoLib.Stereo.GpGpu by a 
	/// <see cref="StereoNodeFactory"/>). This should only be created from within VideoLib.Stereo.GpGpu (even 
	/// though it is publicly visible).
	/// </summary>
	public class TechniqueDescription
	{
		private readonly string name;
		private readonly string typeName;
		private readonly TechniqueType type;
		private readonly StereoNodeFactory factory;
		private bool isValid, isActive;

		/// <summary>
		/// Occurs when the valid state of the technique is changed.
		/// </summary>
		public EventHandler ValidChanged;

		/// <summary>
		/// Occurs when the technique is made active or inactive.
		/// </summary>
		public EventHandler ActiveChanged;

		/// <summary>
		/// Initializes a new instance of the <see cref="TechniqueDescription"/> class.
		/// </summary>
		/// <param name="theName">The friendly name of the technique.</param>
		/// <param name="theTypeName">The name of the factory type.</param>
		/// <param name="fact">The factory used to instantiate nodes of the technique.</param>
		/// <param name="theType">The type of process that the technique referred to implements.</param>
		internal TechniqueDescription(string theName, string theTypeName, StereoNodeFactory fact, TechniqueType theType)
		{
			name = theName;
			typeName = theTypeName;
			factory = fact;
			type = theType;

			isValid = true;
			isActive = false;
		}

		/// <summary>
		/// Triggers the <see cref="ValidChanged"/> event.
		/// </summary>
		private void OnValidChanged()
		{
			if (ValidChanged != null)
				ValidChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// Triggers the <see cref="ActiveChanged"/> event.
		/// </summary>
		private void OnActiveChanged()
		{
			if (ActiveChanged != null)
				ActiveChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// Gets the friendly name of the technique.
		/// </summary>
		public string Name
		{
			get { return name; }
		}

		/// <summary>
		/// Gets the unqualified name of the type of factory creating the technique.
		/// </summary>
		public string TypeName
		{
			get { return typeName; }
		}

		/// <summary>
		/// Gets the internal factory used to instantiate the technique.
		/// </summary>
		internal StereoNodeFactory Factory
		{
			get { return factory; }
		}

		/// <summary>
		/// Gets the type of technique.
		/// </summary>
		public TechniqueType Type
		{
			get { return type; }
		}

		/// <summary>
		/// Indicates if the technique can be applied given the current choice of datatypes.
		/// </summary>
		/// <value><c>true</c> if this instance is valid; otherwise, <c>false</c>.</value>
		/// <remarks>
		/// Publicly read-only, internally mutable.
		/// </remarks>
		public bool IsValid
		{
			get { return isValid; }
			internal set 
			{ 
				isValid = value; 
				OnValidChanged(); 
			}
		}

		/// <summary>
		/// Indicates if this technique is currently in use.
		/// </summary>
		/// <value><c>true</c> if this instance is active; otherwise, <c>false</c>.</value>
		/// <remarks>
		/// Publicly read-only, internally mutable.
		/// </remarks>
		public bool IsActive
		{
			get { return isActive; }
			internal set
			{
				isActive = value;
				OnActiveChanged();
			}
		}

		/// <summary>
		/// Returns a friendly name to identify the technique.
		/// </summary>
		/// <returns>The friendly name of the technique</returns>
		public override string ToString() { return name; }

		#region Internal static information about TechniqueType

		internal const int NTechniqueTypes = 7;
		private static Type[] techniqueNodeTypes = new Type[]
		{
			typeof(DownsamplerStereoNode),
			typeof(PreProcessorStereoNode),
			typeof(CostComputerStereoNode),
			typeof(AggregatorStereoNode),
			typeof(MaximizerStereoNode),
			typeof(UpsamplerStereoNode),
			typeof(StereoNode)
		};

		internal static Type[] TechniqueNodeTypes
		{
			get { return techniqueNodeTypes; }
		}

		#endregion
	}

	/// <summary>
	/// The different classes of techniques.
	/// </summary>
	public enum TechniqueType
	{
		Downsampler,
		PreProcessor,
		CostComputer,
		Aggregator,
		Maximizer,
		Upsampler,
		Unknown
	};
}
