﻿// $Id: DisparityEstimator.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using Stereo.GpGpuLib;
using System.Collections.Generic;

namespace VideoLib.Stereo.GpGpu
{
	/// <summary>
	/// Implements a general disparity estimator
	/// </summary>
	class DisparityEstimator : DisparityEstimatorBase
	{
		private readonly StereoNodeFactory[] inUseFactories;

		// The nodes for each stage in the map generation process
		private DownsamplerStereoNode dsNode;
		private PreProcessorStereoNode preNode;
		private CostComputerStereoNode costNode;
		private AggregatorStereoNode aggNode;
		private MaximizerStereoNode maxNode;
		private UpsamplerStereoNode usNode;

		// The objects that hold access to the cuda data structures
		private InputImage lImage = new InputImage();
		private InputImage rImage = new InputImage();
		private CostSpace costSpace = new CostSpace();
		private CostSpace dsCostSpace = new CostSpace();
		private DepthMap dMap = new DepthMap();

		/// <summary>
		/// Creates an instance of DisparityEstimator
		/// </summary>
		public DisparityEstimator()
		{
			inUseFactories = new StereoNodeFactory[TechniqueDescription.NTechniqueTypes];
			for (int i = 0; i < TechniqueDescription.NTechniqueTypes; ++i)
				inUseFactories[i] = null;

			//activeNodes.Add(gsDn);

			// make sure we select the first valid technique for each type
			CheckAllValid();
		}

		#region Cleanup
		private bool disposed = false;
		/// <summary>
		/// Releases unmanaged resources associated with the object
		/// </summary>
		public override void Dispose()
		{
			if(!disposed)
				DisposeMembers();
			GC.SuppressFinalize(this);
		}
		~DisparityEstimator()
		{
			if(!disposed)
				DisposeMembers();
		}
		private void DisposeMembers()
		{
			lImage.Dispose();
			rImage.Dispose();
			costSpace.Dispose();
			dsCostSpace.Dispose();
			dMap.Dispose();

			// Dispose of any 'disposable' nodes
			DisposeObject(ref dsNode);
			DisposeObject(ref preNode);
			DisposeObject(ref costNode);
			DisposeObject(ref aggNode);
			DisposeObject(ref maxNode);
			DisposeObject(ref usNode);

			disposed = true;
		}
		#endregion

		protected override void GetCurrentConstraints(out InputImageType inputType, out CostSpaceType costType, out DepthMapType depthType)
		{
			inputType = InputImageType.Xrgb_uint;
			costType = (costNode != null ? costNode.GetCostSpaceType() : CostSpaceType.Any);
			depthType = DepthMapType.Single;
		}

		#region Helpers

		// Validates all factories, selecting appropriate defaults if the current one is not valid
		private void CheckAllValid()
		{
			ValidateFactories();
			CheckValid(TechniqueType.CostComputer, ref costNode);

			// The cost space type may have changed, re-validate
			ValidateFactories();

			CheckValid(TechniqueType.Maximizer, ref maxNode);
			CheckValid(TechniqueType.Downsampler, ref dsNode);
			CheckValid(TechniqueType.PreProcessor, ref preNode);
			CheckValid(TechniqueType.Aggregator, ref aggNode);
			CheckValid(TechniqueType.Upsampler, ref usNode);
		}

		// Checks that the node given is currently valid (if not null), and selects the first valid technique if not
		private void CheckValid<T>(TechniqueType type, ref T node) where T : StereoNode
		{
			StereoNodeFactory currentFactory = inUseFactories[(int)type];

			if (node != null && currentFactory != null && !currentFactory.IsValid)
			{
				// Deactivate the current (non-null) technique
				TechniqueDescription currentTechnique = techniqueDescriptions[(int)type].Find(x => x.Factory == currentFactory);
				currentTechnique.IsActive = false;
				DisposeObject<T>(ref node);
			}
			if (currentFactory == null || !currentFactory.IsValid)
			{
				for (int i = 0; i < factories[(int)type].Count; ++i)
				{
					// Select the first valid node in the list
					StereoNodeFactory factory = factories[(int)type][i];

					if (factory.IsValid)
					{
						SetNode(type, factory, ref node);
						break;
					}
				}
			}
		}

		/// <summary>
		/// Sets the current node for technique <paramref name="type"/> to the creation of <paramref name="factory"/> 
		/// - this consists of deactivating the current technique in that category, disposing of <paramref name="node"/>, 
		/// creating the new node, and activating it's technique description.
		/// </summary>
		private void SetNode<T>(TechniqueType type, StereoNodeFactory factory, ref T node) where T : StereoNode
		{
			// De-activate the currently active technique
			StereoNodeFactory current = inUseFactories[(int)type];
			TechniqueDescription activeTechnique = techniqueDescriptions[(int)type].Find(x => x.Factory == current);
			if (activeTechnique != null)
			{
				activeTechnique.IsActive = false;
			}
			DisposeObject<T>(ref node);

			node = ((StereoNodeFactory<T>)factory).Create(ExecutionStream.NullStream);
			activeNodes.Add(node);

			// Activate the new active technique
			inUseFactories[(int)type] = factory;
			activeTechnique = techniqueDescriptions[(int)type].Find(x => x.Factory == factory);
			if (activeTechnique != null)
			{
				activeTechnique.IsActive = true;
			}
		}
		#endregion

		#region Derived public interface

		/// <summary>
		/// Instantiates and applies the technique given by the technique description
		/// </summary>
		/// <param name="description">A description of the technique to use (returned from <see cref="GetTechniques"/>)</param>
		public override void SetTechnique(TechniqueDescription description)
		{
			StereoNodeFactory current = inUseFactories[(int)description.Type];
			List<TechniqueDescription> techniqueList = techniqueDescriptions[(int)description.Type];

			// Only make a change if the new index is different to the current one
			if (description.Factory != current)
			{
				if (description.Factory != null && description.Factory.IsValid)
				{
					// Create the requisite node
					switch (description.Type)
					{
						case TechniqueType.Downsampler:
							SetNode(description.Type, description.Factory, ref dsNode);
							break;
						case TechniqueType.PreProcessor:
							SetNode(description.Type, description.Factory, ref preNode);
							break;
						case TechniqueType.CostComputer:
							SetNode(description.Type, description.Factory, ref costNode);
							break;
						case TechniqueType.Aggregator:
							SetNode(description.Type, description.Factory, ref aggNode);
							break;
						case TechniqueType.Maximizer:
							SetNode(description.Type, description.Factory, ref maxNode);
							break;
						case TechniqueType.Upsampler:
							SetNode(description.Type, description.Factory, ref usNode);
							break;
					}
				}

				// If any active nodes have been made invalid, they must be reset
				CheckAllValid();
			}
		}

		/// <summary>
		/// Fills an image with a greyscale depth-disparity map computed from the two input images
		/// </summary>
		/// <param name="inImageL">The left view (laid out without padding, width-first), internally formatted as an array of uints</param>
		/// <param name="inImageR">The right view (laid out without padding, width-first), internally formatted as an array of uints</param>
		/// <param name="width">The width of both of the input images</param>
		/// <param name="height">The height of both of the input images</param>
		/// <param name="outImage">
		/// The estimated depth map (should be created beforehand, with the same dimensions as the two input images), which is internally structured
		/// as a single-precision floating point array.
		/// </param>
		public override void ProcessFrame(byte[] inImageL, byte[] inImageR, int width, int height, byte[] outImage)
		{
			if (width <= 0 || height <= 0)
				return;

			// The cost computation and wta nodes are essential
			if(costNode != null && maxNode != null)
			{
				// Resize the cuda data image, if required
				lImage.SizeImage(InputImageType.Xrgb_uint, width, height);
				rImage.SizeImage(InputImageType.Xrgb_uint, width, height);

				// Copy in input data
				lImage.CopyDataIn(inImageL, InputImageType.Xrgb_uint, width, height);
				rImage.CopyDataIn(inImageR, InputImageType.Xrgb_uint, width, height);

				//
				// The actual processing (single-pass)

				// Downsample the input images
				InputImage workingImageL = lImage, workingImageR = rImage;
				int dsFactor = 1;
				if (dsNode != null)
					dsNode.DownsampleImages(lImage, rImage, out dsFactor, out workingImageL, out workingImageR);

				if(preNode != null)
					preNode.PreProcessImages(workingImageL, workingImageR);

				int numDisparities;
				costNode.ComputeCosts(workingImageL, workingImageR, costSpace, dsFactor, out numDisparities);

				if (aggNode != null)
					aggNode.AggregateCosts(workingImageL, workingImageR, costSpace);

				maxNode.FindSolution(costSpace, dMap);

				// Upsample the output depth map
				DepthMap result = dMap;
				if (usNode != null)
					usNode.UpsampleDepthMap(dMap, numDisparities / dsFactor, lImage, rImage, out result, numDisparities);

				// Copy out output data
				result.CopyDataOut(outImage, width, height);

				// If debugging output is required next frame, provide it
				OnNextFrameDebug(inImageL, inImageR, width, height, costSpace, costSpace.GetWidth(), costSpace.GetHeight(), costSpace.GetDepth());
			}
		}
		#endregion
	}
}
