/*
 * This is a version of example program 10 that uses explicit mutexes
 * and condition variables built out of semaphores.  See the classes
 * Mutex, CondVar and CountingSemaphore for how these are constructed.
 *
 * The algorithm here is identical to example 10, except for the
 * definition within the "Buffer" class.  Note that:
 *
 *   - An instance of "Mutex" and an instance of "CondVar" are
 *     created for each instance of "Buffer",
 *
 *   - The methods "put" and "get" used to be synchronized but now
 *     they just begin with "m.acqure()" to take the mutual exclusion
 *     lock and "m.release()" to release it.
 *
 *   - The calls to "wait()" have become calls to "CVWait" and 
 *     calls to "notifyAll()" have become calls to "CVNotifyAll".
 *
 * See BufferExample2 for how these classes may be changed to use
 * separate condition variables to signal empty/full buffer states.
 *
 * NB: you might notice that this code exits after printing only 
 * 6 numbers.  This might happen because the consumer thread is
 * interrupted from the "main" method before it has taken the last
 * value.
 */

public class BufferExample {
  public static void main (String [] args) {
    Buffer b = new Buffer ();
    Thread c = new Consumer (b);
    c.start ();
    try {
      for (int i = 1; i <= 7; i++) b.put (i);
    }
    catch (InterruptedException e) {
      System.out.println ("Producer interrupted!");
    }
    c.interrupt ();
    
    try {
      c.join ();
    } catch (InterruptedException e) {
      System.out.println ("Interrupted while waiting for consumer");
    }
  }
}

class Buffer {
  int value;
  boolean valid = false;
  Mutex m = new Mutex ();
  CondVar cv = new CondVar ();

  void put (int i) throws InterruptedException {
    m.acquire ();
    while (valid) { cv.CVWait (m); }
    value = i;
    valid = true;
    cv.CVNotifyAll ();
    m.release ();
  }

  int get () throws InterruptedException {
    int result;

    m.acquire ();
    while (!valid) { cv.CVWait (m); }
    valid = false;
    cv.CVNotifyAll ();
    result = value;
    m.release ();

    return result;
  }
}

class Consumer extends Thread {

  Buffer buffer;

  Consumer (Buffer b) {buffer = b;};

  public void run () {
    try {
      while (true) { 
	System.out.println ("Found " + buffer.get ());
      }
    } catch (InterruptedException e) {
      /* Consumer will be interrupted when it should finish */
    }
  }
}

