/* (C) Cambridge University Computer Laboratory, 2002
 *     All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Systems Research
 *      Group at Cambridge University Computer Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include <pthread.h>
#include <string.h>

#include "stm.h"
#include "stm_test.h"

#ifndef EXTERNAL_STM_CONFIG
#define LENGTH  10
#define THREADS 4
#endif

#define ITERATIONS 50000

static volatile word_t objs[LENGTH];

#define TIMEVAL_TO_MILLIS(tv) ((((long long) (tv.tv_usec)) + (((long long) (tv.tv_sec)) * 1000000)) / 1000)

#define panic(_x)				\
if (1) {					\
  printf _x;					\
  printf ("\n");                                \
  exit(1);					\
}

/***********************************************************************/

static void check_objs (void) 
{
  int i;

  if ((((int)(&objs[0]) & 3) != 0) && (sizeof(word_t) == 8)) {
    panic (("objs not correctly aligned"));
  }

  for (i = 0; i < LENGTH; i++)
    {
      int j;
      word_t a = objs[i];
      for (j = 0; j < i; j ++)
	{
	  if (objs[j] == a) 
	    {
	      panic (("URK! %d duplicate at %d,%d", a, i, j));
	    }
	}
    }
}

/***********************************************************************/

static void dump_objs (void) 
{
  int i;
  for (i = 0; i < LENGTH; i++)
    {
      printf ("%c", 65 + (objs[i] % 26));
    }
}

/***********************************************************************/

#define RND_MULTIPLIER 16807LL
#define RND_MODULUS 2147483647LL

static int get_random (long long *wg)
{
  long long next;

  next = (*wg) * RND_MULTIPLIER;
  next = next % RND_MODULUS;
  (*wg) = next;

  return (int) next;
}

/***********************************************************************/

static int g_ts = 0;

static void *permute (void *dummy)
{
  STM_ENV st;
  int r;
  int i;
  int ts = 0;
  int o_g_ts;
  long long wg = (STMTestRDTick() & 0xffffff);

  STMAddThread (&st);

  for (i = 0; i < ITERATIONS; i ++)
    {
      int idx_1, idx_2;
      word_t v1, v2;

      do
	{
	  idx_1 = (get_random (&wg) % LENGTH);
	  idx_2 = (get_random (&wg) % LENGTH);
	}
      while (idx_1 < 0 || idx_2 <= idx_1);

      STMStartTransaction (&st);
      v1 = STMReadValue (&st, (addr_t) &objs[idx_1]);
      v2 = STMReadValue (&st, (addr_t) &objs[idx_2]);
      STMWriteValue (&st, (addr_t) &objs[idx_1], v2);
      STMWriteValue (&st, (addr_t) &objs[idx_2], v1);
      r = STMCommitTransaction (&st);

      if (r) ts ++;
    }

  STMRemoveThread (&st);

  do 
    {
      o_g_ts = g_ts;
    }
  while (STMTestCASInt (&g_ts, o_g_ts, o_g_ts + ts) != o_g_ts);

  return NULL;
}

/***********************************************************************/

int main (int argc, char **argv)
{
  int i;
  struct timeval ts;
  struct timeval te;
  pthread_t thrs[THREADS];

  for (i = 0; i < LENGTH ; i++)
    {
      objs[i] = i;
    }

  pthread_setconcurrency(THREADS);
  STMInit ();
  check_objs();

  gettimeofday (&ts, NULL);

  for (i = 0; i < THREADS; i ++)
    {
      pthread_create (&thrs[i], NULL, permute, NULL);
    }

  for (i = 0; i < THREADS; i ++)
    {
      pthread_join (thrs[i], NULL);
    }

  gettimeofday (&te, NULL);
  dump_objs();
  printf ("  elapsed = %lld ", TIMEVAL_TO_MILLIS (te) - TIMEVAL_TO_MILLIS (ts));
  printf ("  success = %d", (int) (((long long ) g_ts * 100) / (ITERATIONS * THREADS)));
  printf ("%% %d\n", (int) g_ts);
  check_objs();

  exit(0);
}
