(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 

(* Interpreter 5. 

   Note that the directives of interp_4.ml contain no run-time 
   objects such as values or environments (the exception is a closure). 
   In addition, we have rules in interp_4.ml that "unfold" expressions, such as 

   (E(Pair(e1, e2)) :: ds,   env :: es, vs) 
    -> ((E e1) :: C_PAIR_FST(e2) :: ds, env :: env :: es, vs)

   and 

 | (C_PAIR_FST (e) :: ds, es, vs) -> ((E e) :: C_PAIR :: ds, es, vs)

 In Interpreter 5, defined below, the idea 
 is to "compile" an expression into "code" *before* evaluating it. 
 So for example, we replace "E e" above with "compile e", where 
 compile returns a list of instructions. To capture the first rule above
 we introduce an instruction for duplicating the environment. So combining 
 the above two rules gives 

 compile(Pair(e1, e2)) =  DUP_ENV :: (compile e1) @ (compile e2) @ [IPAIR] 

 where DUP_ENV and IPAIR are instructions. 

 Some rules in interp_4.ml discard the top of the env_stack, such as 

 | (E(Var x) :: ds, env :: es, vs) -> (ds, es, (env x) :: vs)

 We capture this with 

 compile(Var x) = [LOOKUP x; POP_ENV]. 

 In addition, several "optimizations" are applied.  For example, the step 
 function of interp_4.ml might produce a sequence like 

    (ARG (e) :: ds, es,(FUN (x, body, env) ::vs) 
 -> ((E e) :: (APPLY (x, body, env) :: ds, es, vs)
 -> ... 
 -> (APPLY (x, body, env) :: ds, es, v :: vs)
 -> ((E body) :: ds, (update(env, (x, v))) :: es, vs)

 The machine defined below will simply leave the FUN construct on 
 the value stack: 

 compile(App(e1, e2)) = DUP_ENV :: (compile e1) @ (compile e2) @ [APPLY] 

 and 

 step(APPLY :: ds,  es, v :: (FUN (c, env)) :: vs) = (c @ ds, env :: es,  v :: vs)

 This choice is really forced on us by the fact that code does not 
 contain run-time values.  But how do we make a function? 

 In interp_4.ml: 

 step(E(Lambda(x, e)) :: ds, env :: es, vs) = (ds, es, FUN(x, e, env) ::vs)

 Below we have 

 compile(Lambda(x, e)) = [MK_CLOSURE((BIND x) :: (compile e)); POP_ENV]

 and 

 step(MK_CLOSURE(c) :: ds, env :: es, vs) = (ds,  env ::es, (FUN(c, env)) :: vs)
 
 step((BIND x) :: ds, env :: es, v :: vs) = (ds, (update(env, (x, v))) :: es, vs) 

*) 

open Ast 
open Common 

let verbose = ref false 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of closure 

and closure = code * env 

and instruction = 
  | PUSH of value 
  | UNARY of Past.unary_oper 
  | OPER of Past.oper 
  | TEST of code * code
  | CASE of code * code
  | IPAIR 
  | FST
  | SND
  | IINL
  | IINR
  | APPLY
  | DUP_ENV 
  | POP_ENV 
  | LOOKUP of Past.var 
  | BIND of Past.var 
  | MK_CLOSURE of code 
  | MK_REC of Past.var * code 

and code = instruction list 

and env = Past.var -> value 

type env_stack = env list 

type value_stack = value list 

type state = code * env_stack * value_stack 

type binding = Past.var * value

type bindings = binding list

(* auxiliary functions *) 

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = fun y -> if x = y then v else env y

let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | _ -> complain "malformed unary operator"

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))


(* step : state -> state. 

   Evaluation of each expression consumes the env on top of the env_stack 
   and pushes a value onto the value_stack. 

  ((compile e) :: ds, env :: es, vs) 
  -> ... 
  -> ... 
  -> ... 
  -> (ds, es, v :: vs) 

 Questions/problems for a future interp_6.ml: 
 1) How can be make the code linear and also avoid copying code from 
    the environment onto the code stack? 
 2) How can we defunctionalize environments? 
 3) How can we share environments (avoid duplication). 

*) 

let step = function 
 | ((UNARY op) :: ds, es,          v::vs) -> (ds, es, (do_unary(op, v)) :: vs) 
 | ((OPER op) :: ds,  es, v2 :: v1 :: vs) -> (ds, es, (do_oper(op, v1, v2)) :: vs)
 | (IPAIR :: ds, es,      v2 :: v1 :: vs) -> (ds, es, (PAIR(v1, v2)) :: vs)
 | (FST :: ds, es,   (PAIR (v, _)) :: vs) -> (ds, es, v :: vs)
 | (SND :: ds, es,   (PAIR (_, v)) :: vs) -> (ds, es, v :: vs)
 | (IINL :: ds, es,              v :: vs) -> (ds, es, (INL v) :: vs)
 | (IINR :: ds, es,              v :: vs) -> (ds, es, (INR v) :: vs)

 | ((PUSH v) :: ds, es,          vs) -> (ds, es, v :: vs)
 | (DUP_ENV :: ds, env :: es,    vs) -> (ds, env :: env :: es, vs)
 | (POP_ENV :: ds, env :: es,    vs) -> (ds, es, vs)
 | ((LOOKUP x) :: ds, env :: es, vs) -> (ds, env :: es, (env x) :: vs)
 | ((TEST(c1, c2)) :: ds, es,  (CON (BOOL true)) :: vs) -> (c1 @ ds, es, vs) 
 | ((TEST(c1, c2)) :: ds, es, (CON (BOOL false)) :: vs) -> (c2 @ ds, es, vs) 
 | (CASE (c1,  _) :: ds,  es,  (INL v)::vs) -> (c1 @ ds, es, v :: vs) 
 | (CASE ( _, c2) :: ds,  es,  (INR v)::vs) -> (c2 @ ds, es, v :: vs) 
 | (APPLY :: ds,  es,        v :: (FUN (c, env)) :: vs) -> (c @ ds, env :: es,  v :: vs)
 | (MK_CLOSURE(c) :: ds,  env :: es, vs) -> (ds,  env ::es, (FUN(c, env)) :: vs)
 | (MK_REC(f, c) :: ds,  env :: es, vs) -> 
       let rec new_env g = if g = f then FUN(c, new_env) else env g
       in (ds,  new_env ::es,  vs)
 | ((BIND x) :: ds, env :: es, v :: vs) -> (ds, (update(env, (x, v))) :: es, vs) 
 | state -> complain "step : bad state" 

let rec compile = function 
 | Unit           -> [PUSH (CON UNIT); POP_ENV] 
 | Integer n      -> [PUSH (CON (INT n)); POP_ENV] 
 | Boolean b      -> [PUSH (CON (BOOL b)); POP_ENV] 
 | Var x          -> [LOOKUP x; POP_ENV] 
 | Lambda(x, e)   -> [MK_CLOSURE((BIND x) :: (compile e)); POP_ENV]
 | UnaryOp(op, e) -> (compile e) @ [UNARY op]
 | Op(e1, op, e2) -> DUP_ENV :: (compile e1) @ (compile e2) @ [OPER op] 
 | If(e1, e2, e3) -> DUP_ENV :: (compile e1) @ [TEST(compile e2, compile e3)]
 | Pair(e1, e2)   -> DUP_ENV :: (compile e1) @ (compile e2) @ [IPAIR] 
 | App(e1, e2)    -> DUP_ENV :: (compile e1) @ (compile e2) @ [APPLY] 
 | Fst e          -> (compile e) @ [FST] 
 | Snd e          -> (compile e) @ [SND] 
 | Inl e          -> (compile e) @ [IINL] 
 | Inr e          -> (compile e) @ [IINR] 
 | Case(e, (x1, e1), (x2, e2)) -> 
      DUP_ENV :: (compile e)@ [CASE((BIND x1) :: (compile e1), (BIND x2):: (compile e2))] 
 | LetFun(f, (x, body), e)    -> [MK_CLOSURE((BIND x) :: (compile body)); BIND f] @ (compile e) 
 | LetRecFun(f, (x, body), e) -> [MK_REC(f, (BIND x) :: (compile body))] @ (compile e) 


let string_of_env env = "ENV" (* need to defunctionalise envs .... *) 

let rec string_of_value = function 
  | CON c -> string_of_constant c
  | PAIR(v1, v2) -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
  | INL v -> "inl(" ^ (string_of_value v) ^ ")"
  | INR  v -> "inr(" ^ (string_of_value v) ^ ")"
  | FUN (c, env) -> "FUN(" ^ (string_of_code c) ^ ", " ^ (string_of_env env) ^ ")"

and string_of_instruction = function 
 | UNARY op     -> "UNARY " ^ (Pp_past.string_of_unary_oper op) 
 | OPER op      -> "OPER " ^ (Pp_past.string_of_oper op) 
 | IPAIR        -> "IPAIR"
 | FST          -> "FST"
 | SND          -> "SND"
 | IINL         -> "IINL"
 | IINR         -> "IINR"
 | PUSH v       -> "PUSH " ^ (string_of_value v) 
 | DUP_ENV      -> "DUP_ENV"
 | POP_ENV      -> "POP_ENV"
 | LOOKUP x     -> "LOOKUP " ^ x
 | TEST(c1, c2) -> "TEST(" ^ (string_of_code c1) ^ ", " ^ (string_of_code c1) ^ ")"
 | CASE(c1, c2) -> "CASE(" ^ (string_of_code c1) ^ ", " ^ (string_of_code c1) ^ ")"
 | APPLY        -> "APPLY"
 | MK_CLOSURE c -> "MK_CLOSURE(" ^ (string_of_code c) ^ ")"
 | MK_REC(f, c) -> "MK_REC(" ^ f ^ ", " ^ (string_of_code c) ^ ")"
 | BIND x       -> "BIND " ^ x

and string_of_code c = string_of_list "; " string_of_instruction c

let print_state n (ds, es, vs) = 
    let _ = print_string ("\n\nstate " ^ (string_of_int n)) in 
    let _ = print_string "\nCode stack = \n" in 
    let _ = print_string(string_of_list "\n" string_of_instruction ds) in 
    let _ = print_string "\nEnv stack = \n" in 
    let _ = print_string(string_of_list " " string_of_env es) in 
    let _ = print_string "\nValue stack = \n" in 
    let _ = print_string(string_of_list "\n" string_of_value vs) in 
    () 

let rec driver n state = 
  let _ = if !verbose then print_state n state else () in 
  match state with 
  | ([], _, [v]) -> v 
  | state -> driver (n + 1) (step state) 

(* env_empty : env *) 
let env_empty = fun x -> complain (x ^ " is not defined!\n")

let eval(e, env) = 
    let c = compile e in 
    let _ = if !verbose 
            then print_string("Compile code =\n" ^ (string_of_list "\n" string_of_instruction c))
            else () 
    in driver 1 (c, [env], [])

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

let rec string_of_value = function 
     | CON c -> string_of_constant c
     | PAIR(v1, v2) -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN _ -> "FUNCTION( ... )" 
    

      
    
    
