(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 

(*  Interpreter 0. 

    This is a "definitional" interpreter for  for Slang.1 (the defined language) 
    using high-level constructs of Ocaml (the defining language). 
    For examples, Slang functions are represented as Ocaml functions 
    of type 
           basic_value -> constant.  
           
    Slang conditionals are translated to Ocaml conditionals, etc. 
    The most interesting (and tricky) case is the "let rec" construct of 
    Slang --- this is translated using the "lec rec" construct of Ocaml.
    Not with the defined function itself, but with the definition of 
    a recursive environment! (Because when a recursive function 
    calls itself, it must find its own definition in the environment...) 

    Note that some of the auxiliary functions can fail.  However, 
    if the input expressin has passed static analysis, then such "run time" 
    errors should never happen! (Can you prove that?) 
*) 

open Ast 
open Common 

let verbose = ref false 

(* "semenatic" domain *) 

type basic_value = 
     | SIMPLE of constant 
     | TUPLE of  constant list 

type value = 
     | BASIC of basic_value 
     | FUN of (basic_value -> constant)

type env = var -> value 

type state = env * expr 

type binding = var * value

type bindings = binding list

(* auxiliary functions *) 

(* constant_of_value :value -> costant *) 
let constant_of_value = function 
    | BASIC (SIMPLE v) -> v 
    | _ -> complain "constant_of_value : expecting SIMPLE value"


(* function_of_value : value -> (basic_value -> constant) *) 
let function_of_value = function 
    | FUN g -> g 
    | _ -> complain "function_of_value : expecting FUN value"

(* environment updates *) 

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = fun y -> if x = y then v else env y

(* mupdate : (env * bindings) -> env *) 
let rec mupdate(env, bl) = 
    match bl with 
    | [] -> env 
    | (x, v) :: rest -> mupdate(update(env, (x, v)), rest)

(* tuple_to_bindings : (formals * (constant list)) -> bindings *) 
let rec tuple_to_bindings = function 
    | ([],  []) -> [] 
    | ([], _) -> complain "parameter/arg length mismatch"
    | (_, []) -> complain "parameter/arg length mismatch"
    | (x:: fl_rest, v :: vl_rest) -> (x, BASIC(SIMPLE v)) :: tuple_to_bindings (fl_rest, vl_rest)  

(* bind_args : (env * formals * basic_value) -> env *) 
let bind_args = function 
  | (env, [x], SIMPLE v) -> update(env, (x, BASIC (SIMPLE v)))
  | (env, fl,  TUPLE vl) -> mupdate(env, tuple_to_bindings(fl, vl))
  | _ -> complain "badly formed application" 

(*
    eval : state -> constant 
         = (env * expr) -> constant
         = ((var -> value) * expr) -> constant 
*) 
let rec eval (env, e) = 
    match e with 
    | Unit             -> UNIT 
    | Var x            -> constant_of_value (env x) 
    | Integer n        -> INT n
    | Boolean b        -> BOOL b
    | If(e1, e2, e3)   -> if bool_of_constant(eval(env, e1)) then eval(env, e2) else eval(env, e3)
    | App(f, [e])      -> (function_of_value (env f)) (SIMPLE(eval(env, e)))
    | App(f, el)       -> (function_of_value (env f)) (TUPLE(eval_args(env, el)))
    | LetFun(f, fl, e1, e2) -> 
       let new_env = update(env, (f, FUN (fun v -> eval(bind_args(env, fl, v), e1))))
       in eval(new_env, e2) 
    | LetRecFun(f, fl, e1, e2) -> 
       let rec new_env g =    (* a recursive environment! *) 
         if g = f then FUN (fun v -> eval(bind_args(new_env, fl, v), e1)) else env g 
       in eval(new_env, e2) 

(*    eval_args : (env * (expr list) -> constant list *) 
and eval_args(env, el) = 
    match el with 
    | [e] -> [eval(env, e)] 
    | e :: rest -> (eval(env, e)) :: (eval_args(env, rest))
    | [] -> complain "eval_args : empty argument list!" 

(* The "built-in" primitives, each with type 

         basic_value -> constant 
*) 
let _not = function 
    | SIMPLE(BOOL b) -> BOOL(not b)
    | _ -> complain "_not : expecting a boolean!"
let _neg = function 
    | SIMPLE(INT n) -> INT(-n)
    | _ -> complain "_neg : expecting an integer!"
let _plus = function 
    | TUPLE [INT v1; INT v2] -> INT(v1 + v2)  
    | _ -> complain "_plus : expecting two integers!"
let _mult = function 
    | TUPLE [INT v1; INT v2] -> INT(v1 * v2) 
    | _ -> complain "_mult : expecting two integers!"
let _subt = function 
    | TUPLE [INT v1; INT v2] -> INT(v1 - v2) 
    | _ -> complain "_subt : expecting two integers!"
let _lt = function 
    | TUPLE [INT v1; INT v2] -> BOOL(v1 < v2) 
    | _ -> complain "_lt : expecting two integers!"
let _eqi = function 
    | TUPLE [INT v1; INT v2] -> BOOL(v1 = v2) 
    | _ -> complain "_eqi : expecting two integers!"
let _eqb = function 
    | TUPLE [BOOL b1; BOOL b2] -> BOOL(b1 = b2) 
    | _ -> complain "_eqb : expecting two booleans!"
let _and = function 
    | TUPLE [BOOL b1; BOOL b2] -> BOOL(b1 && b2) 
    | _ -> complain "_and : expecting two booleans!"
let _or = function 
    | TUPLE [BOOL b1; BOOL b2] -> BOOL(b1 || b2) 
    | _ -> complain "_or : expecting two booleans!"
let _read = function 
    | SIMPLE(UNIT) -> let _ = print_string "input> "
                in INT(readint())
    | _ -> complain "input_int : expecting UNIT"

(* env_empty : env *) 
let env_empty = fun x -> complain (x ^ " is not defined!\n")

(* env_init : env 
   includes all built-in primitives. 
*) 
let env_init = mupdate(env_empty, 
               [ 
                 ("_neg", FUN _neg); 
                 ("_not", FUN _not); 
                 ("_plus", FUN _plus); 
                 ("_mult", FUN _mult); 
                 ("_subt", FUN _subt); 
                 ("_lt", FUN _lt); 
                 ("_eqi", FUN _eqi); 
                 ("_eqb", FUN _eqb); 
                 ("_and", FUN _and); 
                 ("_or", FUN _or); 
                 ("_read", FUN _read)
             ]) 

(* interpret : expr -> simple_value *) 
let interpret t = eval(env_init, t)

    

      
    
    
