(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 

(* 
This is the main file.
*) 

(* 
   parse command line options and args 
*) 
let infile         = ref ""
let verbose        = ref false
let verbose_front  = ref false
let run_tests      = ref false
let use_i0         = ref false
let use_jc         = ref false
let lambda_lift    = ref false
let use_all        = ref false
let set_infile f   = infile := f 
let set_stack_max m = Jargon.stack_max := m 

let option_spec = [
     ("-V",    Arg.Set verbose_front, "verbose front end"); 
     ("-v",    Arg.Set verbose,       "verbose interpreter(s)"); 
     ("-i0",   Arg.Set use_i0,        "Interpreter 0 (definitional interpreter)"); 
     ("-jc",   Arg.Set use_jc,        "Jargon compiler");
     ("-ll",   Arg.Set lambda_lift,   "Lambda Lift");
     ("-all",  Arg.Set use_all,       "all interpreters"); 
     ("-t",    Arg.Set run_tests,     "run all test/*.slang with each selected interpreter, report unexpected outputs (silent otherwise)");
     ("-stackmax",  Arg.Int (set_stack_max), "set max stack size")
    ] 
let usage_msg = "Usage: slang.byte [options] [<file>]\nOptions are:"

let _ = Arg.parse option_spec set_infile usage_msg

let _ = if !use_all 
        then (use_i0     := true; 
              use_jc     := true; 
             )
        else ()

let _ = if !lambda_lift then Front_end.lambda_lift := true else () 

(* set all verbosity flags *) 
let _ = if !verbose_front then Front_end.verbose := true else () 
let _ = if !verbose 
        then (Interp_0.verbose := true; 
              Jargon.verbose := true; 
              Compile.verbose := true; 
             )
        else () 

let error file action s = print_string ("\nERROR in " ^ file ^ " with " ^ action ^ " : " ^ s ^ "\n") 

let fatal_error file action s = let _ = error file action s in exit(-1) 

let wrap file e interpret msg = 
    try Some(interpret e)
    with Common.Error s -> 
         let _ = error file msg s 
         in None 
       

(* bind interpreters *) 
(* each interpreter i_ has type "(string * Ast.expr) -> Common.constant option" *) 
let i0 (file, e)   = wrap file e Interp_0.interpret "Interpreter 0" 
let jc (file, e)   = wrap file e Compile.eval "Jargon Compiler"

let interpreters = [
    (* use-flag, the interpreter, a description string *) 
    (!use_i0,       i0,   "Interpreter 0"); 
    (!use_jc,       jc,    "Jargon compiler") 
] 

let show_output describe string_out = 
    if !run_tests
    then () 
    else let _ = if !verbose then print_string ("\n" ^ describe ^ " : \n") else ()
         in print_string ("output> " ^ string_out ^ "\n")

(* used for -t option *) 
let check_expected describe file string_out = function 
  | None -> () 
  | Some expected -> 
      if string_out = expected 
      then () 
      else print_string (
	      "\nTEST ERROR with " 
              ^ describe 
              ^ " in "  
	      ^ file 
	      ^ " : expected " 
              ^ expected 
	      ^ " but computed " 
              ^ string_out  
	      ^ "\n" )

(* runs all interpreters with input e : Ast.expr. 
   If expected_option = Some (expected), then 
   checks if the correct result was computed
   (silent if OK). 
*) 
let rec run_interpreters file e expected_option = function 
    | [] -> () 
    | (use, interp, describe) :: rest -> 
      if use 
      then (match interp(file, e) with 
           | None -> run_interpreters file e expected_option rest 
           | Some output -> 
              let string_out = Common.string_of_constant output
              in let _ = show_output describe string_out 
              in let _ = check_expected describe file string_out expected_option 
              in run_interpreters file e expected_option rest )
      else run_interpreters file e expected_option rest 
   
(* process_inputs : runs all (flagged) interpreters on all inputs *) 
let rec process_inputs = function 
       | [] -> () 
       | (file, expected) :: rest -> 
          let in_chan = try 
                           open_in file 
                        with _ -> fatal_error file "slang" ("can't open file " ^ file) 
          in let e = try 
                       Front_end.get_ast(file, in_chan)  
                     with Common.Error s -> fatal_error file "Front End" s
          in let _ = run_interpreters file e expected interpreters 
          in process_inputs rest 
 
let _ = process_inputs 
        (if !run_tests 
        then (try Tests.get_inputs () with Common.Error s -> fatal_error "tests/" "Test.get_inputs" s)
        else [(!infile, None)])
            




