open Format
open Ast 

(*
   Documentation of Format can be found here: 
   http://caml.inria.fr/resources/doc/guides/format.en.html
   http://caml.inria.fr/pub/docs/manual-ocaml/libref/Format.html
*) 


let fstring ppf s = fprintf ppf "%s" s

let rec pp_expr ppf = function 
    | Unit             -> fstring ppf "()" 
    | Var x            -> fstring ppf x 
    | Integer n        -> fstring ppf (string_of_int n)
    | Boolean b        -> fstring ppf (string_of_bool b)
    | If(e1, e2, e3)   -> fprintf ppf "@[if %a then %a else %a @]" 
                                      pp_expr e1 pp_expr e2 pp_expr e3
    | App(f, el)       -> fprintf ppf "%a(%a)" fstring f pp_expr_list el
    | LetFun(f, fl, e1, e2)     -> 
         fprintf ppf "@[let %a(%a) =@ %a @ in %a @ end@]" 
                     fstring f pp_formals fl  pp_expr e1 pp_expr e2
    | LetRecFun(f, fl, e1, e2) -> 
         fprintf ppf "@[letrec %a(%a) =@ %a @ in %a @ end@]" 
                     fstring f pp_formals fl pp_expr e1 pp_expr e2

and pp_expr_list ppf = function 
  | [] -> fstring ppf "" 
  | [e] -> pp_expr ppf e 
  | e:: rest -> fprintf ppf "%a, %a" pp_expr e pp_expr_list rest  

and pp_formals ppf = function 
  | [] -> fstring ppf "" 
  | [x] -> fprintf ppf "%a" fstring x 
  | x:: rest -> fprintf ppf "%a, %a" fstring x pp_formals rest


let string_of_formals fl = 
     let _ = pp_formals str_formatter fl
     in flush_str_formatter () 

let print_expr e = 
    let _ = pp_expr std_formatter e
    in print_flush () 

let eprint_expr e = 
    let _ = pp_expr err_formatter e
    in pp_print_flush err_formatter () 

let string_of_expr e = 
     let _ = pp_expr str_formatter e 
     in flush_str_formatter () 
