

(* 

  Here is a simple example of lambda lifting. 

  BEFORE: 

  let add(x : int, y : int) : int =
      let h(z : int) : int = (z + y)  
      in 
         h(x)  
      end  
  in 
     add(1, add(88, 12))  
  end

  AFTER : 

  let h(y, z) = z + y 
  in 
     let add(x, y) = h(y, x)  
     in 
        add(1, add(88, 12))  
     end  
  end

  This simple example shows why we want to alpha-conversion after lambda 
  lifting --- we introduced a duplicated bound variable y. 

  See below for one reason why we might want to do alpha-conversion BEFORE 
  lambda-lifting. 

  In the end, we will alpha-convert both before and after lambda-lifting. 
  We could avoid this at the cost of greatly complicating the code in this file. 

  I suspect that De Bruijn indices might simplify much of this, at the cost 
  of making it more difficult to inspect our intermediate representations... 

*) 
open Ast 

let rec lookup f = function 
  | [] ->  None 
  | (g, l) :: rest -> if f = g then Some l else lookup f rest

let rec inlist x = function 
  | [] ->  false 
  | y :: rest -> if x = y then true else inlist x rest 

(* "free_vars bound e" returns a 
    list, with no duplicates, of all free variables 
    of e that are not in the list bound. 
*) 
let free_vars bound e = 
    let rec aux bound free = function 
    | If(e1, e2, e3)           -> aux bound (aux bound (aux bound free e1) e2) e3
    | App(f, el)               -> aux_args bound free el 
    | LetFun(f, fl, e1, e2)    -> aux (fl @ bound) (aux bound free e2) e1 
    | LetRecFun(f, fl, e1, e2) -> aux (fl @ bound) (aux bound free e2) e1 
    | Var x -> if (inlist x bound) ||  (inlist x free) then free else x :: free 
    | _     -> free 
   and aux_args bound free = function 
     | [] -> free 
     | e :: rest -> aux_args bound (aux bound free e) rest
   in aux bound [] e 

(* "free_fvars bound e" returns a 
    list, with no duplicates, of all free FUNCTION variables 
    of e that are not in the list bound. 
*) 
let free_fvars bound e = 
    let rec aux bound free = function 
    | If(e1, e2, e3)           -> aux bound (aux bound (aux bound free e1) e2) e3
    | App(f, el)               -> 
        let free1 = aux_args bound free el in 
        if (inlist f bound) ||  (inlist f free) then free1 else f :: free1 
    | LetFun(f, fl, e1, e2)    -> aux (f :: bound) (aux bound free e2) e1 
    | LetRecFun(f, fl, e1, e2) -> aux (f :: bound) (aux bound free e2) e1 
    | _     -> free 
   and aux_args bound free = function 
     | [] -> free 
     | e :: rest -> aux_args bound (aux bound free e) rest
   in aux bound [] e 

type def = bool * var * (var list) * expr

(* make_lets : expr -> def list -> expr 

  A definition of the form (true, f, fl, body) means that 
  f is recursive, while (false, f, fl, body) is not recursive. 
*) 
let rec make_lets e = function 
  | [] -> e 
  | (true, f, fl, body) :: rest  -> LetRecFun(f, fl, body, make_lets e rest) 
  | (false, f, fl, body) :: rest -> LetFun(f, fl, body, make_lets e rest) 


(* check_scope_violation : var -> (def list) -> var option 

  For Slang.1 lambda lifting can fail!  This happens with 
  we lift a function g out of the body of f and the body of 
  g contains a call to f.  If only Slang had mutual recursion 
  we might be able so solve this problem !!! 
*)  
let rec check_scope_violation f = function 
  | [] -> None 
  | (_, g, _, body) :: rest -> 
       if inlist f (free_fvars [] body) then Some g else check_scope_violation f rest 

(* aux : (var * expr_list) -> expr -> (def list) * expr 

  If env : (var * expr_list), then it is a 
  list of bindings of the form (f, var_list). 
  Here var_list = [Var x1; Var x2; .... ; Var xk], which correspond 
  to the free variables in the body of f's definition (not including 
  the formal parameters of f).   Any instance of App(f, el) will be 
  replaced by App(f, val_list @ el). 

  Suppose we have this situation 

    let f(y1, y2, ..., ym) = e1 
    in 
       e2 
    end 

  and e1 has free varuables x1, x2, ... , xk (that are not 
  in the formal parameters y1, ... ym).  We want to transform this 
  to 

    let f(x1, x2, ... , xk, y1, y2, ..., ym) = e1' 
    in 
       e2' 
    end 

  where e1' and e2' have been modifed so that each 
  f(e1, ..., em) is replaced with 
  f(x1, x2, ... , xk, e1, ..., em).  Note that in e1' 
  the x's ae formal parameters, while in e2' they should be 
  free variables. 

  There are two problems: 

  1) As we descend into terms the env binding (f, var_list)
     may enter the scope of a defintion 
      
        g(parameter) = e 

    Where some parameter is in the var_list and gets captured! 
    One simple way to get around this problem is to alpha-convert 
    BEFORE lambda lifting! 

  2) When we replace f(e1, ..., em) with 
     f(x1, x2, ... , xk, e1, ..., em), we may actually introduce new 
     free variables into the definition of other function definitions 
     in e2. This is solved by first transforming the expressions without
     the binding for f, then doing a second time just for f. 
*)       
let rec aux env = function 
    | If(e1, e2, e3) -> 
      let (defs1, e1') = aux env e1 in 
      let (defs2, e2') = aux env e2 in 
      let (defs3, e3') = aux env e3 in (defs1 @ defs2 @ defs3, If(e1', e2', e3')) 
    | App(f, el)     -> 
      let (defs, el') = aux_args env el in 
      (match lookup f env with 
      | None -> (defs, App(f, el')) 
      | Some l -> (defs, App(f, l @ el')) 
      ) 
    | LetFun(f, fl, e1, e2) -> 
      let (defs1, e1') = aux env e1 in 
      let fvs = free_vars fl e1' in 
      let new_env = if fvs = [] then env else (f, List.map (fun x -> Var x) fvs) :: env in 
      let (defs2, e2') = aux new_env e2 in 
      let def = (false, f, fvs @ fl, e1') in 
       (match check_scope_violation f defs1 with 
       | None -> (defs1 @ [def] @ defs2, e2') 
       | Some g -> Common.complain ("Lambda_lift : function " 
                             ^ g ^ " cannot be lifted out of the body of " 
                             ^ f ^ " in Slang.1") 
       )
    | LetRecFun(f, fl, e1, e2) -> 
      let (defs1, e1') = aux env e1 in (* not changing App(f, ..) *)       
      let (defs2, e2') = aux env e2 in (* not changing App(f, ..) *)             
      let fvs = free_vars fl e1' in 
      let (e1'', e2'') =
                 if fvs = [] 
                 then (e1', e2') 
                 else (* now only change App(f, ...). No new defs should emerge! *) 
                      let new_env = [(f, List.map (fun x -> Var x) fvs)] in 
                      let (_, e1'') = aux  new_env e1' in 
                      let (_, e2'') = aux  new_env e2' in (e1'', e2'')  
      in let def = (true, f, fvs @ fl, e1'') in 
       (match check_scope_violation f defs1 with 
       | None -> (defs1 @ [def] @ defs2, e2'') 
       | Some g -> Common.complain ("Lambda_lift : function " 
                             ^ g ^ " cannot be lifted out of the body of " 
                             ^ f ^ " in Slang.1") 
       )
    | e          -> ([], e)

and aux_args env = function 
  | []        -> ([], []) 
  | e :: rest -> let (defs1, e') =  aux env e in 
                 let (defs2, el') = aux_args env rest in (defs1 @ defs2, e' :: el')

let lambda_lift e = let (defs, e') = aux [] e in make_lets e' defs 
