open Ast 
open Common 
open Jargon 

let verbose = ref false 

let nl inst = NotLabelled inst 

let al lab inst = Labelled(lab, inst) 

let rec find y = function 
  | [] -> complain ("Compile.find : " ^ y ^ " is not found")
  | (x, v) :: rest -> if x = y then v else find y rest 

let new_label = 
    let i = ref 0 in 
    let get () = let v = !i in (i := (!i) + 1; "L"^ (string_of_int v))
    in get 


let extend_vmap vmap d fl = 
    let rec aux k carry = function 
      | [] -> carry @ vmap 
      | x :: rest -> aux (k - 1) ((x, (d, k)) :: carry) rest 
    in aux (List.length fl) [] fl 

let rec comp d fmap vmap = function 
  | Unit                       -> ([], [nl (PUSH UNIT)]) 
  | Boolean b                  -> ([], [nl (PUSH (BOOL b))])
  | Integer n                  -> ([], [nl (PUSH (INT n))]) 
  | Var x                      -> let (depth, offset) = find x vmap in ([], [nl (ARG(d - depth, offset))])
  | If(e1, e2, e3)             -> let else_label = new_label () in 
                                  let after_else_label = new_label () in 
                                  let (defs1, c1) = comp d fmap vmap e1 in  
                                  let (defs2, c2) = comp d fmap vmap e2 in  
                                  let (defs3, c3) = comp d fmap vmap e3 in  
                                  (* simplify away the SKIPs someday at a later phase ... *) 
                                  (defs1 @ defs2 @ defs3, 
                                     (c1 
   				     @ [nl(TEST(else_label, None))] 
                                     @ c2 
				     @ [nl(GOTO (after_else_label, None)); al else_label SKIP] 
                                     @ c3 
				     @ [al after_else_label SKIP]))   
  | App(f, [])                 -> complain "comp : function application with no args!" 

  (* OK, OK, it probably was a mistake to eliminate ops from Ast.expr *) 
  | App("_neg", [e])           -> let (defs, c) = comp d fmap vmap e in (defs, c @ [nl(DoUnary NEG)])
  | App("_not", [e])           -> let (defs, c) = comp d fmap vmap e in (defs, c @ [nl(DoUnary NOT)])
  | App("_plus", [e1;e2])      -> let (defs1, c1) = comp d fmap vmap e1 in 
                                  let (defs2, c2) = comp d fmap vmap e2 in (defs1 @ defs2, c1 @ c2 @ [nl(DoOp ADD)])
  | App("_mult", [e1;e2])      -> let (defs1, c1) = comp d fmap vmap e1 in 
                                  let (defs2, c2) = comp d fmap vmap e2 in (defs1 @ defs2, c1 @ c2 @ [nl(DoOp MUL)])
  | App("_subt", [e1;e2])      -> let (defs1, c1) = comp d fmap vmap e1 in 
                                  let (defs2, c2) = comp d fmap vmap e2 in (defs1 @ defs2, c1 @ c2 @ [nl(DoOp SUB)])
  | App("_lt", [e1;e2])        -> let (defs1, c1) = comp d fmap vmap e1 in 
                                  let (defs2, c2) = comp d fmap vmap e2 in (defs1 @ defs2, c1 @ c2 @ [nl(DoOp LT)])
  | App("_eqi", [e1;e2])       -> let (defs1, c1) = comp d fmap vmap e1 in 
                                  let (defs2, c2) = comp d fmap vmap e2 in (defs1 @ defs2, c1 @ c2 @ [nl(DoOp EQI)])
  | App("_eqb", [e1;e2])       -> let (defs1, c1) = comp d fmap vmap e1 in 
                                  let (defs2, c2) = comp d fmap vmap e2 in (defs1 @ defs2, c1 @ c2 @ [nl(DoOp EQB)])
  | App("_and", [e1;e2])       -> let (defs1, c1) = comp d fmap vmap e1 in 
                                  let (defs2, c2) = comp d fmap vmap e2 in (defs1 @ defs2, c1 @ c2 @ [nl(DoOp AND)])
  | App("_or", [e1;e2])        -> let (defs1, c1) = comp d fmap vmap e1 in 
                                  let (defs2, c2) = comp d fmap vmap e2 in (defs1 @ defs2, c1 @ c2 @ [nl(DoOp OR)])
  | App("_read", [Unit])       -> ([], [nl READ])

  | App(f, [e])                -> let (defs, c) = comp d fmap vmap e in 
                                  let k = (find f fmap) + 1 in (defs, c @ [nl(CALL (f, None, d - k))]) 
                               
  | App(f, el)                 -> let (defs, c) = comp_args d fmap vmap el in 
                                  let k = (find f fmap) + 1  in (defs, c @ [nl(CALL (f, None, d - k))])

  | LetFun(f, fl, e1, e2)      -> let (defs1, c1) = comp (d + 1) fmap (extend_vmap vmap (d +1) fl) e1 in  
                                  let (defs2, c2) = comp d ((f, d) :: fmap) vmap e2 in  
                                  let defs = [al f SKIP] @ c1 @ [nl(RETURN (List.length fl))] @ defs1 @ defs2 in (defs, c2)

  | LetRecFun(f, fl, e1, e2)   -> let (defs1, c1) = comp (d + 1) ((f, d) :: fmap) (extend_vmap vmap (d + 1) fl) e1 in  
                                  let (defs2, c2) = comp d ((f, d) :: fmap) vmap e2 in  
                                  let defs = [al f SKIP] @ c1 @ [nl(RETURN (List.length fl))] @ defs1 @ defs2 in (defs, c2)
and comp_args d fmap vmap = function 
  | [e]     -> comp d fmap vmap e 
  | e :: el -> let (defs1, c1) = comp d fmap vmap e in 
               let (defs2, c2) = (comp_args d fmap vmap el)  in (defs1 @ defs2, c1 @ c2) 
  | _ -> complain "compile_args : empty arg list!"

let compile e = 
    let (d, c) = comp 0 [] [] e in 
    let result = c @ [nl HALT] @ d in 
    let _ = if !verbose 
            then print_string ("\ncompiled code = \n" ^ (string_of_listing result))
            else () 
    in result 

let eval e = run (compile e) 
