
open Past 

exception Error of string 

let complain s = raise (Error s) 


let  get_loc = function 
    | Unit loc                      -> loc 
    | What loc                      -> loc 
    | Var (loc, _)                  -> loc 
    | Integer (loc, _)              -> loc 
    | Boolean (loc, _)              -> loc 
    | UnaryOp(loc, _, _)            -> loc 
    | Op(loc, _, _, _)              -> loc 
    | If(loc, _, _, _)              -> loc 
    | App(loc, _, _)                -> loc 
    | Let(loc, _, _)                -> loc 
    | LetFun(loc, _, _, _, _, _)    -> loc 
    | LetRecFun(loc, _, _, _, _, _) -> loc 

let string_of_loc loc = 
    "line " ^ (string_of_int (loc.Lexing.pos_lnum)) ^ ", " ^ 
    "position " ^ (string_of_int ((loc.Lexing.pos_cnum - loc.Lexing.pos_bol) + 1))


let rec find loc x = function 
  | [] -> complain (x ^ " is not defined at " ^ (string_of_loc loc)) 
  | (y, v) :: rest -> if x = y then v else find loc x rest

let rec inlist x = function 
  | [] -> false 
  | y :: rest -> if x = y then true else inlist x rest

let rec check_formals loc f seen = function 
  | [] -> []  
  | (x, t) :: rest -> 
     if inlist x seen 
     then complain ("function " ^ f ^ 
                    " at " ^ (string_of_loc loc) ^ 
                    " declares formal " ^ x ^ " multiple times")
     else (x, t) :: (check_formals loc f (x :: seen) rest)

let make_if loc (e1, t1) (e2, t2) (e3, t3) = 
     match t1 with 
     | TEbool -> 
          if t2 = t3 
          then (If(loc, e1, e2, e3), t2) 
          else complain ("conditional at " 
                         ^ (string_of_loc loc) 
                         ^ " has then-branch of type " 
			 ^ (Pp_past.string_of_type t2) 
                         ^ " and else -branch of type " 
			 ^ (Pp_past.string_of_type t3) 
			)

      | ty -> complain ("condition at " 
			^ (string_of_loc loc)
			^ " is of type " 
	                ^ (Pp_past.string_of_type t1) 
			^ " --- expecting boolean!")

let rec expect loc (e, t) = function 
  | ([], []) -> (e, t) 
  | (t1:: rest1, t2 :: rest2) -> 
    if t1 = t2 
    then expect loc (e, t) (rest1, rest2) 
    else complain (" expression at " 
                   ^ (string_of_loc loc) 
                   ^ " has type "
                   ^ (Pp_past.string_of_type t2)
                   ^ " but should have type "
                   ^ (Pp_past.string_of_type t1)
		  )
  | _ -> complain "expect : internal error" 

let make_unary loc (e, t) = function 
  | NEG -> expect loc (UnaryOp(loc, NEG, e), TEint) ([TEint], [t])
  | NOT -> expect loc (UnaryOp(loc, NOT, e), TEbool) ([TEbool], [t]) 


let make_binary loc (e1, t1) (e2, t2) = function 
  | ADD  -> expect loc (Op(loc, e1, ADD, e2), TEint) ([TEint; TEint], [t1; t2])
  | MUL  -> expect loc (Op(loc, e1, MUL, e2), TEint) ([TEint; TEint], [t1; t2]) 
  | SUB  -> expect loc (Op(loc, e1, SUB, e2), TEint) ([TEint; TEint], [t1; t2])
  | LT   -> expect loc (Op(loc, e1, LT, e2), TEbool) ([TEint; TEint], [t1; t2])
  | EQI  -> expect loc (Op(loc, e1, EQI, e2), TEbool) ([TEint; TEint], [t1; t2]) 
  | EQB  -> expect loc (Op(loc, e1, EQB, e2), TEbool) ([TEbool; TEbool], [t1; t2]) 
  | AND  -> expect loc (Op(loc, e1, AND, e2), TEbool) ([TEbool; TEbool], [t1; t2])
  | OR   -> expect loc (Op(loc, e1, OR, e2), TEbool) ([TEbool; TEbool], [t1; t2]) 
  | EQ   -> (match t1, t2 with 
            | (TEbool, TEbool) -> (Op(loc, e1, EQB, e2), TEbool) 
            | (TEint, TEint)   -> (Op(loc, e1, EQI, e2), TEbool) 
            | _ -> complain ("mismatch in equaliy types at " ^ (string_of_loc loc))
            )

(* typing environment is a mapping from variables to tbinddings *) 
type tbinding = 
  | SIMPLE of type_expr 
  | FUN of formals * type_expr 

let rec check_app loc f carry = function 
  | ([], []) -> List.rev carry 
  | ([], _) -> complain ("function " 
			 ^ f ^ " at " 
			 ^ (string_of_loc loc) 
			 ^ " is applied to too many args")
  | (_, []) -> complain ("function " 
			 ^ f ^ " at " 
			 ^ (string_of_loc loc) 
			 ^ " is applied to too few args")
  | ((x, t1) :: frest, (e, t2) :: arest) -> 
    if t1 = t2 
    then check_app loc f (e :: carry) (frest, arest) 
    else complain ("in applicaton of " ^ f 
		   ^ " at " ^ (string_of_loc loc) 
		   ^ ", the argument for " 
		   ^ x ^ " has wrong type") 

let make_app loc f b l = 
    match b with 
    | FUN (formals, result) -> 
         let el = check_app loc f [] (formals, l)
         in (App(loc, f, el), result) 
    | _ -> complain (f ^ " at " ^ (string_of_loc loc) ^ " is not a function!")

let simple loc x = function 
  | SIMPLE t -> t
  | _ -> complain ("function " ^ x ^ " at " ^ (string_of_loc loc) ^ " cannot be used as a stand-alone variable")

(* 
     1) checks types 
     2) picks correct equality function depending on types of args 
     3) determines if LetFun is really a LetRecFun 

*) 
let rec make_wf env t = 
    match t with 
    | Unit _                -> (t, TEunit)
    | What _                -> (t, TEint) 
    | Var (loc, x)          -> (t, simple loc x (find loc x env))
    | Integer _             -> (t, TEint) 
    | Boolean _             -> (t, TEbool)
    | UnaryOp(loc, op, e)   -> make_unary loc (make_wf env e) op
    | Op(loc, e1, op, e2)   -> make_binary loc (make_wf env e1) (make_wf env e2) op 
    | If(loc, e1, e2, e3)   -> make_if loc (make_wf env e1) (make_wf env e2) (make_wf env e3) 
    | App(loc, f, el)       -> make_app loc f (find loc f env) (List.map (make_wf env) el)
    | Let(loc, bl, e)       -> make_let loc (make_bindings loc [] [] [] env bl ) e 
    | LetFun(loc, f, fl,t, e1, e2) -> make_letfun loc env f fl t e1 e2 
    | LetRecFun(_, _, _, _, _, _)  -> complain "Internal Error --- LetRecFun construct never passed from parser!" 

and make_let loc (env, bl) e = 
    let (e', t) = make_wf env e 
    in (Let(loc, bl, e'), t)

and make_bindings loc seen ecarry bcarry env = function 
  | [] -> (ecarry @ env, List.rev bcarry) 
  | (x, t1, e1) :: rest -> 
    let (e2, t2) = make_wf env e1 
    in if t1 = t2 
       then if inlist x seen 
            then complain ("The let expression at " ^ 
                           (string_of_loc loc) ^ " defines " 
                           ^ x ^ " multiple times")
            else make_bindings loc 
	                       (x :: seen) 
                               ((x, SIMPLE t1) :: ecarry) 
                               ((x, t1, e2) :: bcarry) 
                               env rest 
       else complain ("definition of " ^ x ^ " at " ^ 
		      (string_of_loc (get_loc e1)) ^ 
                      " does not match its declared type")

and make_letfun loc env f fl t e1 e2 =
    let b = (f, FUN(check_formals loc f [] fl, t))
    in let (e2', t2) = make_wf (b :: env) e2 
       in let new_env = (List.map (fun (x, t) -> (x, SIMPLE t)) fl) @ env
          in 
           try (
            let (e1', t1) = make_wf new_env e1
            in if t = t1 
               then (LetFun(loc, f, fl, t, e1', e2'), t2)
               else complain "if at first you don't succeed .... " 
          ) with _ -> 
            let (e1', t1) = make_wf (b :: new_env) e1
            in if t = t1 
               then (LetRecFun(loc, f, fl, t, e1', e2'), t2)
               else complain ("the return tupe of " ^ f ^ " at " ^ 
			      (string_of_loc loc) ^ 
                              "does not match its declared type")

let env_init = [] 

let check e = 
    let (e', _) = make_wf env_init e 
    in e' 

