open Format
open Past

(*
   Documentation of Format can be found here: 
   http://caml.inria.fr/resources/doc/guides/format.en.html
   http://caml.inria.fr/pub/docs/manual-ocaml/libref/Format.html
*) 

let string_of_type = function 
  | TEint -> "int" 
  | TEbool -> "bool" 
  | TEunit -> "unit" 

let string_of_unary = function 
  | NEG -> "-" 
  | NOT -> "~" 


let string_of_binary = function 
  | ADD -> "+" 
  | MUL  -> "*" 
  | SUB -> "-" 
  | LT   -> "<" 
  | EQ   -> "=" 
  | EQI   -> "=" 
  | EQB   -> "=" 
  | AND   -> "&&" 
  | OR   -> "||" 

let fstring ppf s = fprintf ppf "%s" s

let pp_type ppf t = fstring ppf (string_of_type t) 

let pp_unary ppf t = fstring ppf (string_of_unary t) 

let pp_binary ppf t = fstring ppf (string_of_binary t) 

(* ignore locations *) 
let rec pp_expr ppf = function 
    | Unit _              -> fstring ppf "()" 
    | What _              -> fstring ppf "?" 
    | Var (_, x)          -> fstring ppf x 
    | Integer (_, n)      -> fstring ppf (string_of_int n)
    | Boolean (_, b)      -> fstring ppf (string_of_bool b)
    | UnaryOp(_, op, e)   -> fprintf ppf "%a(%a)" pp_unary op pp_expr e 
    | Op(_, e1, op, e2)   -> fprintf ppf "(%a %a %a)" pp_expr e1  pp_binary op pp_expr e2 
    | If(_, e1, e2, e3)   -> fprintf ppf "@[if %a then %a else %a @]" 
                                      pp_expr e1 pp_expr e2 pp_expr e3
    | App(_, f, el)       -> fprintf ppf "%a(%a)" fstring f pp_expr_list el
    | Let(_, bl, e)       -> fprintf ppf "@[<2>let %a in %a end@]" pp_binding_list bl pp_expr e 
    | LetFun(_, f, fl,t, e1, e2)     -> 
         fprintf ppf "@[let %a(%a) : %a =@ %a @ in %a @ end@]" 
                     fstring f pp_formals fl pp_type t pp_expr e1 pp_expr e2
    | LetRecFun(_, f, fl, t, e1, e2) -> 
         fprintf ppf "@[letrec %a(%a) : %a =@ %a @ in %a @ end@]" 
                     fstring f pp_formals fl pp_type t pp_expr e1 pp_expr e2

and pp_expr_list ppf = function 
  | [] -> fstring ppf "" 
  | [e] -> pp_expr ppf e 
  | e:: rest -> fprintf ppf "%a, %a" pp_expr e pp_expr_list rest  

and pp_formals ppf = function 
  | [] -> fstring ppf "" 
  | [(x, t)] -> fprintf ppf "%a : %a" fstring x pp_type t 
  | (x, t):: rest -> fprintf ppf "%a : %a, %a" fstring x pp_type t pp_formals rest

and pp_binding_list ppf = function 
  | [] -> fstring ppf "" 
  | [(x, t, e)] -> fprintf ppf "%a : %a = %a" fstring x pp_type t pp_expr e
  | (x, t, e):: rest -> 
      fprintf ppf "%a : %a = %a @ and %a" fstring x pp_type t pp_expr e pp_binding_list rest

let string_of_formals fl = 
     let _ = pp_formals std_formatter fl
     in flush_str_formatter () 

let print_expr e = 
    let _ = pp_expr std_formatter e
    in print_flush () 

let eprint_expr e = 
    let _ = pp_expr err_formatter e
    in print_flush () 

let string_of_expr e = 
     let _ = pp_expr std_formatter e 
     in flush_str_formatter () 
