
(* for generating new function names *) 
let next_index = ref 0
let get_new_var () = 
    let n = !next_index 
    in let _ = (next_index := !next_index + 1)
    in "_" ^ (string_of_int n) 

let unary_name = function 
  | Past.NEG -> "_neg" 
  | Past.NOT -> "_not" 

let binary_name = function 
  | Past.ADD -> "_plus" 
  | Past.MUL -> "_mult" 
  | Past.SUB -> "_subt" 
  | Past.LT  -> "_lt" 
  | Past.EQ  -> Common.complain "Internal error : EQ should be resolved by Static.check"
  | Past.EQI -> "_eqi" 
  | Past.EQB -> "_eqb" 
  | Past.AND -> "_and" 
  | Past.OR  -> "_or" 

(*   translate_expr : Past.expr -> Ast.expr 
     Translates parsed AST to internal AST : 
     1) drop file locations 
     2) drop types 
     3) remove let 
     4) replace unary and binary ops with calls to "build-in" functions 
     5) replace "?" (What) with function call t _read 

  Note : our front-end drops type information.  Is this really a good idea? 
  Could types be useful in later phases of the compiler? 

*) 
let rec translate_expr = function 
    | Past.Unit _            -> Ast.Unit
    | Past.What _            -> Ast.App("_read", [Ast.Unit])
    | Past.Var(_, x)         -> Ast.Var x 
    | Past.Integer(_, n)     -> Ast.Integer n
    | Past.Boolean(_, b)     -> Ast.Boolean b
    | Past.UnaryOp(_, op, e) -> Ast.App(unary_name op, [translate_expr e])
    | Past.Op(_, e1, op, e2) -> Ast.App(binary_name op, [translate_expr e1; translate_expr e2])
    | Past.If(_, e1, e2, e3) -> Ast.If(translate_expr e1, translate_expr e2, translate_expr e3)
    | Past.App(_, f, el)     -> Ast.App(f, List.map translate_expr el)
    (*
       Replace "let" with abstraction and application. For example, translate 

        let x = e1 and y = e2 in e3 end 
        to 
        let f (x, y) = e3 in f(e1, e2) end 

        where f is a new name.  These will be of the form _n where n is an int. 
        
    *) 
    | Past.Let(_, bl, e)       -> 
       let f = get_new_var()
       in Ast.LetFun(f, 
		    List.map (fun (x, _, _) -> x) bl,
		    translate_expr e, 
		    Ast.App(f, List.map (fun (_, _, e) -> translate_expr e) bl))
    | Past.LetFun(_, f, fl,t, e1, e2)     -> 
         Ast.LetFun(f, 
		    List.map (fun (x, _) -> x) fl,
		    translate_expr e1, 
		    translate_expr e2)
    | Past.LetRecFun(_, f, fl,t, e1, e2)     -> 
         Ast.LetRecFun(f, 
		    List.map (fun (x, _) -> x) fl,
		    translate_expr e1, 
		    translate_expr e2)

