package uk.ac.cam.cl.rkh23.photo.view;

import java.awt.*;
import java.awt.event.*;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.swing.*;
import javax.swing.tree.*;

import uk.ac.cam.cl.rkh23.photo.ModelInterface;
import uk.ac.cam.cl.rkh23.photo.ViewInterface;
import uk.ac.cam.cl.rkh23.photo.model.Photo;



/**
 * This class represents the view part of the MVC design pattern
 * As such it contains all the GUI layout, which is done here
 * using Java Swing. Decoupling this completely from the controller
 * hardly ever works, so we don't attempt it in this code. There IS a 
 * separate class PhotoOrganiserController, but you will see it is
 * tightly coupled to this one. 
 * @author rkh23
 *
 */
public class PhotoOrganiserView extends JFrame implements ViewInterface {
	
	private JPanel mPhotoPanel = new JPanel();
	private JPanel mThumbPanel = new JPanel();
	
	private JButton mAddPicButton= null; // Add a picture button
	private JButton mDelPicButton= null; // Delete a picture button
	private JButton mAddAlbumButton= null; // Add an album
	private JButton mDelAlbumButton= null; // Delete an album
	
	private JList mAlbumList = null; // Selectable album list
	private DefaultListModel mAlbumListModel = new DefaultListModel();

	private JScrollPane mThumbPanelScroller = null; // A scrollable panel for thumbnails
	
	private PhotoTile mSelectedPhoto = null; // The currently selected photo
	private String mSelectedAlbumName = null; // The currently selected album
	
	private PhotoOrganiserController mController; // Controller for this view
	
	private ModelInterface mModel; // Access to the model
	
	
	
	
	/**
	 * The constructor, which sets up a frame and 
	 * adds components to it.  We couple this to an interface definition that
	 * means we are loosely coupled to the actual controller code
	 * @param listener A reference to the controller we want to use
	 */
	public PhotoOrganiserView(ModelInterface model) {
		
		// Set the title bar. This goes up the
		// inheritance tree and calls the JFrame(String) 
		// constructor
		super("Simple Photo Organiser");
		
		// This just causes the whole application to quite
		// when the window is closed.
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent we) {
				System.exit(0);
			 }	
		});

		// Store the model so we can access it
		mModel=model;
		
		// Register with the model so it tells us when it changes
		mModel.register(this);
		
		// We need to have a controller
		mController=new PhotoOrganiserController(this);
		
		// Initialise the window
		this.setSize(600,500);
		this.getContentPane().setLayout(new BorderLayout());
		
		// Create the scrolling panel of thumbs
		mThumbPanelScroller = new JScrollPane(JScrollPane.VERTICAL_SCROLLBAR_NEVER, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		mThumbPanelScroller.getViewport().add(mThumbPanel);
		
		// Create the top and bottom panels
		this.getContentPane().add(mThumbPanelScroller,BorderLayout.SOUTH);
		
		// Create a panel for the left
		JPanel leftpanel = new JPanel();
		leftpanel.setLayout(new BorderLayout());
		this.getContentPane().add(leftpanel,BorderLayout.WEST);
		
		// Setup the album list
		mAlbumList = new JList();
		mAlbumList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		mAlbumList.addListSelectionListener(mController);
		createAlbumList();		
		leftpanel.add(mAlbumList, BorderLayout.CENTER);
		
		// Create some button objects
		mAddPicButton = new JButton("Add a Photo");
		mDelPicButton = new JButton("Delete Current Photo");
		mAddAlbumButton = new JButton("Create new album");
		mDelAlbumButton = new JButton("Delete current album");
		
		// Set the buttons to send input to this class when they are pressed
		mAddPicButton.addActionListener(mController);
		mDelPicButton.addActionListener(mController);
		mAddAlbumButton.addActionListener(mController);
		mDelAlbumButton.addActionListener(mController);
		
		// Set unique strings that are passed on when a button is pressed
		// and are used to quickly identify what has happened
		mAddPicButton.setActionCommand("ADDPHOTO");
		mDelPicButton.setActionCommand("DELPHOTO");
		mAddAlbumButton.setActionCommand("ADDALBUM");
		mDelAlbumButton.setActionCommand("DELALBUM");
		
		// Put the buttons on a panel and then add the panel to the frame
		JPanel buttonpanel = new JPanel();
		buttonpanel.setLayout(new GridLayout(4,1));
		buttonpanel.add(mAddPicButton);
		buttonpanel.add(mDelPicButton);
		buttonpanel.add(mAddAlbumButton);
		buttonpanel.add(mDelAlbumButton);
		leftpanel.add(buttonpanel, BorderLayout.NORTH);
		
		// Stick the photo panel in the main bit
		mPhotoPanel.setLayout(new BorderLayout());
		getContentPane().add(mPhotoPanel, BorderLayout.CENTER);
		
		// Finally display it all
		this.setVisible(true);
	}
	
	
	protected void createAlbumList() {
		// We are going to change the display of albums
		// and we don't want the listener to get in the way
		mAlbumList.removeListSelectionListener(mController);
		
		// Get a clone of the albums
		mAlbumListModel = new DefaultListModel();
		Set<String> albums = mModel.getAlbums();
		// The Set is sorted because we used a TreeMap in the model
		int idx=0, selected=-1;
		for (String s: albums) {
			mAlbumListModel.addElement(s);
			if (s.equals(this.mSelectedAlbumName)) {
				selected=idx;
			}
			idx++;
		}
		mAlbumList.setModel(mAlbumListModel);
		if (selected>=0) mAlbumList.setSelectedIndex(selected);
		
		// Now we can add the listener back
		mAlbumList.addListSelectionListener(mController);
	}
	
	protected void displayAlbum(String album) {
		mThumbPanel.removeAll();
		validate();
		mSelectedAlbumName = album;
		if (album==null) return;
		// Retrieve all the photos
		Set<Photo> photos = mModel.getAlbumPhotos(mSelectedAlbumName);
		Iterator<Photo> pit = photos.iterator();
		while (pit.hasNext()) {
			Photo photo = pit.next();
			PhotoTile pt = new PhotoTile(photo);
			pt.addMouseListener(mController);
			mThumbPanel.add(pt);
		}
		validate();
	}
	
	/**
	 * Select a photo to display in the main area
	 * @param p
	 */
	protected void selectPhoto(Photo p) {
		mPhotoPanel.removeAll();
		PhotoTile pt = new PhotoTile(p);
		mSelectedPhoto = pt;
		mPhotoPanel.add(mSelectedPhoto, BorderLayout.CENTER);
		validate();
	}
	
	
	/**
	 * Provide access to the model to other package classes
	 */
	protected ModelInterface getModel() {
		return mModel;
	}
	
	/**
	 * Get method for the selected album name
	 */
	protected String getCurrentAlbum() {
		return mSelectedAlbumName;
	}
	
	/**
	 * Get method for the selected photo
	 */
	protected Photo getSelectedPhoto() {
		return mSelectedPhoto.getPhoto();
	}

	@Override
	public void update() {
		// This retrieves all the albums
		createAlbumList();
		// This retrieves the thumbnails have been loaded
		displayAlbum(this.mSelectedAlbumName);
	}
	
	
}
