package uk.ac.cam.cl.rkh23.bouncingball3;

import javax.swing.JOptionPane;

import uk.ac.cam.cl.rkh23.bouncingball.Vector2D;
import uk.ac.cam.cl.rkh23.bouncingball.Visualisation;

public class Simulation {
	/**
	 * Time increment on each loop
	 */
	private final double mDeltaTime = 0.01;

	/**
	 * The ball we are simulating
	 */
	private final Ball mBall = new Ball(0.8,
										0.0, 5.0, 
										0.0, 0.0, 
										0.0, -9.81);
	
	private Floor mFloor = new Floor(0.5, 30.0/(2.0*Math.PI), 0.5);
	
	private final double mMinReboundSpeed = 9.81*mDeltaTime/2.0;
	
	/**
	 * A visualisation component (not important to understand in depth)
	 */
	private final Visualisation mVisualisation = new Visualisation(1.0, 0.0, 5.0);
	
	
	
	/**
	 * The simulation entry point.  This enters a loop which continually
	 * increments the simulated 'time', updates the state of the ball, and
	 * produces some result.
	 */
	public void run() {
		while (true) {
			
			// Save the current state in case we
			// need to backtrack due to a collision
			Ball oldBall = null;
			Floor oldFloor = null;
			
			try {
				oldBall = (Ball)mBall.clone();
				oldFloor = (Floor)mFloor.clone();
			}
			catch (CloneNotSupportedException cnse) {
				System.err.println ("Can't clone() Balls or Floors - this should not pe possible?");
				System.exit(0);
			}
			
			// Update the ball
			mBall.updateState(mDeltaTime);
			mFloor.updateState(mDeltaTime);
			
			// Check for a collision
			if (mBall.getPosition().getY() < mFloor.getPosition().getY()) {
				// Ball has gone through the floor in the last step
				// so we must correct it.
				correctBall(oldBall.getPosition(), oldBall.getVelocity(), 
							mBall.getAcceleration(),
							oldFloor.getPosition(), oldFloor.getVelocity(), oldFloor.getAcceleration());
			}
			if (mBall.getPosition().getY() < mFloor.getPosition().getY()) {
				mBall.setPosition(mFloor.getPosition());
			}
			
			
			if (mBall.getPosition().getY()!=mBall.getPosition().getY()) {
				JOptionPane.showMessageDialog(mVisualisation,
					    "Bounces are now too small to track. Simulation will end",
					    "Warning",
					    JOptionPane.WARNING_MESSAGE);

				System.exit(0);
			}
			// Update the visualisation
			mVisualisation.setBallPosition(mBall.getPosition().getY());
			
			mVisualisation.setFloorHeight(mFloor.getPosition().getY());
			
			// Make the simulation wait after each iteration otherwise
			// the simulated time runs so fast that we don't see anything
			// in the visualisation!!!
			try {
				Thread.sleep(10);
			}
			catch(Exception e) {}
		}
	}
	
	/**
	 * This function is called when we detect that the ball has collided
	 * with the floor in the current iteration. It backtracks and figures out
	 * where the ball should be at the end of the time period
	 * @param p Ball position at the start of the iteration
	 * @param v Ball velocity at the start of the iteration
	 * @param a Ball acceleration at the start of the iteration
	 */
	public void correctBall(Vector2D p, Vector2D v, Vector2D a,
				Vector2D fp, Vector2D fv, Vector2D fa) {
		// we assume the velocity remains constant during each iteration
		// This allows us to work in a constant-velocity frame
		
		// First figure out when it struck the floor
		// We solve s=ut+1/2at^2  for t
		// But u is now in the floor's frame of reference
		double s = p.getY()- fp.getY();
		double u = v.getY()-fv.getY();
		double t = (-u + Math.sqrt(u*u-2.0*a.getY()*s))/a.getY();
		if (t<0.0) t = (-u - Math.sqrt(u*u-2.0*a.getY()*s))/a.getY();
		
		// How fast was it going when it hit in that FoR?
		double vy = v.getY()-fv.getY()+a.getY()*t;
		
		// What *speed* will it rebound at in that FoR?
		double vr = Math.abs(mBall.getK() * vy);
		
		// What speed will it be moving at in the world FoR?
		// We use the new frame speed just to ensure that
		// the floor can't go faster than the ball!!
		vr += mFloor.getVelocity().getY();
		
		// Where will it have gotten to in the remainder of the time step?
		// s = ut + 1/2 at^2
		double t2 = mDeltaTime-t;
		double py = vr*t2 + a.getY()*t2*t2;
		
		// How fast will it be moving?
		vy = vr+a.getY()*t2;
		
		// Check that this is a sufficient speed
		if (vy<=mMinReboundSpeed) {
			// It's going to bounce again faster that we will be checking!
			// The simulation is not correct any more
			JOptionPane.showMessageDialog(mVisualisation,
				    "Bounces are now too small to track. Simulation will end",
				    "Warning",
				    JOptionPane.WARNING_MESSAGE);

			System.exit(0);
		}
		
		// Update the Ball
		mBall.setPosition(new Vector2D(p.getX(), mFloor.getPosition().getY()+py));
		mBall.setVelocity(new Vector2D(v.getX(), vy));
	}

public static void main(String[] args) {
	Simulation sim = new Simulation();
	sim.run();
}
}
