package uk.ac.cam.cl.rkh23.bouncingball3;

import uk.ac.cam.cl.rkh23.bouncingball.Vector2D;

public abstract class MovingObject implements Cloneable {
	
	/**
	 * Current position
	 */
	protected Vector2D mPosition = null;
	/**
	 * Current velocity
	 */
	protected Vector2D mVelocity = null;
	/**
	 * Current acceleration
	 */
	protected Vector2D mAcceleration =null;
	
	/**
	 * Constructor for a moving object
	 * @param px x position
	 * @param py y position
	 * @param vx x velocity
	 * @param vy y velocity
	 * @param ax x acceleration
	 * @param ay y acceleration
	 */
	public MovingObject(double px, double py, double vx, double vy, double ax, double ay) {
		mPosition = new Vector2D(px,py);
		mVelocity = new Vector2D(vx, vy);
		mAcceleration = new Vector2D(ax, ay);
	}
	
	/**
	 * This function is provided for subclasses to override if they
	 * need to do anything more funky than a simple application of
	 * motion equations (e.g. dynamically vary the acceleration etc)
	 * @param delta The time since the last update
	 */
	public void updateState(double delta) {
		applyMotionEquations(delta);
	}
	
	/**
	 * A straight application of the motion equations for
	 * a time step of delta, using all the current states
	 * Provided to subclasses as a utility
	 * @param delta
	 */
	protected void applyMotionEquations(double delta) {
		// Update the position on the assumption that the
		// velocity was constant over the update period
		// (this is justifiable for small delta t)
		// s = ut + 1/2 at^2
		Vector2D ut = mVelocity.multiply(delta);
		Vector2D att = mAcceleration.multiply(delta*delta*0.5);
		mPosition = ut.add(att).add(mPosition);
		
		// Update the velocity
		// v = u+at
		mVelocity = mAcceleration.multiply(delta).add(mVelocity);
	}
	
	public Object clone() throws CloneNotSupportedException {
		MovingObject mo = (MovingObject)super.clone();
		mo.mPosition = new Vector2D(mPosition.getX(), mPosition.getY());
		mo.mVelocity = new Vector2D(mVelocity.getX(), mVelocity.getY());
		mo.mAcceleration = new Vector2D(mAcceleration.getX(), mAcceleration.getY());
		return mo;
	}
	
	
	
	/**
	 * Get current position
	 * @return Current position
	 */
	public Vector2D getPosition() { return mPosition; }
	/**
	 * Get current velocity
	 * @return Current velocity
	 */
	public Vector2D getVelocity() { return mVelocity; }
	/**
	 * Gett acceleration
	 * @return Current acceleration
	 */
	public Vector2D getAcceleration() { return mAcceleration; }
	
	
	/**
	 * Set the position (mutator)
	 * @param p New position
	 */
	public void setPosition(Vector2D p) { mPosition=p; }
	/**
	 * Set the velocity (mutator)
	 * @param v New velocity
	 */
	public void setVelocity(Vector2D v) { mVelocity=v; }
	/**
	 * Set the acceleration (mutator)
	 * @param a New acceleration
	 */
	public void setAcceleration(Vector2D a) { mAcceleration=a; }
	

}
