package nurbs2d.client;

import gwt.canvas.client.Canvas;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import nurbs2d.client.math.M3d;

import com.google.gwt.core.client.EntryPoint;
import com.google.gwt.i18n.client.NumberFormat;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Grid;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.MouseListener;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;

/**
 * Entry point classes define <code>onModuleLoad()</code>.
 */
public class Nurbs2d implements EntryPoint {

  Canvas splineCanvas;
  Canvas weightsCanvas;
  Canvas controlFnsCanvas;
  TextBox kInput;
  TextBox nInput;
  TextBox knotInput;
  TextBox weightInput;
  M3d currentDragTarget = null;
  
  double[] knot = { 0,1,2,3,4,5,6,7,8,9,10 };
  M3d[] ctrlPt = { new M3d(3,2,1),
                   new M3d(1,3,1),
                   new M3d(5,3,1),
                   new M3d(5,4,1),
                   new M3d(4,5,1),
                   new M3d(7,7,1), 
                   new M3d(7,2,1),
                   new M3d(4,1,1),
                 };
  int k = 2;
  int n = Math.min(knot.length-k-2, ctrlPt.length-1);
  String[] colors = { "#F00", 
                      "#0F0",
                      "#00F",
                      "#FF0",
                      "#0FF",
                      "#F0F",
                      "#C22",
                      "#2C2",
                      "#22C",
                      "#111",
  };
  
  /**
   * Main entry point
   */
  public void onModuleLoad() {
    buildCanvas();
    buildMenu();
  }
  
  ///////////////////////////////////////////////
  
  private double N(int i, int k, double t) {
    if (k==0) {
      return (t >= knot[i] && t < knot[i+1]) ? 1 : 0;
    } else {
      double left = (knot[i+k] == knot[i]) ? 0 : ((t - knot[i]) / (knot[i+k] - knot[i]));
      double right = (knot[i+k+1] == knot[i+1]) ? 0 : ((knot[i+k+1] - t) / (knot[i+k+1] - knot[i+1]));
      
      return left * N(i,k-1,t) + right * N(i+1, k-1, t); 
    }
  }

  ///////////////////////////////////////////////

  int getSplineX(M3d pt) {
    return (int)(pt.getX()*50);
  }
  
  int getSplineY(M3d pt) {
    return (int)(splineCanvas.getHeight() - pt.getY()*50);
  }
  
  double fromSplineX(int x) {
    return ((double)x) / 50;
  }
  
  double fromSplineY(int y) {
    return ((double)(splineCanvas.getHeight() - y)) / 50;
  }
  
  private void redraw() {
    boolean first = true;
    Map<Integer, List<Double>> fnWeights = new HashMap<Integer, List<Double>>();
    List<Double> totalWeights = new ArrayList<Double>();
    double[] localWeight = new double[n+1]; 
    double scale;

    // Clear all
    splineCanvas.clear();
    weightsCanvas.clear();
    controlFnsCanvas.clear();
    
    for (int i = 0; i <= n; i++) {
      fnWeights.put(i,new ArrayList<Double>());
    }
    
    // Control points
    for (int i = 0; i < ctrlPt.length; i++) {
      double radius = (ctrlPt[i] == currentDragTarget) ? 10 : 5;

      splineCanvas.beginPath();
      splineCanvas.setFillStyle(i <= n ? "#0101F0" : "040404");
      splineCanvas.arc(getSplineX(ctrlPt[i]), getSplineY(ctrlPt[i]), radius, 0, Math.PI*2, true);  
      splineCanvas.closePath();
      splineCanvas.fill();
    }
    
    // Draw spline and collect weights
    for (double t = 0; t <= knot[knot.length-1]; t+=0.1) {
      M3d pt = new M3d(0,0,0);
      double totalWeight = 0;
      double localUnalloyedWeight = 0;
      double localTotalWeight = 0;

      for (int i = 0; i <= n; i++) {
        double f = N(i,k,t);
        
        localWeight[i] = f * ctrlPt[i].getZ();
        localUnalloyedWeight += f;
        localTotalWeight += localWeight[i]; 
      }
      for (int i = 0; i <= n; i++) {
        localWeight[i] = localWeight[i] * localUnalloyedWeight / localTotalWeight; 
      }
      
      for (int i = 0; i <= n; i++) {
        totalWeight += localWeight[i];
        fnWeights.get(i).add(localWeight[i]);
        pt = pt.plus(ctrlPt[i].times(localWeight[i]));
      }
      totalWeights.add(totalWeight);

      if (totalWeight >= 0.9999 && totalWeight <= 1.00001) {
        int x = (int)(pt.getX()*50);
        int y = (int)(splineCanvas.getHeight() - pt.getY()*50);
        
        if (first) {
          splineCanvas.moveTo(x, y);
          first = false;
        } else {
          splineCanvas.lineTo(x, y);
        }
      }
    }
    splineCanvas.stroke();
    
    // Plot function contributions
    first = true;
    int i = 0;
    scale = (double)weightsCanvas.getWidth() / (double)totalWeights.size();
    for (double d : totalWeights) {
      int x = (int)(i++ * scale);
      int y = (int)(weightsCanvas.getHeight() - (weightsCanvas.getHeight()*0.8)*d);
      
      if (first) {
        weightsCanvas.moveTo(x, y);
        first = false;
      } else {
        weightsCanvas.lineTo(x,y);
      }
    }
    weightsCanvas.stroke();
    
    // Plot control functions 
    for (int j = 0; j <= n; j++) {
      List<Double> values = fnWeights.get(j);
      
      i = 0;
      first = true;
      scale = (double)controlFnsCanvas.getWidth() / (double)values.size();
      controlFnsCanvas.setStrokeStyle(colors[j]);
      controlFnsCanvas.beginPath();
      for (double d : values) {
        int x = (int)(i++ * scale);
        int y = (int)(controlFnsCanvas.getHeight() - (controlFnsCanvas.getHeight()*0.8)*d);
        
        if (first) {
          controlFnsCanvas.moveTo(x, y);
          first = false;
        } else {
          controlFnsCanvas.lineTo(x,y);
        }
      }
      controlFnsCanvas.moveTo(0,controlFnsCanvas.getHeight());
      controlFnsCanvas.closePath();
      controlFnsCanvas.stroke();
    }
  }
  
  private void buildCanvas() {
    splineCanvas = new Canvas(400,400);
    weightsCanvas = new Canvas(400,200);
    controlFnsCanvas = new Canvas(400,200);
    splineCanvas.setBackgroundColor("#eee");
    weightsCanvas.setBackgroundColor("#eee");
    controlFnsCanvas.setBackgroundColor("#eee");
    RootPanel.get("canvas").add(splineCanvas);
    RootPanel.get("weights").add(weightsCanvas);
    RootPanel.get("controlFunctions").add(controlFnsCanvas);
    
    splineCanvas.addMouseListener(new MouseListener() {
      public void onMouseDown(Widget sender, int x, int y) {
        M3d pos = new M3d(x,y,0);
        
        currentDragTarget = null;
        for (M3d target : ctrlPt) {
          M3d targetPos = new M3d(getSplineX(target), getSplineY(target), 0);
          
          if (pos.minus(targetPos).length() <= 5) {
            currentDragTarget = target;
            redraw();
          }
        }
      }

      public void onMouseEnter(Widget sender) { }

      public void onMouseLeave(Widget sender) { }

      public void onMouseMove(Widget sender, int x, int y) {
        if (currentDragTarget != null) {
          currentDragTarget.setX(fromSplineX(x));
          currentDragTarget.setY(fromSplineY(y));
          redraw();
        }
      }

      public void onMouseUp(Widget sender, int x, int y) {
        if (currentDragTarget != null) {
          currentDragTarget = null;
          redraw();
        }
      }
    });
    
    redraw();
  }
  
  private String getKnotString() {
    String s = String.valueOf(knot[0]);
    
    for (int i = 1; i < knot.length; i++) {
      s = s + ", " + String.valueOf(knot[i]);
    }
    return s;
  }
  
  private double[] parseKnotString(String s) {
    String[] arr = s.split(",");
    double[] newKnot = new double[arr.length];

    for (int i = 0; i<arr.length; i++) {
      newKnot[i] = Double.valueOf(arr[i].trim());
    }
    return newKnot;
  }
  
  private String getWeightString() {
    String s = NumberFormat.getDecimalFormat().format(ctrlPt[0].getZ());
    
    for (int i = 1; i < ctrlPt.length; i++) {
      s = s + ", " + NumberFormat.getDecimalFormat().format(ctrlPt[i].getZ());
    }
    return s;
  }
  
  private void parseWeightString(String s) {
    String[] arr = s.split(",");

    for (int i = 0; i<arr.length && i<ctrlPt.length; i++) {
      ctrlPt[i].setZ(Double.valueOf(arr[i].trim()));
    }
  }
  
  private void buildMenu() {
    VerticalPanel flow = new VerticalPanel();
    Grid g = new Grid(5,2);

    kInput = new TextBox();
    kInput.setText(String.valueOf(k));
    kInput.setVisibleLength(4);
    nInput = new TextBox();
    nInput.setText(String.valueOf(n));
    nInput.setVisibleLength(4);
    nInput.setEnabled(false);
    knotInput = new TextBox();
    knotInput.setText(getKnotString());
    knotInput.setVisibleLength(35);
    weightInput = new TextBox();
    weightInput.setText(getWeightString());
    weightInput.setVisibleLength(35);
    
    g.setText(0, 0, "k");
    g.setText(1, 0, "n");
    g.setText(2, 0, "knots");
    g.setText(3, 0, "weights");
    g.setWidget(0, 1, kInput);
    g.setWidget(1, 1, nInput);
    g.setWidget(2, 1, knotInput);
    g.setWidget(3, 1, weightInput);
    g.setWidget(4, 1, new Button("Update", new ClickListener() {
      public void onClick(Widget sender) {
        k = Integer.valueOf(kInput.getText());
        knot = parseKnotString(knotInput.getText());
        parseWeightString(weightInput.getText());
        n = Math.min(knot.length-k-2, ctrlPt.length-1);
        kInput.setText(String.valueOf(k));
        knotInput.setText(getKnotString());
        weightInput.setText(getWeightString());
        nInput.setText(String.valueOf(n));
        redraw();
      }}));
    flow.add(new HTML("<hr>"));
    flow.add(g);
    RootPanel.get("canvas").add(flow);
  }
}
