/* (C) Cambridge University Computer Laboratory, 2000
 *     All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Systems Research
 *      Group at Cambridge University Computer Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <netinet/in.h>
#include <errno.h>

#include "ecn.h"

/* require enough space for the header plus an octet for the TOS byte */
#define BUFSZ CMSG_SPACE(sizeof(char))

/* Do a recv(), and also return the TOS byte for the packet received. */

int
recvfrom_withtos (int fd, void *buf, int buflen, unsigned int flags,
		  struct sockaddr *peer, int *peerlen,
		  /* RETURNS: */
		  int *tos)
{
    int ret;
    struct iovec iov;
    struct msghdr mh;
    struct cmsghdr *cm;
    unsigned char cmsg_buf[BUFSZ];

    iov.iov_base = buf;
    iov.iov_len  = buflen;

    mh.msg_name = peer;
    mh.msg_namelen = peerlen? *peerlen : 0;
    mh.msg_iov = &iov;
    mh.msg_iovlen = 1;
    mh.msg_control = cmsg_buf;
    mh.msg_controllen = sizeof (cmsg_buf);
    mh.msg_flags = 0;

    ret = recvmsg (fd, &mh, flags);
    if (ret < 0)
	return ret;

    if (peerlen)
	*peerlen = mh.msg_namelen;

    if (mh.msg_flags & MSG_CTRUNC)
    {
	/* metadata truncated */
	errno = ENOMSG;
	return -1;
    }

    /* did we get any ancilliary data back? */
    if (mh.msg_controllen < CMSG_LEN (sizeof(char)))
    {
	/* most likely user didn't enable IP_RECVTOS option */
	errno = ENOMSG;
	return -1;
    }

    *tos = -1; /* invalid TOS */

    cm = CMSG_FIRSTHDR (&mh);
    while (cm)
    {
	if (cm->cmsg_level == IPPROTO_IP && cm->cmsg_type == IP_TOS)
	{
	    if (cm->cmsg_len >= CMSG_LEN (sizeof(char)))
		*tos = * (unsigned char*) CMSG_DATA (cm);
	    break;
	}
	cm = CMSG_NXTHDR (&mh, cm);
    }

    /* if we didn't get a TOS value, report an error */
    if (*tos == -1)
    {
	errno = ENOMSG;
	return -1;
    }

    return ret;
}



#if 0
/* now done with a #define */
int
recv_withtos (int fd, void *buf, int buflen, unsigned int flags,
	      /* RETURNS: */
	      int *tos)
{
    return recvfrom_withtos (fd, buf, buflen, flags, NULL, 0, tos);
}
#endif /* 0 */


/* End of tos-recv.c */
