/***********************************************************************
 *
 * stm-holdrel.c
 *
 * Software transactional memory built over hold-release operations
 */

#ifndef __STM_HOLDREL_H_
#define __STM_HOLDREL_H_

#include <stddef.h>

#include "portable_defns.h"
#include "holdrel.h"
#include "ptst.h"

typedef struct stm_state_t stm_state_t;

/***********************************************************************
 *
 * Start-of-day and end-of-day operations. 
 *
 * stm_init should be called once before invoking any other function here.
 * 
 * stm_init_cluster should be called once with each value of "i" from
 * 0 to NUM_CLUSTERS-1 inclusive before invoking any transaction 
 * operations.
 *
 * stm_done should be called at end-of-day.
 */

extern void stm_init (void);
extern void stm_init_cluster (int i);
extern void stm_done (void);

/***********************************************************************
 *
 * Thread managemnt operations.
 *
 * stm_add_thread returns an STM state for the calling thread.
 *
 * stm_remove_thread relinquishes the supplied STM state.
 */

extern stm_state_t *stm_add_thread (void);
extern void         stm_remove_thread (stm_state_t *stm);

extern stm_state_t *get_stm_st (ptst_t *ptst);

/***********************************************************************
 *
 * Transaction operations.
 */

extern void   stm_start (stm_state_t *st, sigjmp_buf **jbv);
extern bool_t stm_validate (stm_state_t *st);
extern bool_t stm_commit (stm_state_t *st);
extern void   stm_abort (stm_state_t *st);
extern void   stm_write_value (stm_state_t *st, addr_t addr, word_t val);
static word_t stm_read_value (stm_state_t *st, addr_t addr);

/***********************************************************************
 *
 * Optimisation operations.
 *
 * stm_remove_update removes an update to the specified address from the
 * current transaction.  It is used in the example red-black tree
 * implementation where the data structure is simplified by introducing
 * a single 'null' tree node.  Updates to the null node are not removed 
 * from any transactions in order to avoid false conflicts.
 */

extern void stm_remove_update (stm_state_t *st, addr_t addr);

/***********************************************************************/

#define LOG_NUM_OWNERSHIP_RECORDS 16
#define NUM_OWNERSHIP_RECORDS     (1 << LOG_NUM_OWNERSHIP_RECORDS)
#define OREC_CLUSTER_SIZE         256
#define OREC_TABLE_SIZE           (NUM_OWNERSHIP_RECORDS / OREC_CLUSTER_SIZE)
#define MAX_LOCATIONS             256

typedef struct {
  addr_t addr;
  word_t old_val;
  word_t new_val;
  word_t old_version;
  word_t new_version;
} stm_entry_t;

struct stm_state_t {
  // Hot fields
  volatile int          status; 
  int                   idx;
  stm_entry_t          *next_entry;
  bool_t                is_read_only;
  stm_entry_t          *orec_cache[NUM_OWNERSHIP_RECORDS];
  stm_entry_t           entries[MAX_LOCATIONS];
  per_thread_t          hrpt;

  // Cold fields
  int                   trans_started;
  int                   trans_finished;
  stm_state_t          *next_state;
  sigjmp_buf            jb;
};


#define CACHED_NEXT_ENTRY(_st) (&((_st)->next_entry))

#define OREC_TABLE_BOFFSET(ptr) (((uintptr_t) (ptr)) & ((NUM_OWNERSHIP_RECORDS<<2) - 1))
#define CACHE_BUCKET_IS_EMPTY(_o)  (((((uintptr_t) (_o)) & 1)) == 0)
#define CACHE_BUCKET_IS_ALIASED(_o)  ((_o) == ((stm_entry_t *) (((uintptr_t) &invalid_addr_entry) | 1)))
#define CACHE_BUCKET_GET_ENTRY(_o) ((stm_entry_t *)(((uintptr_t)(_o)) & (~1)))

#define SET_CACHE_BUCKET_EMPTY(_o,_i) ((_o) = (stm_entry_t *) (&(orec_table[(_i)/OREC_CLUSTER_SIZE]->t[(_i)%OREC_CLUSTER_SIZE].i)))
#define SET_CACHE_BUCKET_TO_ENTRY(_o,_e) ((_o) = (stm_entry_t *) (((uintptr_t)(_e)) |  1))
#define SET_CACHE_BUCKET_ALIASED(_o,_i) ((_o) = ((stm_entry_t *) (((uintptr_t) &invalid_addr_entry) | 1 )))

extern word_t stm_read_value1b (stm_state_t *stm, 
				addr_t addr, 
				addr_t ptr, 
				word_t seen, 
				stm_entry_t **ct);

extern word_t stm_read_value1 (stm_state_t *st, 
			       addr_t addr, 
			       stm_entry_t **where);

static word_t stm_read_value (stm_state_t *st, addr_t addr)
{
  stm_entry_t      **ct;
  stm_entry_t       *ptr;
  unsigned int       ofset;

  ofset = OREC_TABLE_BOFFSET(addr);
  ct = ((stm_entry_t **) (((char *) &(st->orec_cache[0])) + ofset));
  ptr = *ct;
  if (CACHE_BUCKET_IS_EMPTY(ptr))
    {
      int seen, val;
      stm_entry_t *entry;

      seen = (*((int volatile *) (ptr)));
      if ((seen & 1) == 0) {
	return stm_read_value1b (st,addr,(addr_t) ptr,seen,ct);
      }
      //      seen |= 1;
      entry = (*CACHED_NEXT_ENTRY(st)) ++;
      
      // NB we must ensure that discard_ownership_cache will restore
      // the cache if we seg-fault reading *addr.
      SET_CACHE_BUCKET_TO_ENTRY(*ct, entry);
      entry -> addr = addr;
      entry -> old_version = seen;
      entry -> new_version = seen;

      RMB();
      val = (word_t) *addr; 
      entry -> old_val = val; 
      entry -> new_val = val; 
      return val;
    }
  else if (CACHE_BUCKET_GET_ENTRY(ptr)->addr == addr)
    {
      return CACHE_BUCKET_GET_ENTRY(ptr) -> new_val;
    }
  else
    {
      return stm_read_value1 (st, addr, NULL);
    }
}

#define stm_start(st,jbv) do {			\
    stm_state_t *st0 = (st);		        \
    sigjmp_buf **jbv0 = (jbv);			\
    sigsetjmp (st0 -> jb, 1);			\
    stm_start (st0,jbv0);			\
  } while (0)

#endif /* __STM_H_ */
