/***********************************************************************
 *
 * stm_holdrel.c
 *
 * Software transactional memory built over hold-release operations
 */

#include "stm_holdrel.h"
#include "portable_defns.h"
#include "ptst.h"
#include "gc.h"
#include <assert.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <setjmp.h>
#include <signal.h>

/***********************************************************************/

typedef enum { STM_PASSIVE,           /* No commit in progress */
	       STM_ACQUIRING,         /* Commit started: acquiring ownership */
               STM_UPDATING,          /* Commit succeeded: updating heap */
               STM_COMMITTED,         /* Commit succeeded: releasing */
	       STM_ABORTED,           /* Commit failed: releasing */
	       STM_PASSIVE_COMMITTED, /* No commit in progress, prev committed */
	       STM_PASSIVE_ABORTED,   /* No commit in progress, prev aborted */
	       BEYOND_LAST_STATUS     /* Internal use */
} stm_status_t;

/***********************************************************************/

/*
 * Mapping from a heap pointer to an index in the array of ownership
 * records.
 */

#define OREC_HASH(ptr) ((((uintptr_t) (ptr)) >> 2) & (NUM_OWNERSHIP_RECORDS - 1))

#undef stm_start

/***********************************************************************/

#define FOR_ENTRIES(_st,_bv,_c) do {				\
  stm_entry_t *_bv;						\
  stm_entry_t *_end = (_st)->next_entry;			\
  for (_bv=(_st)->entries; _bv<_end; _bv++) {  	                \
    _c								\
  } 								\
} while (0)

#define FOR_UPDATES(_st,_bv,_c) do {				\
  stm_entry_t *_bv;						\
  stm_entry_t *_end = (_st)->next_entry;			\
  for (_bv=(_st)->entries; _bv<_end; _bv++) {  	                \
    if ((_bv)->old_version != (_bv)->new_version) {             \
      _c							\
    }                                                           \
  } 								\
} while (0)

#define FOR_READS(_st,_bv,_c) do {				\
  stm_entry_t *_bv;						\
  stm_entry_t *_end = (_st)->next_entry;			\
  for (_bv=(_st)->entries; _bv<_end; _bv++) {  	                \
    if ((_bv)->old_version == (_bv)->new_version) {             \
      _c							\
    }                                                           \
  } 								\
} while (0)

/***********************************************************************/


/* 
 * invalid_addr_entry must be an STMEntry_t whose address field is 
 * an address whose contents must never be updated through STM.  A pointer
 * to it is used to indicate an orec for which the descriptor holds
 * multiple entries.  
 *
 * Hence to if "addr" is the address we're seeking, 
 *
 *  st -> orec_cache[OREC_HASH(addr)] -> addr != addr
 *
 * can detect cache misses and aliasing in a single test.
 */



static stm_entry_t invalid_addr_entry;
static stm_state_t *stm_states[MAX_THREADS]; 

#define OREC_NOT_FOUND (-1)
#define OWNERSHIP_VALUE_FOR(st) (((st) -> idx) * 2)
#define GET_OWNER_ST(orec) ((stm_states[(orec) / 2]))

#define IS_OWNED(v)   (((v) & 1) == 0)
#define IS_VERSION(v) (((v) & 1) != 0)

typedef struct {
  int i;
} orec_t;

typedef struct {
  orec_t t[OREC_CLUSTER_SIZE];
} orec_cluster_t;


orec_cluster_t *orec_table[OREC_TABLE_SIZE];

#define OREC_OF(ptr) (&(orec_table[OREC_HASH(ptr)/OREC_CLUSTER_SIZE] ->   \
			t[OREC_HASH(ptr)%OREC_CLUSTER_SIZE].i))

static void help_held_transaction (stm_state_t *me, stm_state_t *st);
static void release_ownership (stm_state_t *me, stm_state_t *st);
static bool_t check_reads (stm_state_t *me, stm_state_t *st);

/***********************************************************************/

/* Helper functions to access a descriptor's status field.  All
 * accesses must go throguh hr_read / hr_write so that they interface
 * correctly with the software hold/release implementation which
 * displaces the contents of the location.  
 */

static stm_status_t get_status (stm_state_t *me, stm_state_t *st)
{
  stm_status_t result;
  result = hr_read (&(me -> hrpt), (addr_t) &(st -> status));
  return result;
}

/*.....................................................................*/

static void set_status (stm_state_t *me, stm_state_t *st, stm_status_t t)
{
  hr_write (&(me -> hrpt), (addr_t) &(st -> status), t);
}

/***********************************************************************/

/* Help the descriptor (if any) that holds "orec".  Note that we check
 * that "helpee" still holds the orec after we take hold of its status
 * field so that we can try to get out of its way again quickly if it
 * is still running -- we may well have been delayed waiting to hold
 * its status.
 */

static void help_owner_of (stm_state_t *me, int volatile *orec)
{
  int seen = *orec;
  if (IS_OWNED(seen)) 
    {
      stm_state_t *helpee = GET_OWNER_ST(seen);
      holding (&(me -> hrpt), (addr_t) &(helpee -> status), {
	if (*orec == seen)
	  {
	    help_held_transaction (me, helpee);
	  }
      });
    }
}

/***********************************************************************/

/* Reset all of the orec cache entries that we may have updated during
 * STM read/write operations.  A transaction is responsible for
 * clearing up the cache before commit/abort returns => start can
 * assume that the cache is empty.  
 */

static void discard_ownership_cache (stm_state_t *st)
{
  FOR_ENTRIES(st, entry, {
    int oi = OREC_HASH(entry -> addr);
    SET_CACHE_BUCKET_EMPTY (st -> orec_cache[oi], oi);
  });
}

/***********************************************************************/

/* Acquire ownership of each orec updated by the transaction.  We CAS
 * from the old version number that we read into a marked pointer to
 * the transaction descriptor.  Note that some locations may already
 * be owned by the descriptor: either through aliasing of locations to
 * orecs or through other threads having got part of the way through
 * this function.  
 *
 * For read-only locations we do not need to publicly acquire them --
 * we just need to check that the orec still holds the version number
 * that we saw.  This ensures that all of the locations held the
 * intended version numbers at the start of the call to check_reads.
 */

static void acquire_and_check (stm_state_t *me,
			       stm_state_t *st)
{
  int desired  = OWNERSHIP_VALUE_FOR (st);

  FOR_UPDATES(st, entry, {
    int *orec     = OREC_OF (entry -> addr);
    int  expected = entry -> old_version;
    int  seen     = CASIO (orec, expected, desired);
    if (seen != expected && seen != desired) {
      set_status (me, st, STM_ABORTED);
      release_ownership (me, st);
      if (IS_OWNED(seen)) {
	help_owner_of (me, orec);
      }
      return;
    }
  });

  if (check_reads (me, st)) {
    set_status (me, st, STM_UPDATING);
  }
}

/***********************************************************************/

/* Check that the orecs managing locations that "st" has read from
 * (but not updated) are acceptable.  We generally require that the
 * orec holds the version we saw when we performed the read.
 *
 * We can be more clever in two cases in order to reduce the number of
 * times that "st" gets aborted due to aliasing of separate locations
 * to the same orec:
 *
 * (1) If a location is owned then we're OK as long as the logical
 *     version number is consistent with the version that we are
 *     expecting.  We check the old version held by the owner and that
 *     the owner has aborted (=> the version they saw is still current).
 *
 * (2) If a location is not owned but happens to contain the same
 *     value as we read then we can try to revise our descriptor to
 *     have the version number seen as both the 'old' and 'new'
 *     versions.  If we can then re-read *all* of our read-only
 *     locations then they must all have held the values we expected
 *     at the start of that final pass over them.
 *
 * Note that we do nothing to prevent subsequent changes to these
 * locations: the "check_reads" operation therefore linearizes at the
 * *start* of its execution rather than at the end.  Consequently,
 * successful commit operations will linearize at the start of their
 * check_reads operation.
 */

static bool_t check_reads (stm_state_t *me,
			   stm_state_t *st)
{
 retry_all:
  FOR_READS (st, entry, {
    int volatile *orec = OREC_OF (entry -> addr);

  retry_this_read:  {
      int seen = *orec;
	
      if (IS_OWNED(seen)) {
	if (seen != OWNERSHIP_VALUE_FOR(st)) {
	  /* The orec is owned and is not owned by us: determine the
	   * logical version number and continue if it's OK, abort if
	   * it is not OK or if we are not sure */
	  stm_state_t *owner = GET_OWNER_ST(seen);
	  int c = owner -> trans_finished;
	  RMB();
	  FOR_ENTRIES(owner, oe, {
	    if (OREC_OF(oe -> addr) == orec) {
	      int owner_st = get_status (me, owner);
	      if (oe -> addr == entry -> addr ||
		  (oe -> old_version != entry -> old_version) ||
		  ((owner_st != STM_ABORTED) && (owner_st != STM_PASSIVE_ABORTED)))
		{
		  set_status (me, st, STM_ABORTED);
		  if (!st -> is_read_only)  release_ownership (me, st);
		  help_owner_of (me, orec);
		  return FALSE;
		}
	    }
	  });
	  RMB();
	  if (owner -> trans_started > c) goto retry_this_read;
	}
      } else {
	if (seen != entry -> old_version) {
	  /* The orec is not owned but it doesn't contain the version
	   * number that we were expecting.  If the location holds the
	   * value that we were expecting then try to update our
	   * descriptor and re-try the reads. */
	  int val; 
	  RMB();
	  val = *entry->addr;
	  RMB();
	  if (val != entry->old_val) { 
	    set_status (me, st, STM_ABORTED);
	    return FALSE;
	  }
	  if (*orec != seen) goto retry_this_read;
	  entry->old_version = seen;
	  entry->new_version = seen;
	  goto retry_all;
	}
      }
    }
  });

  return TRUE;
}

/***********************************************************************/

/* Make all of the updates specified by "st".
 */

static void make_updates (stm_state_t *me,
			  stm_state_t *st)
{
  FOR_UPDATES (st, entry, {
    *(entry -> addr) = entry -> new_val;
  });

  set_status (me, st, STM_COMMITTED);
}

/***********************************************************************/

/* Release ownership of any orecs acquired by "st".  Note that we can
 * use read-modify-write rather than CAS because as we are running
 * here we know that we still hold the status field of "st".  
 */

static void release_ownership (stm_state_t *me,
			       stm_state_t *st)
{
  bool_t succeeded = (get_status(me, st) == STM_COMMITTED);
  int expected = OWNERSHIP_VALUE_FOR(st);
 
  if (succeeded)
    {
      FOR_UPDATES (st, entry, {
	int *orec = OREC_OF (entry -> addr);
	if (*orec == expected) *orec = entry -> new_version;
      });
    } 
  else 
    {
      FOR_UPDATES (st, entry, {
	int *orec = OREC_OF (entry -> addr);
	if (*orec == expected) *orec = entry -> old_version;
      });
    }
  
  set_status (me, st, succeeded ? STM_PASSIVE_COMMITTED : STM_PASSIVE_ABORTED);
}
  
/***********************************************************************/

/* Start a new transaction in "st".  The previous transaction is
 * responsible for cleaning up the orec cache.
 */

void stm_start (stm_state_t *st, sigjmp_buf **jbv) 
{
  st -> is_read_only = TRUE;
  st -> next_entry = &(st -> entries[0]);
  st -> trans_started ++; 
  *jbv = &(st -> jb);
}

/***********************************************************************/

/* "stm_validate" tests whether the current transaction may commit --
 *  we assume that it is run only sporadically in order to prevent
 *  looping and so we make less effort here than in (say) check_reads.
 *  We assume that "st" is doomed if any locations involved has been
 *  updated or is owned (in either case *orec!=entry->old_version. 
 */

bool_t stm_validate (stm_state_t *st)
{
  bool_t result = ((st -> status) == STM_PASSIVE);
  FOR_ENTRIES(st, entry, {
    int *orec = OREC_OF (entry -> addr);
    if (*orec != entry -> old_version) {
      result = FALSE;
      break;
    }
  });

  return result;
}

/***********************************************************************/

/* Helper function used in commit and to ensure progress when
 * encountering an owned orec.  The caller guarantees that they hold
 * the status field of "st".  This function must be prepared to pick
 * up the transaction at any non-passive state.  (The passive states
 * are used to signal the result to the transaction's initiating
 * thread: we do not clear them up, but if we see them then we know
 * that the transaction no longer owns any orecs and so cannot be in
 * the way any more).
 */

static void help_held_transaction (stm_state_t *me, stm_state_t *st)
{
  if (get_status (me, st) == STM_ACQUIRING) {
    acquire_and_check (me, st);
  }

  if (get_status (me, st) == STM_UPDATING) {
    make_updates(me, st);
  }
  
  if (get_status (me,st) == STM_COMMITTED ||
      get_status (me,st) == STM_ABORTED) {
    release_ownership(me, st);
  }
}

/***********************************************************************/

/* Attempt to commit the current transaction.  We distinguish three cases:
 *
 *   (i) The transaction was doomed to fail because it performed
 *       multiple reads from the same location and saw different version
 *       numbers in them.
 *
 *  (ii) The transaction was entirely read-only -- we just need to
 *       check the version numbers we saw remain valid and so elide
 *       the make_updates and release_ownership loops.
 *
 * (iii) The transaction was a 'normal' read-write transaction: hold
 *       its status field and call help_held_transaction.
 */

bool_t stm_commit (stm_state_t *st)
{
  bool_t result;

  discard_ownership_cache (st);

  st->trans_finished ++; 

  if (get_status (st, st) == STM_PASSIVE_ABORTED)
    {
      /* Case (i) : doomed to fail */
      result = FALSE;
    }
  else if (st -> is_read_only)
    {
      /* Case (ii) : read-only */
      result = check_reads(st, st);
    }
  else
    {
      /* Case (iii) : read-write.  Note: done => we got through
       * help_held_transaction without losing hold of our status
       * field.  If another transaction takes hold of it then we exit
       * the "holding" construct and will re-try here.  Alternatively,
       * we may drop the status field ourselves if we go to help
       *  someone. */
      bool_t done = FALSE;
      set_status (st, st, STM_ACQUIRING); 
      do {
	holding (&(st -> hrpt), (addr_t)&(st -> status), {
	  help_held_transaction (st, st);
	  done = TRUE;
	});
      } while (!done);
      result = (get_status (st, st) == STM_PASSIVE_COMMITTED);
    }

  set_status (st, st, STM_PASSIVE);
  
  return result;
}

/***********************************************************************/

/* Abort the current transaction */

void stm_abort (stm_state_t *st)
{
  discard_ownership_cache (st);
  st->trans_finished ++; 
  set_status (st, st, STM_PASSIVE_ABORTED);
}

/***********************************************************************/

/* Optimisation function to attempt to remove any updates to "addr".
 * We make some effort to do this since it is important in the
 * red-black tree case to avoid contention on the dummy nodes.
 *
 * We remove updates in two cases:
 *
 *  (i) When "addr" is the only address that we have associated with
 *      its orec.
 *
 * (ii) When all of the other locations we have associated with the
 *      orec do not appear to have made updates to the values held at
 *      them.
 */

void stm_remove_update (stm_state_t *st, addr_t addr)
{
  stm_entry_t *entry;

  entry = st -> orec_cache[OREC_HASH(addr)];
  if (!CACHE_BUCKET_IS_EMPTY(entry))
    {
      if (CACHE_BUCKET_GET_ENTRY(entry) -> addr == addr) 
	{
	  entry = CACHE_BUCKET_GET_ENTRY(entry);
	  entry -> new_val = *addr;
	  entry -> new_version = entry -> old_version;
	}
      else if (CACHE_BUCKET_IS_ALIASED(entry))
	{
	  bool_t safe = TRUE;
	  int *orec = OREC_OF(addr);
	  FOR_ENTRIES(st, entry, {
	    if ((OREC_OF(entry -> addr) == orec) &&
		(entry -> addr != addr) &&
		(entry -> new_val != entry -> old_val)) {
	      safe = FALSE;
	    }
	  });
	  if (safe) {
	    FOR_ENTRIES(st, entry, {
	      if (OREC_OF(entry->addr) == orec) {
		entry->new_val = entry -> old_val;
		entry->new_version = entry->old_version;
	      }
	    });
	  }
	}
    }
}

/*......................................................................*/

// Read slow-path code: we've not read from the location but saw "seen" at
// its orec indicating that another thread owns it.

word_t stm_read_value1b (stm_state_t *st, 
			 addr_t addr, 
			 addr_t ptr, 
			 word_t seen, 
			 stm_entry_t **ct)
{
  stm_state_t *owner;
  stm_entry_t *entry;
  int val;
  int ver;
  stm_entry_t *oe;
  int c;
  
  do
    {
      val = *addr;
      ver = (seen | 1);
      owner = GET_OWNER_ST(seen);
      c = owner -> trans_finished;
      FOR_ENTRIES(owner, oe, {
	if (OREC_OF(oe->addr) == OREC_OF(addr) &&
	    addr != oe->addr) {
	  ver = oe -> old_version;
	}
      });

      if ((*ptr == seen) && (owner -> trans_started <= c)) {
	break;
      }

      seen = *ptr;
      if ((seen & 1) == 1) {
	ver = seen;
	RMB();
	val = *addr;
	break;
      }
    }
  while (1);

  entry = (st -> next_entry) ++;
  SET_CACHE_BUCKET_TO_ENTRY(*ct, entry);
  entry -> addr = addr;
  entry -> old_version = ver;
  entry -> new_version = ver;
  val = *addr;
  entry -> old_val = val;
  entry -> new_val = val;

  return val;

}

// Read slow-path code: entry in our orec cache is already in use for
// another address.

word_t stm_read_value1 (stm_state_t *st, 
			addr_t addr, 
			stm_entry_t **where)
{
  word_t           result;
  int volatile *orec;
  int seen_before;
  int seen_after;
  int              version;
  int              wanted;
  int              wanted_new;
  int              i;
  stm_entry_t *cached;
  stm_entry_t *entry;

  wanted = -1;
  wanted_new = -1;
  cached  = st -> orec_cache[OREC_HASH(addr)];
     
  if (!CACHE_BUCKET_IS_EMPTY(cached))
    {
      if (CACHE_BUCKET_IS_ALIASED(cached))
	{
	  /* May be in our descriptor */
	  FOR_ENTRIES (st, oe, {
	    if (OREC_OF(oe -> addr) == OREC_OF(addr))
	      {
		if (oe -> addr == addr)
		  {
		    result = oe -> new_val;
		    if (where != NULL) *where = oe;
		    goto done_read;
		  }
		wanted = oe -> old_version;
		wanted_new = oe -> new_version;
	      }
	  });
	}
      else 
	{
	  wanted = CACHE_BUCKET_GET_ENTRY(cached) -> old_version;
	  wanted_new = CACHE_BUCKET_GET_ENTRY(cached) -> new_version;
	}
    }
  
  /* Not yet read it: fetch it from memory */
  orec = OREC_OF(addr);
  do
    {
    retry_read_location:
      version = -1;
      seen_before = *orec;
      RMB();
      result = (word_t) *addr; 
      RMB();
      
      if (IS_VERSION(seen_before)) {
	/* Ordinary read of version vi_before */
	version = seen_before;
      } else {
	/* Read from an owned location */
	stm_state_t *owner = GET_OWNER_ST (seen_before);
	int c = owner -> trans_finished;
	bool_t done;

	done = FALSE;
	FOR_ENTRIES (owner, oe, {
	  if (OREC_OF(oe->addr) == OREC_OF(addr)) 
	    {
	      if (addr != oe->addr) {
		result = oe -> old_val;
		version = oe -> old_version;
	      } else {
		version = seen_before | 1;
	      }
	    }
	});

	if ((owner -> trans_started > c) || (version == -1)) 
	  goto retry_read_location;
      }

      RMB();
      seen_after = *orec;
    }
  while (seen_before != seen_after);
  
  if (wanted != -1 && version != wanted) 
    {
      /* 
       * We are now doomed to fail: we would have two entries
       * with conflicting old version numbers.  
       */
      
      set_status (st, st, STM_PASSIVE_ABORTED);
    }
  else
    {
      /* Record value and version in descriptor */
      entry = (st -> next_entry) ++;
      entry -> addr = addr;
      entry -> old_val = result;
      entry -> new_val = result;
      entry -> old_version = version;
      entry -> new_version = (wanted_new == -1) ? version : wanted_new;
      if (where != NULL) *where = entry;
      
      if (CACHE_BUCKET_IS_EMPTY(cached)) {
	/* Cache was not in use: fill it */
	SET_CACHE_BUCKET_TO_ENTRY (st -> orec_cache[OREC_HASH(addr)], entry);
      } else {
	/* Cache was in use: mark it aliased */
	SET_CACHE_BUCKET_ALIASED (st -> orec_cache[OREC_HASH(addr)], OREC_HASH(addr));
      }
    }

 done_read:

  return result;
}

/**********************************************************************/

// Write slow-path code: our orec cache entry is aliased or missing

static void stm_write_value0 (stm_state_t *st, addr_t addr, word_t val)
{
  stm_entry_t *where = NULL;
  int old_version;
  int new_version;
  int i;

  stm_read_value1 (st, addr, &where);
  if (where != NULL)
    {  
      /* Update it in the descriptor */
      old_version = where -> old_version;
      new_version = old_version + 2;
      where -> new_val = val;
      where -> new_version = new_version;
      
      for (i = 0; i < st -> next_entry - st -> entries; i++)
	{
	  if (OREC_OF(st -> entries[i].addr) == OREC_OF(addr))
	    {
	      st -> entries[i].new_version = new_version;
	    }
	}
      
    }

  return;
}

void stm_write_value (stm_state_t *st, addr_t addr, word_t val)
{
  stm_entry_t *ct;

  st -> is_read_only = FALSE;
  ct = st -> orec_cache[OREC_HASH(addr)];
  if (!CACHE_BUCKET_IS_EMPTY(ct) && CACHE_BUCKET_GET_ENTRY(ct) -> addr == addr)
    {
      CACHE_BUCKET_GET_ENTRY(ct) -> new_val = val;
      CACHE_BUCKET_GET_ENTRY(ct) -> new_version = CACHE_BUCKET_GET_ENTRY(ct) -> old_version + 2;
    }
  else
    {
      stm_write_value0 (st, addr, val);
    }

  return;
}

/***********************************************************************/

/* Simple spin-lock to protect tctr, free_state, first_state*/

static volatile int state_management_lock = 0;

/* Used to assign unique idx values to stm_state_t structs */
static int tctr = 1;

/* Head of a list of free state structures  */
static stm_state_t *free_state = NULL;

/* Head of a list of active state structures */
static stm_state_t *first_state = NULL;

static void lock_state_management (void) {
  while (CASIO (&state_management_lock, 0, 1) != 0) {
    /* Nothing */
  }
}

static void unlock_state_management (void) {
  state_management_lock = 0;
}

stm_state_t *stm_add_thread (void) {
  stm_state_t *result;
  int          i;

  /* Obtain state structure */
  result = NULL;
  lock_state_management ();
  if (free_state != NULL) {
    /* Re-use free state */
    result = free_state;
    free_state = result -> next_state;

    result -> next_state = first_state;
    first_state = result;

  } else {
    /* Allocate new structure and idx value */
    result = ALIGNED_ALLOC(sizeof(stm_state_t));
    result -> idx = tctr;
    result -> trans_started = 0;
    result -> trans_finished = 0;

    stm_states[tctr] = result;
    hr_init (&(result->hrpt), result->idx);
    tctr ++;

    /* Initialize structure */
    result -> next_state = first_state;
    result -> next_entry = result -> entries;
    set_status (result, result, STM_PASSIVE);
    first_state = result;

    for (i = 0; i < NUM_OWNERSHIP_RECORDS; i ++)
      {
	SET_CACHE_BUCKET_EMPTY(result -> orec_cache[i], i);
      }
  }

  unlock_state_management ();

  return result;
}

/*......................................................................*/

void stm_remove_thread (stm_state_t *st) {
  stm_state_t **tmp;

  lock_state_management ();

  /* Remove from list of active structures */
  tmp = &first_state;
  while (*tmp != st) {
    tmp = &((*tmp) -> next_state);
  }
  *tmp = st -> next_state;

  /* Add to list of free structures */
  st -> next_state = free_state;
  free_state = st;
  unlock_state_management();
}


stm_state_t *stms[MAX_THREADS];

stm_state_t *get_stm_st (ptst_t *ptst)
{
  stm_state_t *r = stms[ptst -> id];
  if (r == NULL) {
    r = stm_add_thread ();
    stms[ptst->id] = r;
  }

  return r;
}

/*......................................................................*/

static void handle_fault(int sig)
{
  ptst_t *ptst = critical_enter();
  stm_state_t *stm = get_stm_st(ptst);
  if (stm != NULL && !stm_validate(stm)) {
    stm_abort(stm);
    critical_exit(ptst);
    siglongjmp(stm -> jb, 0);
  } else {
    fprintf(stderr, "Could not handle %d", sig);
    abort();
  }
}

void stm_init (void) 
{
  struct sigaction act;

  act.sa_handler = handle_fault;
  sigemptyset (&act.sa_mask);
  act.sa_flags = 0;
  sigaction (SIGSEGV, &act, NULL);
  sigaction (SIGBUS, &act, NULL);

  invalid_addr_entry.addr = (addr_t) &(invalid_addr_entry.addr);
}

void stm_init_cluster (int c) 
{
  orec_cluster_t *cluster = ALIGNED_ALLOC(sizeof (orec_cluster_t));

  orec_table[c] = cluster;
  for (int i = 0; i < OREC_CLUSTER_SIZE; i ++) {
    cluster->t[i].i = (MAX_THREADS * 2) + 1;
  }
}

void stm_done (void)
{
  for (int i = 0; i < OREC_TABLE_SIZE; i ++) {
    free (orec_table[i]);
    orec_table[i] = NULL;
  }
}


