/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.server;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import t3.remotehrd.protocol.BurstTerminator;
import t3.remotehrd.protocol.OpCursorPos;
import t3.remotehrd.protocol.OpCursorShape;
import t3.remotehrd.protocol.OpDestroyCursor;
import t3.remotehrd.protocol.OpDestroyLink;
import t3.remotehrd.protocol.OpDestroyTile;
import t3.remotehrd.protocol.OpSetLinkAff;
import t3.remotehrd.protocol.OpSetTileAff;
import t3.remotehrd.protocol.OpUpdateTileContents;


/*
 * Methods here can be called by any thread
 * We synchronize on this to protect the queue, output stream and serverPush 
 */

class NetworkSender {
    
    // don't use bandwidthmeasuring here as it really slows things down!
    // don't know why!
    
	private OutputStream os;
	private BufferedOutputStream bos;
	private ObjectOutputStream obos;
	// private NonBlockingObjectSender nbos;
	
	private List<Serializable> clientPullToSendOther = new LinkedList<Serializable>();	
	private Map<Integer, OpSetTileAff> clientPullToSendOpSetTileAffByElId = new HashMap<Integer, OpSetTileAff>();
	private Map<Integer, OpSetLinkAff> clientPullToSendOpSetLinkAffByElId = new HashMap<Integer, OpSetLinkAff>();
	private Map<Integer, List<OpUpdateTileContents>> clientPullToSendOpUpdateTileContentsByTileId = new HashMap<Integer, List<OpUpdateTileContents>>();
	boolean clientPullMessagesWaitingToBeSent;
	private Map<Integer, OpCursorPos> clientPullToSendOpCursorPosByCursorId = new HashMap<Integer, OpCursorPos>();
    private Map<Integer, OpCursorShape> clientPullToSendOpCursorShapeByCursorId = new HashMap<Integer, OpCursorShape>();
    
    private int numMessagesInCurrentUpdateFrame = 0;
    private List<Serializable> clientPullMessagesInCurrentUpdateFrame = new LinkedList<Serializable>();
    
	boolean serverPush = true;
	private Client client;
	boolean alwaysServerPush;
    private int sendCount=0;
    

	private static final Logger logger = Logger.getLogger("t3.remotehrd.server");
	
	NetworkSender(OutputStream os, Client client, boolean alwaysServerPush) {
		try {
			this.os = os;
            this.bos = new BufferedOutputStream(os);
			this.obos = new ObjectOutputStream(bos);
			this.client = client;
			this.alwaysServerPush =alwaysServerPush;
			this.clientPullMessagesWaitingToBeSent = false;
            /*this.nbos = new NonBlockingObjectSender(
                false,
                this.obos,
                new NonBlockingObjectSender.OnThrowable() {
                    public void hasThrown(Throwable t) {
                        NetworkSender.this.client.errorWithThisClient(t);
                    }
                }            
            );*/
		} catch(Throwable e) {
			this.client.errorWithThisClient(e);
		}
	}
	
	
	synchronized void addToSendQueue(Serializable o) {
		/* if server push:
		 * 		we don't amalgamate messages because we assume there's not much
		 * 		overlap in a burst so we don't bother 
         *      we could put it on the current update frame queue
         *      but why not just send it straight away
         *      so we send it straight away.
		 * if not server push
		 * 		we queue it because client has to request
		 * 		and we also amalgamate it with other requests to save bandwidth
		 * 
		 */
		
		assert !(o instanceof BurstTerminator);
        if(this.serverPush) {
            this.actuallyCompressSendAndFlushIfBT(o);
        } else {
            clientPullMessagesInCurrentUpdateFrame.add(o);
        }
        this.numMessagesInCurrentUpdateFrame++;
    }
	
	synchronized void endOfUpdateFrame() {
		if(this.serverPush) { // note this.serverPush can be true even if alwaysServerPush is false.
			if(this.numMessagesInCurrentUpdateFrame>0) {
				// something has been sent in this update frame so send a burst terminator and flush
                actuallyCompressSendAndFlushIfBT(new BurstTerminator() );            
				this.serverPush = alwaysServerPush ? true : false;
                //System.out.println("Had already sent in this frame "+this.numMessagesInCurrentUpdateFrame);
                //System.out.println("Server push was true but sent burst so now "+this.serverPush);
			} else {
				// there was nothing in this frame anyway.
			}
		} else {
            // put messages from messagesInCurrentUpdateFrame into messagesWaitingToBeSent
            // and amalgamate.
            for(Serializable o: this.clientPullMessagesInCurrentUpdateFrame) {
                this.clientPullMessagesWaitingToBeSent = true;
                if(o instanceof OpUpdateTileContents) {
                    OpUpdateTileContents oo = (OpUpdateTileContents) o; 
                    if(oo.r.height*oo.r.width>50*50) {
                        // remove any that it subsumes
                        List<OpUpdateTileContents> opUpdateTileContentss = this.clientPullToSendOpUpdateTileContentsByTileId.get(oo.elId);
                        if(opUpdateTileContentss==null) {
                            opUpdateTileContentss = new LinkedList<OpUpdateTileContents>();
                            this.clientPullToSendOpUpdateTileContentsByTileId.put(oo.elId, opUpdateTileContentss);
                        } else {
                            ListIterator<OpUpdateTileContents> i = opUpdateTileContentss.listIterator();
                            while(i.hasNext()) {
                                if(oo.r.contains(i.next().r)) {
                                    i.remove();
                                    logger.fine("Removed subsumed tile content update.");
                                }
                            }
                        }
                        opUpdateTileContentss.add(oo);
                    } else {
                        // just stick it in the other list
                        this.clientPullToSendOther.add(o);
                    }
                } else if(o instanceof OpSetLinkAff) {
                    OpSetLinkAff oo = (OpSetLinkAff) o; 
                    this.clientPullToSendOpSetLinkAffByElId.put(oo.elId, oo);
                } else if(o instanceof OpDestroyLink) {
                    OpDestroyLink oo = (OpDestroyLink) o;
                    this.clientPullToSendOpSetLinkAffByElId.remove(oo.elId);
                    this.clientPullToSendOther.add(o);
                } else if(o instanceof OpSetTileAff) {
                    OpSetTileAff oo = (OpSetTileAff) o; 
                    this.clientPullToSendOpSetTileAffByElId.put(oo.elId, oo);
                } else if(o instanceof OpDestroyTile) {
                    OpDestroyTile oo = (OpDestroyTile) o;
                    this.clientPullToSendOpUpdateTileContentsByTileId.remove(oo.elId);
                    this.clientPullToSendOpSetTileAffByElId.remove(oo.elId);
                    this.clientPullToSendOther.add(o);
                } else if (o instanceof OpCursorPos) {
                    OpCursorPos oo = (OpCursorPos) o;
                    this.clientPullToSendOpCursorPosByCursorId.put(oo.cursorId, oo);
                } else if (o instanceof OpCursorShape) {
                    OpCursorShape oo = (OpCursorShape) o;
                    this.clientPullToSendOpCursorShapeByCursorId.put(oo.cursorId, oo);
                } else if (o instanceof OpDestroyCursor) {
                    OpDestroyCursor oo = (OpDestroyCursor) o;
                    // remove if present.
                    this.clientPullToSendOpCursorPosByCursorId.remove(oo.cursorId);
                    this.clientPullToSendOpCursorShapeByCursorId.remove(oo.cursorId);
                    this.clientPullToSendOther.add(o);
                } else {
                    this.clientPullToSendOther.add(o);
                }
            }            
            this.clientPullMessagesInCurrentUpdateFrame.clear();
			// we're in client pull mode so don't do anything. wait until client sends ready message.
		}
        this.numMessagesInCurrentUpdateFrame = 0;
	}
	
	
    
    
    
    private synchronized void actuallyCompressSendAndFlushIfBT(Serializable msg) {
        if(msg instanceof OpUpdateTileContents) {
            ((OpUpdateTileContents)msg).compressIfRequired();
        } else {
            // no need to compress
        }
        //this.nbos.sendAndFlushIfBTNonBlocking(msg);
        try {
            if(msg instanceof BurstTerminator) {
                // have to reset BEFORE burst terminator sent
                // otherwise client hangs trying to read object
                this.obos.reset();
            } else {
                // do nothing
            }
            //System.out.println("Writing "+msg);
            this.obos.writeObject(msg);
            if(msg instanceof BurstTerminator) {
                obos.flush();
            }
        } catch(IOException e) {
            NetworkSender.this.client.errorWithThisClient(e);
        }
    }
    
    
	private synchronized boolean sendEverythingInClientPullBuffersAndBurstTerminatorAndClearQueue() {
		// returns true if there was actually anything sent other than a burst terminator
		try {
			boolean result = this.clientPullMessagesWaitingToBeSent;
			for(Serializable msg: this.clientPullToSendOther) {		
                actuallyCompressSendAndFlushIfBT(msg);
			}
			for(List<OpUpdateTileContents> msgs: this.clientPullToSendOpUpdateTileContentsByTileId.values()) {
				for(OpUpdateTileContents msg: msgs) {
                    actuallyCompressSendAndFlushIfBT(msg);
				}
			}
			for(OpCursorPos msg: this.clientPullToSendOpCursorPosByCursorId.values()) {
                actuallyCompressSendAndFlushIfBT(msg);
			}
            for(OpCursorShape msg: this.clientPullToSendOpCursorShapeByCursorId.values()) {
                actuallyCompressSendAndFlushIfBT(msg);
            }
			for(OpSetTileAff msg: this.clientPullToSendOpSetTileAffByElId.values()) {
                actuallyCompressSendAndFlushIfBT(msg);
			}
			for(OpSetLinkAff msg: this.clientPullToSendOpSetLinkAffByElId.values()) {
                actuallyCompressSendAndFlushIfBT(msg);
			}
			
			this.actuallyCompressSendAndFlushIfBT(new BurstTerminator() );         
			
			this.clientPullToSendOther.clear();
			this.clientPullToSendOpCursorPosByCursorId.clear();
            this.clientPullToSendOpCursorShapeByCursorId.clear();
			this.clientPullToSendOpSetTileAffByElId.clear();
			this.clientPullToSendOpSetLinkAffByElId.clear();
			this.clientPullToSendOpUpdateTileContentsByTileId.clear();
			this.clientPullMessagesWaitingToBeSent = false;
			return result;
		} catch(Throwable e) {
			this.client.errorWithThisClient(e);
			// we'll have died by this point anyway
			return false;
		}
	}
	
	
	synchronized void receivedClientReadyMessage() {
		// send everything in queue
		// then send burst terminator
		// if queue was empty then set serverpush=true
        assert this.serverPush == false;
        long t1 = System.currentTimeMillis();
        boolean anythingSentExceptBT = sendEverythingInClientPullBuffersAndBurstTerminatorAndClearQueue();
		if(!anythingSentExceptBT) {
			this.serverPush = true;
            // when we change to server push mode we also need to send anything that has
            // accumulated in the clientPullMessagesInCurrentUpdateFrame.
            // this hasn't been sent with the stuff in the client pull buffers
            // and in fact it can't be because it's not yet part of a finished frame.
            for(Serializable o: this.clientPullMessagesInCurrentUpdateFrame) {
                actuallyCompressSendAndFlushIfBT(o);
            }
            this.clientPullMessagesInCurrentUpdateFrame.clear();
            //System.out.println("Received crm, sent bt only, server push now "+this.serverPush+" time to respond = "+(System.currentTimeMillis()-t1));
		} else {
		    // still in client pull mode
            //System.out.println("Received crm, sent bt and others, Server push now "+this.serverPush+" time to respond = "+(System.currentTimeMillis()-t1));
		}
	}
	
	
		
}
