/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.client;


import java.awt.geom.Area;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.Socket;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Logger;

import t3.hrd.input.PointInputDevice;
import t3.hrd.input.ShapeInputDevice;
import t3.hrd.renderer.BlendOptions;
import t3.hrd.renderer.HRDRenderer;
import t3.hrd.renderer.Projector;
import t3.hrd.renderer.ProjectorConfig;
import t3.remotehrd.protocol.HelloMessageToServer;

/**
 * This class represents a RemoteHRD client, consisting of a HRDRenderer whose 
 * state is updated according to messages from a RemoteHRD server. 
 * 
 * @author pjt40
 *
 */

public class RemoteHRDClient {

	private static final Logger logger = Logger.getLogger("t3.remotehrd.client");
	/**
	 * The HRDRenderer that actually does the rendering for the client
	 */
	public final HRDRenderer hrdRenderer;
	
	
	final BufferedInputStream bis;
	final ObjectInputStream obis;
	final ObjectOutputStream obos;
	final Socket socket;
	final RemoteHRDClientCallBacks callBacks;
	
	
	
	/**
	 * Creates a new RemoteHRD client that listens on a specific socket. Most of the
	 * arguments are the same as for HRDRenderer. Call doRenderLoop to actually begin
	 * rendering.
	 * 
	 * @param projectorConfigs 
	 * @param socket	Socket on which to send/recieve messages.
	 * @param rCallBacks
	 * @param pointInputDevices
	 * @param mouseAsPointingDevice
	 * @throws Projector.ProjectorOpeningException
	 * @throws IOException
	 */
	public RemoteHRDClient(
			int clientId,
			List<ProjectorConfig> projectorConfigs, 
			Socket socket, 
			RemoteHRDClientCallBacks rCallBacks,
			LinkedList<PointInputDevice> pointInputDevices,
            ShapeInputDevice shapeInputDevice,
			boolean shareTexturesBetweenContexts,
			boolean mouseAsPointingDevice,
			BlendOptions bo            
	) throws Projector.ProjectorOpeningException, IOException {
		
        
        // care!
        // protocol is: 
        //      client creates outputstream
        //      client sends hello
        //      server creates outputstream
        this.socket = socket;
		this.socket.setTcpNoDelay(true);
        this.socket.setReceiveBufferSize(1024*1024*8);
		
        this.obos = new ObjectOutputStream(new BufferedOutputStream(this.socket.getOutputStream()));
        this.sendMessageToServer(new HelloMessageToServer(clientId));

        this.bis = new BufferedInputStream(new BandwidthMeasuringInputStream(this.socket.getInputStream()));
        this.obis = new ObjectInputStream(bis);
        
		this.callBacks = rCallBacks;
		
		this.hrdRenderer = new HRDRenderer(
				clientId,
				projectorConfigs,
				new HRDRendererCallbacksForRemoteHRDClient(this),
				pointInputDevices,
                shapeInputDevice,
				shareTexturesBetweenContexts,
				mouseAsPointingDevice,
                bo
				);
        
        
	}
	
	/**
	 * Serializes an object and sends it to the server. This method is thread-safe:
	 * it can be called from any thread. The objectstream is reset every 1000 objects.
	 * @param msg
	 */
	
	private int messageCount=0;
	
	public void sendMessageToServer(Serializable msg)  {
		synchronized(obos) {
			try {
				obos.writeObject(msg);
                //System.out.println("Sent message to server "+msg);
				logger.fine("Sent message to server "+msg);
				if(++messageCount>1000) {
					// reset so that we don't leak memeory
					obos.reset();
					messageCount=0;
				}
				obos.flush();
			} catch(IOException e) {
				throw new RuntimeException(e);
			}
		}
	}
	
	/**
	 * As HRDRenderer.doItSingleThreaded.
	 */
	public void doRenderLoop() {
		this.hrdRenderer.doItSingleThreaded();
	}
}
