/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios.commonbehaviours;


import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;

import t3.hrd.state.JOGLHelper;
import t3.hrd.state.ScaRotTraTransformImmutableUniformScale;
import t3.portfolios.PointInputDevice;
import t3.portfolios.PointInputDeviceTypeAndButton;
import t3.portfolios.PointInputDeviceTypeAndButtonSet;
import t3.portfolios.Portfolio;
import t3.portfolios.PortfolioCommonBehaviour;
import t3.portfolios.PortfolioEvent;
import Jama.Matrix;

/**
 * This class allows portfolios to be translated and rotated by dragging with a
 * specified pid type and button, using the RotateNTranslate idea. However, in this
 * implementation there is no "translation-only" region at the centre.
 * 
 * See:
 * Kruger, R., Carpendale, S., Scott, S.D., Tang, A. (2005). 
 * Fluid Integration of Rotation and Translation. 
 * In Proceedings of the ACM Conference on Human Factors in Computing Systems 
 * (CHI)'05, April 2-7, 2005, Portland, Oregon, USA. 
 * 
 * @author pjt40
 *
 */
public class RotateNTranslate  implements PortfolioCommonBehaviour {
	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios.commonbehaviours");
	
	public final PointInputDeviceTypeAndButtonSet pidtbs;
    private Set<Portfolio> portfoliosBeingDragged = new HashSet<Portfolio>();

    public RotateNTranslate() {
        this(new PointInputDeviceTypeAndButton(0,1));
    }
    
	public RotateNTranslate(PointInputDeviceTypeAndButton pidTypeAndButton) {
		this( new PointInputDeviceTypeAndButtonSet(pidTypeAndButton) );
	}
	

	public RotateNTranslate(PointInputDeviceTypeAndButtonSet pidtbs) {
		this.pidtbs = pidtbs;
	}

	public boolean customProcessEventForThisPortfolioNotChildren(Portfolio p, PortfolioEvent e, boolean bubbled) {
		if (
			e.pointInputDevice != null
			&& e.positionAndButtonsKnown
			&& this.pidtbs.contains(
				new PointInputDeviceTypeAndButton(e.pointInputDevice.pointInputDeviceType, e.pidButton)
			)
		) {
			if(e.eventType==e.EVENT_PID_PRESS && !this.portfoliosBeingDragged.contains(p)) {
                p.getParent().bringChildToFront(p);
				p.portfolioServer.enterFDOPmodeForPointInputDevice(p, e.pointInputDevice, e.pidButton, e.getPORTx(p), e.getPORTy(p));
                this.portfoliosBeingDragged.add(p);
                return true;
			} else {
				// we still don't want any events for this button to get through to the underlying
				// portfolio.
				return true;
			}
		} else {
			return false;
		}
	}

	public boolean customProcessFDOPevent(Portfolio p, PortfolioEvent e, double PORTxWhenEnteredFDOPmode, double PORTyWhenEnteredFDOPmode) {
		if(this.pidtbs.contains(
			new PointInputDeviceTypeAndButton(e.pointInputDevice.pointInputDeviceType, e.pidButton)
		)) {
			
			double PORTcentreX, PORTcentreY;
			if(p.childrenTopToBottomReadOnly.size()==0) {
				PORTcentreX=0.0;
				PORTcentreY=0.0;
			} else {
				double DESKcentreX = p.getRDESKboundingBoxOfOurTileAndAllDescendantsTiles().getCenterX();
				double DESKcentreY = p.getRDESKboundingBoxOfOurTileAndAllDescendantsTiles().getCenterY();
				double[] PORTcentre = JOGLHelper.getDFromM( 
						p.getmDESKtoPORTReadOnly().times(
							JOGLHelper.getMFromD(DESKcentreX, DESKcentreY, 1.0)
						)
					);
				PORTcentreX = PORTcentre[0]/PORTcentre[2];
				PORTcentreY = PORTcentre[1]/PORTcentre[2];
			}	
			
			
			// now lets work out our rotn
			// point W is PORTwhenenteredfdopmode
			// point N is PORTcursornow
			// point O is origin
			// we want to find alpha, the angle between WO and NO
			// WO.NO = |WO| |NO| cos alpha
			double WOdx = PORTcentreX-PORTxWhenEnteredFDOPmode;
			double WOdy = PORTcentreY-PORTyWhenEnteredFDOPmode;
			double NOdx = PORTcentreX-e.getPORTx(p);
			double NOdy = PORTcentreY-e.getPORTy(p);
			double WOdotNO = WOdx*NOdx+WOdy*NOdy;
			double modWO = Math.sqrt(WOdx*WOdx+WOdy*WOdy);
			double modNO = Math.sqrt(NOdx*NOdx+NOdy*NOdy);
			double cosAlpha = WOdotNO/(modWO*modNO);
			if(cosAlpha>1.0) {
				// correct for numerical errors
				cosAlpha = 1.0;
			} else {
				// no need to correct
			}
			double alpha = Math.acos(cosAlpha);
			// 0<=alpha<pi
			
			// now we need to work out whether alpha is clockwise or not
			// clockwise iff WOdy*NOdx-WOdx*NOdy>0
			// beta is the clockwise angle from WO to NO		
			double beta = (WOdy*NOdx-WOdx*NOdy>0.0) 
				? alpha
				: 2.0*Math.PI-alpha;
				
            // now rotate by beta clockwise 
            ScaRotTraTransformImmutableUniformScale curDESKtoPORT = 
                new ScaRotTraTransformImmutableUniformScale(p.getmDESKtoPORTReadOnly());
            ScaRotTraTransformImmutableUniformScale idealNewDESKtoPORT = 
                new ScaRotTraTransformImmutableUniformScale(
                    curDESKtoPORT.getScale(),
                    curDESKtoPORT.getThetaClockwise()-beta,
                    curDESKtoPORT.getTx(),
                    curDESKtoPORT.getTy()
                );
			
			
			// now where do we end up if we use this on our pen coords
			Matrix PORTpoint = idealNewDESKtoPORT.getMatrix2dHomogReadOnly().times( 
					JOGLHelper.getMFromD(e.DESKx, e.DESKy,1.0)
				);
			
			// now adjust based on this
			// we can do the translation comparison in portfolio space because 
			// the translation is applied last, after the rotn and scale.
			double tx = ( PORTpoint.get(0,0)/PORTpoint.get(2,0)-PORTxWhenEnteredFDOPmode );
			double ty = ( PORTpoint.get(1,0)/PORTpoint.get(2,0)-PORTyWhenEnteredFDOPmode );
			
			ScaRotTraTransformImmutableUniformScale s = new ScaRotTraTransformImmutableUniformScale(
				idealNewDESKtoPORT.getScale(),
				idealNewDESKtoPORT.getThetaClockwise(),
				idealNewDESKtoPORT.getTx() - tx,
				idealNewDESKtoPORT.getTy() - ty
				);


            p.setDESKtoPORT(s.getMatrix2dHomogReadOnly());
			return true;
		} else {
			return false;
		}
	}

	public boolean customProcessEndOfFDOPmode(Portfolio p, PointInputDevice pid, int button) {
		if(this.pidtbs.contains(
			new PointInputDeviceTypeAndButton(pid.pointInputDeviceType, button)
		)) {
            this.portfoliosBeingDragged.remove(p);
			return true;
		} else {
			return false;
		}
	}
		
	
}


