/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios;

import java.io.Serializable;
import java.util.logging.Logger;





/**
 * @author pjt40
 * 
 * Data structure class representing an event in a portfolio. The event can be generated
 * by a PointInputDevice (eg a stylus) (a "pid" event) or by a keyboard.
 * 
 * See PortfolioServer for more information on the event model.
 * 
 * This class is not thread safe. Its methods should be called only from within a T3 callback
 * or by calling portfolioserver.performActionAsynchronously(). Its public fields are all final
 * and so cannot be modified. 
 *
 */
public class PortfolioEvent {
	
	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios");
	
	// is a pen event iff pointInputDevice !=null

	public static final int
	  EVENT_PID_PRESS			= 1,
	  EVENT_PID_RELEASE			= 2,
	  EVENT_PID_CLICK			= 3,
	  EVENT_PID_MOVE			= 4,
	  EVENT_PID_ENTER			= 6,
	  EVENT_PID_EXIT			= 7,
	  EVENT_KEYBOARD_PRESSED	= 8,
	  EVENT_KEYBOARD_RELEASED	= 9,
	  EVENT_KEYBOARD_TYPED		= 10,
	  EVENT_PID_FDOPMODE		= 11,
      EVENT_PID_PRESSOUTSIDE    = 12;

	/**
	 * Type of event.
	 */
	public final int eventType;
	

	/**
	 * The Person that generated the event.
	 */
	public final Person person;
		
	/**
	 * For pid events, this is the pid that generated the event. For non-pid events
	 * this is null.
	 */
	public final PointInputDevice pointInputDevice;
	
	/**
	 * For pid events, this is true iff the position and buttons of the pid are known.
	 * For non-pid events this will be false.
	 * <p>
	 * It will always be true for pid press, pressoutside, pid release, pid click, pid move and pid enter
	 * events. It might be true for pid exit events. 
	 */
	public final boolean positionAndButtonsKnown;
	

	/**
	 * For pid events, this is true iff the position of the pid was known before this
	 * latest event occurred. For non-pid events this will be false.	 * 
	 */
	public final boolean oldPositionKnown;
	
	/**
	 * If positionAndButtonsKnown then this is the coordinates of event in DESK space, 
	 * i.e. in the coordinate space of the display surface.
	 */
	public final double DESKx, DESKy;
	
    public final Serializable extra;
	
	/**
	 * If oldPositionKnown then this is the coordinates of event in DESK space, 
	 * before this event occurred.
	 */
	public final double DESKxOld, DESKyOld;
	

	/**
	 * For pid press, pressoutside, release and click events, this is the button number that caused the event.
	 * For fdop mode events, this is the button number that you entered fdop mode with.
	 */
	public final int pidButton; 
	
	
	/**
	 * If positionAndButtonsKnown then this is the pid's buttons bit field.
	 */
	public final int pidButtons;
	
	
	/**
	 * For keyboard events.
	 */
	public final int keyboardAWTKeyCode; 
	
	/**
	 * For keyboard events.
	 */
	public final char keyboardAWTChar; 
	
	
	/**
	 * For keyboard events.
	 */
	public final int keyboardAWTModifiers; 
	
	
	private double PORTx, PORTy;
	private double PORTxOld, PORTyOld; // only set for mouse move
	private int TILEx, TILEy;
	private int TILExOld, TILEyOld; // only set for mouse move
	private boolean onTile, oldOnTile;
	private Portfolio relToPortfolio;
	private Portfolio relToTile;
	
	
	/**
	 * Creates a new data structure to store the specified data.
	 * @param penButton
	 * @param penButtons
	 * @param client
	 * @param pen
	 * @param DESKx
	 * @param DESKy
	 * @param DESKxOld
	 * @param DESKyOld
	 * @param eventType
	 * @param positionAndButtonsKnown
	 * @param oldPositionKnown
	 * @param keyboardAWTKeyCode
	 * @param keyboardAWTChar
	 * @param keyboardAWTModifiers
	 */
	public PortfolioEvent(
			int penButton, 
			int penButtons, 
			Person person, 
			PointInputDevice pen, 
			double DESKx, 
			double DESKy, 
			double DESKxOld, 
			double DESKyOld,
            Serializable extra, 
			int eventType, 
			boolean positionAndButtonsKnown, 
			boolean oldPositionKnown,
			int keyboardAWTKeyCode,
			char keyboardAWTChar, 
			int keyboardAWTModifiers
	) {
		// TODO Auto-generated constructor stub
		this.pidButton = penButton;
		this.pidButtons = penButtons;
		this.person = person;
		this.pointInputDevice = pen;
		this.DESKx = DESKx;
		this.DESKy = DESKy;
		this.DESKxOld = DESKxOld;
		this.DESKyOld = DESKyOld;
        this.extra = extra;
		this.eventType = eventType;
		this.positionAndButtonsKnown = positionAndButtonsKnown;
		this.oldPositionKnown = oldPositionKnown;
		this.relToPortfolio=null;
		this.relToTile=null;
		this.keyboardAWTKeyCode = keyboardAWTKeyCode;
		this.keyboardAWTChar = keyboardAWTChar;
		this.keyboardAWTModifiers = keyboardAWTModifiers;
		logger.fine("Created new event - "+this);
	}
	

	  
	
	/**
	 * If positionAndButtonsKnown then this returns the x coordinate of the event
	 * relative to the specified portfolio's PORT space. Results are cached. 
	 * @param p
	 * @return
	 */
	public  final double getPORTx(Portfolio p) {
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		this.makeRelToPortfolio(p);
		return this.PORTx;
	}
	
	/**
	 * If positionAndButtonsKnown then this returns the y coordinate of the event
	 * relative to the specified portfolio's PORT space. Results are cached. 
	 * @param p
	 * @return
	 */
	public  final double getPORTy(Portfolio p) {
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		this.makeRelToPortfolio(p);
		return this.PORTy;
	}
	
	/**
	 * If oldPositionKnown then this returns the x coordinate of the pid before the
	 * event happened, relative to the specified portfolio's PORT space. Results are cached.
	 * @param p
	 * @return
	 */
	public  final double getPORTxOld(Portfolio p) {
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		this.makeRelToPortfolio(p);
		return this.PORTxOld;
	}
	
	/**
	 * If oldPositionKnown then this returns the y coordinate of the pid before the
	 * event happened, relative to the specified portfolio's PORT space. Results are cached.
	 * @param p
	 * @return
	 */
	public  final double getPORTyOld(Portfolio p) {
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		this.makeRelToPortfolio(p);
		return this.PORTyOld;
	}
	
	/**
	 * If positionAndButtonsKnown then this returns iff the coordinates of the event
	 * lie on the specified portfolio's tile. Results are cached.  
	 * @param p
	 * @return
	 */
	public  final boolean isPointOnTile(Portfolio p) {
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		this.makeRelToTile(p);
		return this.onTile;
	}
	
	/**
	 * If oldPositionKnown then this returns true iff the coordinates of the pid before the
	 * event happened lie on the specified portfolio's tile. Results are cached. 
	 * @param p
	 * @return
	 */
	public  final boolean isOldPointOnTile(Portfolio p) {
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		this.makeRelToTile(p);
		return this.oldOnTile;
	}
	
	/**
	 * If positionAndButtonsKnown then this returns the x coordinate of the event
	 * relative to the specified portfolio's TILE space. Results are cached.
	 * @param p
	 * @return
	 */
	public  final int getTILEx(Portfolio p) {
		// not guaranteed to actually be within p's tile, of course
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		this.makeRelToTile(p);
		if(!this.onTile) {
			throw new IllegalArgumentException("Event is not on portfolio's tile "+p);
		}
		return this.TILEx;
	}
	
	/**
	 * If positionAndButtonsKnown then this returns the y coordinate of the event
	 * relative to the specified portfolio's TILE space. Results are cached. 
	 * @param p
	 * @return
	 */
	public  final int getTILEy(Portfolio p) {
		// not guaranteed to actually be within p's tile, of course
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		this.makeRelToTile(p);
		if(!this.onTile) {
			throw new IllegalArgumentException("Event is not on portfolio's tile "+p);
		}
		return this.TILEy;
	}
	
	/**
	 * If oldPositionKnown then this returns the x coordinate of the pid before the
	 * event happened, relative to the specified portfolio's TILE space. Results are cached.
	 * Of course, the result is not guaranteed to lie within the portfolio's tile.
	 * @param p
	 * @return
	 */
	public  final int getTILExOld(Portfolio p) {
		// not guaranteed to actually be within p's tile, of course
		if(!this.oldPositionKnown) {
			throw new IllegalStateException("Old position is not known for this event.");
		}
		this.makeRelToTile(p);
		if(!this.oldOnTile) {
			throw new IllegalArgumentException("Event's old coordinates are not on portfolio's tile "+p);
		}
		return this.TILExOld;
	}
	
	/**
	 * If oldPositionKnown then this returns the y coordinate of the pid before the
	 * event happened, relative to the specified portfolio's TILE space. Results are cached.
	 * Of course, the result is not guaranteed to lie within the portfolio's tile.
	 * @param p
	 * @return
	 */
	public  final int getTILEyOld(Portfolio p) {
		// not guaranteed to actually be within p's tile, of course
		if(!this.oldPositionKnown) {
			throw new IllegalStateException("Old position is not known for this event.");
		}
		this.makeRelToTile(p);
		if(!this.oldOnTile) {
			throw new IllegalArgumentException("Event's old coordinates are not on portfolio's tile "+p);
		}
		return this.TILEyOld;
	}	
	
	private  final void makeRelToPortfolio(final Portfolio p) {
		if(!this.positionAndButtonsKnown) {
			throw new IllegalStateException("Position and buttons are not known for this event.");
		}
		if(this.relToPortfolio!=p) {
			double[] portfolioXAndY = p.getUd2PORTfromUd2DESK(new double[] { PortfolioEvent.this.DESKx, PortfolioEvent.this.DESKy } );
			PortfolioEvent.this.PORTx = portfolioXAndY[0];
            PortfolioEvent.this.PORTy = portfolioXAndY[1];
			if(PortfolioEvent.this.oldPositionKnown) {
				portfolioXAndY = p.getUd2PORTfromUd2DESK(new double[] { PortfolioEvent.this.DESKxOld, PortfolioEvent.this.DESKyOld } );
                PortfolioEvent.this.PORTxOld = portfolioXAndY[0];
                PortfolioEvent.this.PORTyOld = portfolioXAndY[1];
			} else {
				// no need
			}
            PortfolioEvent.this.relToPortfolio = p;
		} else {
			// do nothing
		}
		return;
	}
	
	private  final void makeRelToTile(Portfolio p) {
		if(this.relToTile!=p) {
			if(!p.hasTile()) {
				throw new IllegalStateException("Portfolio has no tile "+p);
			}
			if(!this.positionAndButtonsKnown) {
				throw new IllegalStateException("Position and buttons are not known for this event.");
			}
			
			int[] xy = new int[2];
			
			this.onTile = 
				p.getIntegerTileSpaceCoordsFromDESK(
					xy,
					new double[] { this.DESKx, this.DESKy, 1.0 }
				);
			if(this.onTile) {
				this.TILEx = xy[0];
				this.TILEy = xy[1];
			} else {
				// do nothing
			}
			
			if(this.oldPositionKnown) {
				this.oldOnTile = 
					p.getIntegerTileSpaceCoordsFromDESK(
						xy,
						new double[] { this.DESKxOld, this.DESKyOld, 1.0 }
					);
				if(this.oldOnTile) {
					this.TILExOld = xy[0];
					this.TILEyOld = xy[1];
				} else {
					// do nothing
				}
			} else {
				// no need
			}

			this.relToTile = p;
		} else {
			// do nothing
		}
		return;
	}
	
	public String toString() {
		return super.toString() + " type="+this.eventType;
	}
}
