/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef BINARYPATCHTREE_H
#define BINARYPATCHTREE_H

#include <tr1/array>
#include "patchtree.h"

namespace snurbs {

/**
 * Splits a PatchTree in two, either North-South or West-East.
 *
 * @ingroup Mesh
 */
class BinaryPatchTree : public PatchTree
{
public:
    /**
     * BinaryPatchTree constructor.
     * @param splitType Specifies whether the binary split is North-South
     *                  (@c splitType PatchTree::VERTICAL ) or West-East
     *                  (@c splitType PatchTree::HORIZONTAL ).
     * @param child1    The North or West child, depending on @c splitType .
     * @param child2    The South or East child, depending on @c splitType .
     */
    BinaryPatchTree(PatchTree::Orientation splitType,
                    PatchTree *child1,
                    PatchTree *child2);

    /** BinaryPatchTree destructor. */
    virtual ~BinaryPatchTree(void);

    bool isLeaf(void) const;

    bool isRoot(void) const;

    bool splits(PatchTree::Orientation orientation) const;

    PatchTree *&getChild(Direction dir);

    PatchTree *&getParent(void);

    void setParent(PatchTree *newParent);

    Direction dirFromParent(void) const;

    Direction childDir(const PatchTree *child) const;

    /**
     * Used to implement
     * operator<<(std::ostream &os, const PatchTreeRoot &patch);
     */
    void stream(std::ostream &os, unsigned char level) const;

private:
    /** The direction of the split for this BinaryPatchTree. */
    const Orientation splitType;
    /** The parent PatchTree. */
    PatchTree        *parent;
    /** The two child PatchTree objects. */
    std::tr1::array<PatchTree *, 2> children;
};

inline bool BinaryPatchTree::isLeaf(void) const
{
    return false;
}

inline bool BinaryPatchTree::isRoot(void) const
{
    return false;
}

inline bool BinaryPatchTree::splits(PatchTree::Orientation orientation) const
{
    return splitType == orientation;
}

inline PatchTree *&BinaryPatchTree::getParent(void)
{
    return parent;
}

inline void BinaryPatchTree::setParent(PatchTree *newParent)
{
    parent = newParent;
}

}

#endif
