/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QFileDialog>
#include <QMessageBox>
#include <QSlider>
#include <QTimer>
#include "glwidget.h"
#include "localelabel.h"
#include "parityspinbox.h"
#include "snurbsviswindow.h"

SnurbsVisWindow::SnurbsVisWindow(QWidget *parent) : QMainWindow(parent)
{
    // The Qt documentation says that the stencil and alpha buffers are
    // disabled by default, but I find they are enabled even without these
    // calls. We'll make the calls anyway -- perhaps my particular OpenGL
    // implementation is making a difference somehow
    QGLFormat fmt;
    fmt.setStencil(true);
    fmt.setAlpha(true);
    QGLFormat::setDefaultFormat(fmt);

    setupUi(this);
    glWidget->setVisible(false);

    const QColor bg = statusBar()->palette().color(QPalette::Background);
    QPalette newPalette = statusBar()->palette();
    newPalette.setColor(QPalette::Light, bg);
    newPalette.setColor(QPalette::Midlight, bg);
    newPalette.setColor(QPalette::Dark, bg);
    newPalette.setColor(QPalette::Mid, bg);
    newPalette.setColor(QPalette::Shadow, bg);
    statusBar()->setPalette(newPalette);

    LocaleLabel *vertexCount = new LocaleLabel();
    vertexCount->setNum(0);
    QWidget *vertexCountHolder = new QWidget();
    QHBoxLayout *vertexLayout  = new QHBoxLayout();
    vertexCountHolder->setLayout(vertexLayout);
    vertexLayout->setMargin(4);
    vertexLayout->setSpacing(2);
    vertexLayout->addWidget(new QLabel("Vertices: "));
    vertexLayout->addWidget(vertexCount);
    statusBar()->addPermanentWidget(vertexCountHolder);

    LocaleLabel *faceCount = new LocaleLabel();
    faceCount->setNum(0);
    QWidget *faceCountHolder = new QWidget();
    QHBoxLayout *faceLayout  = new QHBoxLayout();
    faceCountHolder->setLayout(faceLayout);
    faceLayout->setMargin(4);
    faceLayout->setSpacing(2);
    faceLayout->addWidget(new QLabel("Faces: "));
    faceLayout->addWidget(faceCount);
    statusBar()->addPermanentWidget(faceCountHolder);

    LocaleLabel *knotCount = new LocaleLabel();
    knotCount->setNum(0);
    QWidget *knotCountHolder = new QWidget();
    QHBoxLayout *knotLayout  = new QHBoxLayout();
    knotCountHolder->setLayout(knotLayout);
    knotLayout->setMargin(4);
    knotLayout->setSpacing(2);
    knotLayout->addWidget(new QLabel("Knot intervals: "));
    knotLayout->addWidget(knotCount);
    statusBar()->addPermanentWidget(knotCountHolder);

    insertToolBarBreak(subdivToolBar);

    subdivDegreeSpinBox = new ParitySpinBox();
    subdivDegreeSpinBox->setMaximum(27);
    subdivDegreeSpinBox->setMinimum(3);
    subdivDegreeSpinBox->setSingleStep(2);
    subdivDegreeSpinBox->setValue(3);
    subdivDegreeSpinBox->setMinimumSize(55,25);
    subdivDegreeSpinBox->setMaximumSize(55,25);
    QWidget *degreeHolder = new QWidget();
    QHBoxLayout *degreeLayout = new QHBoxLayout();
    degreeHolder->setLayout(degreeLayout);
    subdivToolBar->addWidget(degreeHolder);
    degreeLayout->addWidget(new QLabel("Degree  "));
    degreeLayout->addWidget(subdivDegreeSpinBox);
    degreeLayout->setMargin(4);
    degreeLayout->setSpacing(2);

    subdivLevelSpinBox = new QSpinBox();
    subdivLevelSpinBox->setMaximum(8);
    subdivLevelSpinBox->setMinimumSize(55,25);
    subdivLevelSpinBox->setMaximumSize(55,25);
    QWidget *levelHolder = new QWidget();
    QHBoxLayout *levelLayout = new QHBoxLayout();
    levelHolder->setLayout(levelLayout);
    subdivToolBar->addWidget(levelHolder);
    levelLayout->addWidget(new QLabel("Level  "));
    levelLayout->addWidget(subdivLevelSpinBox);
    levelLayout->setMargin(4);
    levelLayout->setSpacing(2);

    knotSpacingSpinBox = new QDoubleSpinBox();
    knotSpacingSpinBox->setMinimumSize(55,25);
    knotSpacingSpinBox->setMaximumSize(55,25);
    knotSpacingHolder = new QWidget();
    QHBoxLayout *knotSpacingLayout = new QHBoxLayout();
    knotSpacingHolder->setLayout(knotSpacingLayout);
    subdivToolBar->addWidget(knotSpacingHolder);
    knotSpacingLayout->addWidget(new QLabel("Knot interval  "));
    knotSpacingLayout->addWidget(knotSpacingSpinBox);
    knotSpacingLayout->setMargin(4);
    knotSpacingLayout->setSpacing(2);
    newKnotSpacing(-1);

    QSlider *stripeDensitySlider = new QSlider(Qt::Horizontal);
    stripeDensitySlider->setMinimum(2);
    stripeDensitySlider->setMaximum(20);
    stripeDensitySlider->setValue(8);
    QWidget *stripeDensityHolder = new QWidget();
    QHBoxLayout *stripeDensityLayout = new QHBoxLayout();
    stripeDensityHolder->setLayout(stripeDensityLayout);
    subdivToolBar->addWidget(stripeDensityHolder);
    stripeDensityLayout->addWidget(new QLabel("Reflection line density  "));
    stripeDensityLayout->setMargin(4);
    stripeDensityLayout->setSpacing(2);

    stripeDensityHolder = new QWidget();
    stripeDensityLayout = new QHBoxLayout();
    stripeDensityHolder->setLayout(stripeDensityLayout);
    subdivToolBar->addWidget(stripeDensityHolder);
    stripeDensityLayout->addWidget(stripeDensitySlider);
    stripeDensityLayout->setMargin(4);
    stripeDensityLayout->setSpacing(2);

    QSlider *isoparmDensitySlider = new QSlider(Qt::Horizontal);
    isoparmDensitySlider->setMinimum(0);
    isoparmDensitySlider->setMaximum(2);
    isoparmDensitySlider->setValue(2);
    QWidget *isoparmDensityHolder = new QWidget();
    QHBoxLayout *isoparmDensityLayout = new QHBoxLayout();
    isoparmDensityHolder->setLayout(isoparmDensityLayout);
    subdivToolBar->addWidget(isoparmDensityHolder);
    isoparmDensityLayout->addWidget(new QLabel("Isoparm density  "));
    isoparmDensityLayout->setMargin(4);
    isoparmDensityLayout->setSpacing(2);

    isoparmDensityHolder = new QWidget();
    isoparmDensityLayout = new QHBoxLayout();
    isoparmDensityHolder->setLayout(isoparmDensityLayout);
    subdivToolBar->addWidget(isoparmDensityHolder);
    isoparmDensityLayout->addWidget(isoparmDensitySlider);
    isoparmDensityLayout->setMargin(4);
    isoparmDensityLayout->setSpacing(2);

    connect(subdivLevelSpinBox,    SIGNAL(valueChanged(int)),
            this,                    SLOT(changeSubdivLevel(int)));

    connect(glWidget,              SIGNAL(subdivLevelChanged(int)),
            subdivLevelSpinBox,      SLOT(setValue(int)));

    connect(subdivDegreeSpinBox,   SIGNAL(valueChanged(int)),
            this,                    SLOT(changeDegree(int)));

    connect(actionWireframe,       SIGNAL(triggered(void)),
            glWidget,                SLOT(setWireframe(void)));

    connect(actionFlat,            SIGNAL(triggered(void)),
            glWidget,                SLOT(setFlat(void)));

    connect(actionSmooth,          SIGNAL(triggered(void)),
            glWidget,                SLOT(setSmooth(void)));

    connect(actionIsophotes,       SIGNAL(triggered(void)),
            glWidget,                SLOT(setIsophotes(void)));

    connect(actionIsoparams,       SIGNAL(triggered(void)),
            glWidget,                SLOT(setIsoparams(void)));

    connect(actionDrawControlMesh, SIGNAL(toggled(bool)),
            glWidget,                SLOT(drawControlMesh(bool)));

    connect(glWidget,              SIGNAL(numVertsChanged(int)),
            vertexCount,             SLOT(setNum(int)));

    connect(glWidget,              SIGNAL(numFacesChanged(int)),
            faceCount,               SLOT(setNum(int)));

    connect(glWidget,              SIGNAL(numKnotIntsChanged(int)),
            knotCount,               SLOT(setNum(int)));

    connect(knotSpacingSpinBox,    SIGNAL(valueChanged(double)),
            this,                    SLOT(newKnotSpacing(double)));

    connect(glWidget,              SIGNAL(knotSpacingChanged(double)),
            this,                    SLOT(newKnotSpacing(double)));

    connect(stripeDensitySlider,   SIGNAL(valueChanged(int)),
            glWidget,                SLOT(changeStripeDensity(int)));

    connect(isoparmDensitySlider,  SIGNAL(valueChanged(int)),
            glWidget,                SLOT(changeIsoparamSubd(int)));

    // Only display the GLWidget when it has finished initialising
    QTimer::singleShot(0, glWidget, SLOT(show()));
}

void SnurbsVisWindow::newKnotSpacing(double interval)
{
    if (interval >= 0)
    {
        if (interval != knotSpacingSpinBox->value())
        {
            knotSpacingSpinBox->setValue(interval);
        }

        knotSpacingSpinBox->setDisabled(true);
        glWidget->changeKnotSpacing(interval);
        knotSpacingSpinBox->setEnabled(true);
        knotSpacingHolder->setEnabled(true);
    }
    else
    {
        knotSpacingHolder->setEnabled(false);
    }
}

void SnurbsVisWindow::on_actionOpen_triggered(void)
{
    QString fileName = QFileDialog::getOpenFileName(this,
                           tr("Open File"), QDir::currentPath(),
                           tr("Stanford PLY meshes (*.ply)"));

    if (!fileName.isNull())
    {
        glWidget->loadFile(fileName);
        statusBar()->showMessage("Loaded " + fileName, 5000);
    }
}

void SnurbsVisWindow::on_actionSave_As_triggered(void)
{
    QStringList filters;
    filters << "Stanford PLY meshes (*.ply)"
            << "UV format meshes (*.uvf)";

    QFileDialog fileDialog(this, tr("Save file"), QDir::currentPath(), NULL);
    fileDialog.setFilters(filters);
    fileDialog.setFileMode(QFileDialog::AnyFile);
    fileDialog.setAcceptMode(QFileDialog::AcceptSave);
    fileDialog.setDefaultSuffix("ply");

    if (fileDialog.exec())
    {
        QStringList files = fileDialog.selectedFiles();
        if (!files.isEmpty())
        {
            if (files[0].endsWith(".uvf"))
            {
                glWidget->saveFile(files[0], GLWidget::UVF);
            }
            else
            {
                glWidget->saveFile(files[0], GLWidget::PLY);
            }
            statusBar()->showMessage("Saved " + files[0], 5000);
        }
    }
}

void SnurbsVisWindow::on_actionSave_Image_triggered(void)
{
    QStringList filters;
    filters << "Raster image files (*.png)"
            << "Vector image files (*.eps *.pdf *.pgf)";

    QFileDialog fileDialog(this, tr("Save image"), QDir::currentPath(), NULL);
    fileDialog.setFilters(filters);
    fileDialog.setFileMode(QFileDialog::AnyFile);
    fileDialog.setAcceptMode(QFileDialog::AcceptSave);
    fileDialog.setDefaultSuffix("png");

    // Save the snapshot in memory before the frame buffer is covered by the
    // file save dialog
    QImage snapshot = glWidget->grabFrameBuffer(true);
    if (fileDialog.exec())
    {
        QStringList files = fileDialog.selectedFiles();
        if (!files.isEmpty())
        {
            if (files[0].endsWith(".eps"))
            {
                glWidget->saveVectorImage(files[0], GLWidget::EPS);
            }
            else if (files[0].endsWith(".pdf"))
            {
                glWidget->saveVectorImage(files[0], GLWidget::PDF);
            }
            else if (files[0].endsWith(".pgf"))
            {
                glWidget->saveVectorImage(files[0], GLWidget::PGF);
            }
            else
            {
                // Save the snapshot to disk
                snapshot.save(files[0]);
            }
            statusBar()->showMessage("Saved image " + files[0], 5000);
        }
    }
}

void SnurbsVisWindow::display_ModelView_Matrix(void)
{
    float mvMatrix[16];
    glWidget->getModelViewMatrix(mvMatrix);
    QString text = "{";

    for(int i = 0; i < 15; i++)
    {
        text += QString::number(mvMatrix[i]) + ", ";
    }
    text += QString::number(mvMatrix[15]) + "};";

    QMessageBox::information(this, "ModelView Matrix", text);
}

void SnurbsVisWindow::display_Selected_Knot_Details(void)
{
    QString text = "Face " + QString::number(glWidget->getSelectedFace());
    text += ", edge " + QString::number(glWidget->getSelectedEdge());

    QMessageBox::information(this, "Selected Knot", text);
}

void SnurbsVisWindow::changeSubdivLevel(int newLevel)
{
    subdivLevelSpinBox->setDisabled(true);
    glWidget->changeSubdivLevel(newLevel);
    subdivLevelSpinBox->setDisabled(false);
    subdivLevelSpinBox->setFocus(Qt::OtherFocusReason);
}

void SnurbsVisWindow::changeDegree(int newDegree)
{
    subdivDegreeSpinBox->setDisabled(true);
    glWidget->changeDegree(newDegree);
    subdivDegreeSpinBox->setDisabled(false);
    subdivDegreeSpinBox->setFocus(Qt::OtherFocusReason);
}

void SnurbsVisWindow::autoCropAndSaveGLWidget(QString fileName, bool withAlpha)
{
    QImage image = glWidget->grabFrameBuffer(withAlpha);
    QImage mask;

    if (withAlpha)
    {
        mask = image.createAlphaMask();
    }
    else
    {
        mask = image.createHeuristicMask();
    }

    int x_min = image.width(), y_min = image.height(),
        x_max = -1, y_max = -1;

    for (int i = 0; i < mask.width(); ++i)
    {
        for (int j = 0; j < mask.height(); ++j)
        {
            if (!(mask.pixel(i, j) & 0x00ffffff))
            {
                if (i < x_min) x_min = i;
                if (i > x_max) x_max = i;
                if (j < y_min) y_min = j;
                if (j > y_max) y_max = j;
            }
        }
    }

    image.copy(x_min, y_min, x_max - x_min, y_max - y_min).save(fileName);
}
