/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <cstdlib>
#include <getopt.h>
#include <iostream>
#include <snurbs.h>

using namespace snurbs;
using namespace std;

static const char *optString = "s:d:o:h?ui";

static const struct option longOpts[] = {
    { "steps", required_argument, NULL, 's' },
    { "degree", required_argument, NULL, 'd' },
    { "outfile", required_argument, NULL, 'o' },
    { "help", no_argument, NULL, 'h' },
    { "uvf", no_argument, NULL, 'u' },
    { "incompatible", no_argument, NULL, 'i' },
    { NULL, no_argument, NULL, 0 }
};

void display_usage(void)
{
    cout << endl;
    cout << "subdnurbs -- Subdivider for NURBS with Extraordinary Points" <<
            endl << endl;
    cout << "  Usage: subdnurbs [options] plyinput" << endl << endl;
    cout << "  Subdivides the mesh read from 'plyinput' and writes out the new"
 << endl << "  mesh in PLY format" << endl << endl;
    cout << "  Options:" << endl << endl;
    cout << "    --steps=#, -s #   The number of subdivision steps to take"
         << endl;
    cout << "    --degree=#, -d #  The degree at which to subdivide"
         << endl;
    cout << "    --outfile=file,   Write to 'file' instead of standard output"
         << endl;
    cout << "           -o file" << endl;
    cout << "    --help, -h        Display this help" << endl;
    cout << endl;
    exit(EXIT_FAILURE);
}

int main(int argc, char *argv[])
{
    int       degree = 3, steps = 1;
    string    fileIn, fileOutName;
    ofstream  fileOut;
    ostream  *fileOutStream = &cout;
    int       opt = 0;
    bool      uvfWrite = false;
    bool      compatibleOut = true;

    while (opt != -1)
    {
        opt = getopt_long( argc, argv, optString, longOpts, NULL );

        switch (opt)
        {
            case 's':
                steps = atoi(optarg);
                if (steps < 0)
                {
                    cerr << "Cannot use a negative number of subdivision steps"
                         << endl;
                    return EXIT_FAILURE;
                }
                break;
            case 'd':
                degree = atoi(optarg);

                if (degree % 2 == 0 || degree < 3)
                {
                    cerr << "Degree must be 3 or greater and odd" << endl;
                    return EXIT_FAILURE;
                }
                break;
            case 'o':
                fileOutName = optarg;
                fileOutStream = &fileOut;
                break;
            case 'h':
            case '?':
                display_usage();
                break;
            case 'u':
                uvfWrite = true;
                break;
            case 'i':
                compatibleOut = false;
                break;
            default:
                break;
        }
    }

    if (argc - optind > 0)
    {
        fileIn = *(argv + optind);
        fileOut.open(fileOutName.c_str());
    }
    else
    {
        display_usage();
    }

    Mesh mesh(degree);

    MeshBuilder *meshBuilder = MeshBuilder::create(mesh);
    PlyReader plyReader(*meshBuilder);
    plyReader.read(fileIn.c_str());
    MeshBuilder::dispose(meshBuilder);

    while (steps)
    {
        mesh.subdivide();
        --steps;
    }

    FlatMeshHandler *meshHandler;
    MeshFlattener *meshFlattener = MeshFlattener::create(mesh);
    if (compatibleOut)
    {
        meshFlattener->setCompatible(true);
    }
    if (uvfWrite)
    {
        meshHandler = new UvfWriter(fileOutStream);
    }
    else
    {
        meshHandler = new PlyWriter(fileOutStream);
    }
    meshFlattener->flatten(*meshHandler);
    delete meshHandler;
    MeshFlattener::dispose(meshFlattener);
    fileOut.close();

    return EXIT_SUCCESS;
}
