/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "plywriter.h"

namespace snurbs {

using namespace std;

PlyWriter::PlyWriter(ostream *fileStream) : file(fileStream),
                                            numVertices(0),
                                            numFaces(0),
                                            numKnotIntervals(0)
{
}

PlyWriter::~PlyWriter(void)
{
}

// The header is written in endFile(), when we know the number of each type of
// element. A stringstream buffer is used to save the contents of the PLY
// file in memory until we've written the header. So we don't implement
// startMesh.

void PlyWriter::addVertex(VertexPrecision x,
                          VertexPrecision y,
                          VertexPrecision z,
                          VertexPrecision w)
{
    buffer << x << " " << y << " " << z << " " << w << endl;
    ++numVertices;
}

void PlyWriter::startFace(unsigned int numEdges)
{
    buffer << numEdges;
}

void PlyWriter::addToFace(unsigned int vertNum)
{
    buffer << " " << vertNum;
}

void PlyWriter::closeFace(void)
{
    buffer << endl;
    ++numFaces;
}

void PlyWriter::addKnotInterval(unsigned int vertex1,
                                unsigned int vertex2,
                                KnotPrecision interval)
{
    buffer << vertex1 << " " << vertex2 << " " << interval << endl;
    ++numKnotIntervals;
}

void PlyWriter::finishKnotIntervals(void)
{
    *file << "ply" << endl;
    *file << "format ascii 1.0" << endl;
    *file << "comment created by libsnurbs" << endl;
    *file << "element vertex " << numVertices << endl;
    *file << "property float32 x" << endl;
    *file << "property float32 y" << endl;
    *file << "property float32 z" << endl;
    *file << "property float32 w" << endl;
    *file << "element face " << numFaces << endl;
    *file << "property list uint8 int32 vertex_indices" << endl;
    *file << "element knot_interval " << numKnotIntervals << endl;
    *file << "property int32 vertex1" << endl;
    *file << "property int32 vertex2" << endl;
    *file << "property float32 value" << endl;
    *file << "end_header" << endl;

    // Copy the contents of the buffer to the output file.
    istreambuf_iterator<char> in(buffer);
    istreambuf_iterator<char> eos;
    ostreambuf_iterator<char> out(*file);
    copy(in, eos, out);
}

}
