/***************************************************************************
 *   Copyright (C) 2008 by Tom Cashman                                     *
 *   Tom.Cashman@cantab.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef BCCOEFFS_H
#define BCCOEFFS_H

#include "types.h"

/** @file */

namespace snurbs {

/**
 * Provides aliases for the third index into primalCoeffs.
 */
enum Coeffs {
    ALPHA, BETA, GAMMA
};

/**
 * The tuned coefficients for bounded curvature, for all valencies from 3 to
 * 256 and all odd degrees from 3 to 27. The ALPHA values for valency 3 are
 * actually the 'delta' values used in the additional smoothing stage -- see
 * our papers for details
 */
const VertexPrecision primalCoeffs[254][13][3] = {
{ {0.0012886511489462, 1.2559772456395570, 1.4012464430010092},
{0.0813483700637746, 1.1369515006745343, 1.1003166147671977},
{0.1915497577311861, 1.1204258760070218, 1.0757763885557989},
{0.3536944401478457, 1.1172064783228202, 1.0608143988506675},
{0.5732296613175291, 1.1149242372312997, 1.0526045362517606},
{0.8549610427969004, 1.1142798207328306, 1.0472722412342501},
{1.2030610446829895, 1.1135952478199185, 1.0436499348495836},
{1.6212300081685374, 1.1135488474039925, 1.0409005149484751},
{2.1128481429051660, 1.1132891684573454, 1.0389129829681563},
{2.6810526063187061, 1.1133946594372404, 1.0372427302491660},
{3.3282767936169990, 1.1132924602906689, 1.0359982653500110},
{4.0572298886661597, 1.1134334859778319, 1.0348795173576215},
{4.8695569895401025, 1.1133992625651357, 1.0340307395806700} },
{ {1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000},
{1.0000000000000000, 1.0000000000000000, 1.0000000000000000} },
{ {3.2598603450138537, 0.6499456120303999, 0.6223823475311345},
{10.0801710677166483, 0.7918658572608680, 0.7726540849956470},
{16.9678751322614687, 0.8151905113647110, 0.8227530074611977},
{24.3880571655063960, 0.8338864920256170, 0.8466083793678272},
{32.0751914592940963, 0.8406886389217954, 0.8625014458591526},
{40.1827164716683356, 0.8471743160705354, 0.8722102562811674},
{48.5633522396298360, 0.8503174302535742, 0.8798018899948948},
{57.2906901177225905, 0.8534461021637173, 0.8851151419618336},
{66.2651095373971515, 0.8552139254452672, 0.8895605827850168},
{75.5289148985315677, 0.8570137290263192, 0.8929301308857391},
{85.0123218976032575, 0.8581281343504853, 0.8958613684506902},
{94.7440864512279575, 0.8592798474578948, 0.8981999514263773},
{104.6719426937132624, 0.8600372436296078, 0.9002867479955811} },
{ {6.9618173725961014, 0.4364618289671282, 0.4114906093344772},
{23.3396333697176814, 0.6047899465979556, 0.5638482894738585},
{44.0209330264004635, 0.6416688165442822, 0.6420008485469273},
{68.1695790364963301, 0.6769788359809824, 0.6812575211842696},
{95.4102572670689710, 0.6907427371285828, 0.7096243304118508},
{125.3349737466774769, 0.7043709857882867, 0.7273202209896069},
{157.7673742981335181, 0.7112730873636841, 0.7416960712609389},
{192.4852820138941638, 0.7182568256648422, 0.7518309833400817},
{229.3404317780048416, 0.7223134992765637, 0.7605538516997684},
{268.1971889512575444, 0.7264977455887661, 0.7671690223308814},
{308.9273598256032756, 0.7291340483224786, 0.7730658066071221},
{351.4450352512631639, 0.7318943973155260, 0.7777562427941288},
{395.6362805530536093, 0.7337291126187708, 0.7820356847335300} },
{ {12.3877634929254583, 0.3106287789801418, 0.2907684073746467},
{42.6036847835042281, 0.4627877188449809, 0.4127536751978838},
{82.9798171376977933, 0.5026449682305189, 0.4958383305750639},
{131.4096523862202162, 0.5458500246173086, 0.5396690992584521},
{187.7103873356010979, 0.5640772341266093, 0.5739435476234905},
{250.6842250372276339, 0.5826520224568462, 0.5958861537655099},
{320.3146444390156375, 0.5925396795416351, 0.6143816591818130},
{395.9256627133100892, 0.6026322560429110, 0.6275830629811101},
{477.4429023080585921, 0.6087027764662307, 0.6392242748435665},
{564.4365770676146212, 0.6149809196225515, 0.6481020694253140},
{656.8045338092739485, 0.6190440853822474, 0.6561660312304937},
{754.2549282445070276, 0.6232992685237211, 0.6625933243080060},
{856.6732809288554336, 0.6261907693183003, 0.6685510263744304} },
{ {19.8170200230427085, 0.2320877028211801, 0.2163888734146038},
{69.1419870785275634, 0.3604310403078776, 0.3097656741361853},
{136.3065022846630541, 0.3978631815566770, 0.3869697074599593},
{217.7762328116942854, 0.4421189729883522, 0.4288839287575929},
{313.7482180833746384, 0.4620902584083985, 0.4639996187283675},
{421.9561123880703235, 0.4830011363386737, 0.4869970665625544},
{542.7262815797279245, 0.4946409694287579, 0.5070751435012918},
{674.7302096314530218, 0.5066428565734641, 0.5215903100384505},
{818.1051564544559369, 0.5140996795207871, 0.5346850515144980},
{971.9850258797439437, 0.5218366369921346, 0.5447418466558165},
{1136.3989509304888088, 0.5269748921522143, 0.5540322207845707},
{1310.7423821460570252, 0.5323548990820077, 0.5614652819806679},
{1494.9840297450498383, 0.5360919279004233, 0.5684492359599957} },
{ {29.5312842949109431, 0.1800708422081156, 0.1674747883889731},
{104.0798440442586639, 0.2867542112749758, 0.2392322614524005},
{206.2703058363334492, 0.3200209521526838, 0.3074317208351558},
{330.9138607370775276, 0.3617883779855068, 0.3449031222202527},
{478.6759253743640556, 0.3816439290928499, 0.3781199680467794},
{645.9244719619097168, 0.4029411216079677, 0.4002403389448126},
{833.4952365841099891, 0.4152431168703409, 0.4201743351796904},
{1039.1809834825858161, 0.4280709204990331, 0.4347432283351035},
{1263.4688222821805539, 0.4362658143682208, 0.4481679179911077},
{1504.8863238534834181, 0.4448101953396544, 0.4585474902103364},
{1763.7034163950638685, 0.4506144107466040, 0.4682874956847222},
{2038.8711083807743307, 0.4566996269209532, 0.4761123772028160},
{2330.5295789784759108, 0.4610095778522355, 0.4835561595575862} },
{ {41.8136861973061471, 0.1438788789054671, 0.1335904229753444},
{148.4935426265632259, 0.2328252932759561, 0.1897066781454031},
{295.0471797566154919, 0.2618019612168468, 0.2489142206861222},
{474.3666285755288641, 0.2997422383827265, 0.2814986769362187},
{687.6007043804477235, 0.3184956435987003, 0.3117148402799408},
{929.4326059416951011, 0.3390222587039151, 0.3320583117979186},
{1201.3555589890590909, 0.3512352880712853, 0.3509001107707453},
{1500.0382132532092783, 0.3641119759385520, 0.3647852472675001},
{1826.4542672437673900, 0.3725301570939076, 0.3778269855495578},
{2178.3363096544117070, 0.3813585062769190, 0.3879667548508850},
{2556.3084022439338696, 0.3874713776592386, 0.3976198588145461},
{2958.7281860765497186, 0.3938966099192103, 0.4054034515846607},
{3386.0007125931651899, 0.3985236879209418, 0.4128932490794268} },
{ {56.9481099302284974, 0.1176759884724130, 0.1091307074493210},
{203.4420128599603572, 0.1924880058419247, 0.1539096708750006},
{404.7701137411983154, 0.2176038201263679, 0.2051359469932653},
{651.6131901044973347, 0.2514863680262883, 0.2332177076578858},
{945.5716706495318249, 0.2687292056637461, 0.2602085462812856},
{1279.2975081928354939, 0.2879160925912300, 0.2784981704409520},
{1655.0734486080789338, 0.2996014086896870, 0.2958342868689989},
{2068.1791388736860426, 0.3120468379714974, 0.3086837168330236},
{2520.2081154101106222, 0.3203377221636424, 0.3209577684466165},
{3007.8930735478661518, 0.3290848958121833, 0.3305408463924612},
{3532.3273176542261353, 0.3352387047915397, 0.3397835135390562},
{4091.1119695783668249, 0.3417280386321508, 0.3472580354164053},
{4685.0140688407918788, 0.3464671307057797, 0.3545262676580890} },
{ {75.2188488089775689, 0.0980841014998332, 0.0908808682229852},
{269.9779498005294158, 0.1616584642077884, 0.1273105957429412},
{537.5521798682866574, 0.1834661396337166, 0.1717467386836972},
{866.0935981024415469, 0.2135337775229312, 0.1959464455899839},
{1257.5943986041838798, 0.2291753764564736, 0.2198827266724731},
{1702.2989346452843620, 0.2468127040625630, 0.2361560956356012},
{2203.4007463764523891, 0.2577532356454190, 0.2518826489229245},
{2754.5045916917847535, 0.2695089862742797, 0.2635822649563608},
{3357.9537767295637423, 0.2774608982000309, 0.2749225880905490},
{4009.2599713987924588, 0.2858985915487872, 0.2838025494062372},
{4710.1014244090674765, 0.2919132607886716, 0.2924668398517409},
{5457.1461673258909286, 0.2982780579036121, 0.2994886113775405},
{6251.6227529919142398, 0.3029808999951412, 0.3063816961073651} },
{ {96.9104298954369057, 0.0830426068749007, 0.0768918187089398},
{349.1518886924509388, 0.1376218752589488, 0.1070503105544025},
{695.4970626100175650, 0.1566453966155676, 0.1457941514842342},
{1121.2258144698041633, 0.1833044694877618, 0.1667339356822640},
{1628.6452284326435347, 0.1974084909019570, 0.1879315675797662},
{2205.1847232477002763, 0.2134823084085093, 0.2023604447179293},
{2855.0647380311120287, 0.2235987357077935, 0.2165318749187173},
{3569.9071353984568304, 0.2345516689704182, 0.2270970775635313},
{4352.9333080869155310, 0.2420528448373338, 0.2374675333745886},
{5198.1960680895917903, 0.2500539240974030, 0.2456031121975136},
{6108.0749205423853709, 0.2558196030656295, 0.2536224843318571},
{7078.1148401924847349, 0.2619419663800512, 0.2601307870769428},
{8110.1072630598564501, 0.2665100644402784, 0.2665743566930779} },
{ {122.3075207828811273, 0.0712372473055930, 0.0659257956728607},
{442.0137446549898641, 0.1185443571550278, 0.0912792622018234},
{880.7039542613423464, 0.1352354386639880, 0.1252649144852451},
{1420.4152438958999483, 0.1589181857888106, 0.1434910582689507},
{2063.6816985172863497, 0.1716130497076032, 0.1622946673019081},
{2794.6788274097325484, 0.1862058528165014, 0.1750924356744324},
{3618.7694156020406808, 0.1954971116454179, 0.1878333623784575},
{4525.2615871381995021, 0.2056215129289904, 0.1973416633643327},
{5518.3837296895599138, 0.2126255495578252, 0.2067761011111363},
{6590.4747915988273235, 0.2201311342875023, 0.2141849547081120},
{7744.7332457474440162, 0.2255885202295314, 0.2215535216480715},
{8975.3774459810865665, 0.2314023236088087, 0.2275385138087779},
{10284.8650184882135363, 0.2357757346093869, 0.2335089617845300} },
{ {151.6948774599734691, 0.0617975562589851, 0.0571656321362580},
{549.6133942612838155, 0.1031615285236105, 0.0787680399169203},
{1095.2698878971164049, 0.1178955254683319, 0.1087662974545515},
{1767.0601282992686265, 0.1390046470023764, 0.1247343579008182},
{2567.6494496323043677, 0.1504380655850873, 0.1414684489199227},
{3477.4883287300490338, 0.1636731585314562, 0.1528471272815010},
{4503.1999161560934226, 0.1721789590878561, 0.1643052865622467},
{5631.4244264603339616, 0.1814979921290295, 0.1728582548840288},
{6867.5288207439980397, 0.1879984000081352, 0.1814239315935911},
{8201.8656278686921723, 0.1949927448681955, 0.1881532885990316},
{9638.5724969754955964, 0.2001164572538374, 0.1948985125806706},
{11170.3237658565230959, 0.2055910034120608, 0.2003790894036274},
{12800.3489851758786244, 0.2097374778855530, 0.2058831140472483} },
{ {185.3573137316421366, 0.0541281886827010, 0.0500538231598886},
{673.0008813467775326, 0.0905831155136695, 0.0686775786797703},
{1341.2908760962166070, 0.1036676363906801, 0.0953175831241843},
{2164.5545977242309164, 0.1225580204655849, 0.1093995914560965},
{3145.4866177751359828, 0.1328754290088939, 0.1243514848497630},
{4260.3086954786658680, 0.1448875379774094, 0.1345041520006953},
{5517.0271145727692783, 0.1526662752783804, 0.1448270704124531},
{6899.2363107623086762, 0.1612280500889685, 0.1525304058585384},
{8413.5777708542354958, 0.1672411654869746, 0.1603071170754176},
{10048.1272010287448211, 0.1737342299488308, 0.1664164467909535},
{11808.0852387413324323, 0.1785204362666905, 0.1725821062833292},
{13684.3507667774247238, 0.1836480423534323, 0.1775917417414314},
{15681.0340767236193642, 0.1875541802537351, 0.1826527307714005} },
{ {223.5796825113918089, 0.0478105923417549, 0.0441991902302635},
{813.2264731794883801, 0.0801694909477906, 0.0604210059203954},
{1620.8624767337623780, 0.0918557661716478, 0.0842153609044803},
{2616.2904104631652444, 0.1088321878922960, 0.0967130365452483},
{3802.1265906235412331, 0.1181674914081234, 0.1101298912758963},
{5149.8274831212529534, 0.1290881435143835, 0.1192253136773197},
{6668.9114379877291867, 0.1362044909265472, 0.1285498291662753},
{8339.5252483495769411, 0.1440678913078618, 0.1355037755539340},
{10169.7265501144211157, 0.1496220196610692, 0.1425726455217940},
{12145.0057350513470737, 0.1556380383854588, 0.1481240287896342},
{14271.7548149001886486, 0.1600958694268372, 0.1537601522045490},
{16538.8515503874150454, 0.1648830765147783, 0.1583383407528649},
{18951.3995406033463951, 0.1685477344331627, 0.1629877848943941} },
{ {266.6468639453826199, 0.0425434065123545, 0.0393205299225566},
{971.3406582765993562, 0.0714524122205838, 0.0535785707421482},
{1936.0800762202395617, 0.0819459799629488, 0.0749460417677939},
{3125.6579570928065550, 0.0972670941925269, 0.0861040867252461},
{4542.4997349945251699, 0.1057395317842086, 0.0981956572082977},
{6152.7268658953880731, 0.1156904323024538, 0.1063783898147514},
{7967.5057709266757229, 0.1222087047753072, 0.1148268062910432},
{9963.1095036674305447, 0.1294353340426643, 0.1211221561399219},
{12149.1600913334514189, 0.1345641206439353, 0.1275600522152612},
{14508.2358317787275155, 0.1401343975871569, 0.1326131883248751},
{17048.0538609087489021, 0.1442802531836559, 0.1377704049527455},
{19755.2107639954665501, 0.1487418752414043, 0.1419577495766212},
{22635.9203672657167772, 0.1521715081306064, 0.1462300661108666} },
{ {314.8437576260822084, 0.0381050845746589, 0.0352113849108667},
{1148.3941235452341516, 0.0640831567610713, 0.0478439334179616},
{2289.0390301422617085, 0.0735528970814256, 0.0671282953935429},
{3696.0468464552059231, 0.0874372174765972, 0.0771458870135911},
{5371.5344848354261558, 0.0951511346758609, 0.0880892038543141},
{7275.6853494569695613, 0.1042417447269752, 0.0954818811883522},
{9421.4575806783614098, 0.1102225878147597, 0.1031613849511636},
{11780.7999713055996835, 0.1168724106871099, 0.1088783667360946},
{14365.0544764786627638, 0.1216109363843679, 0.1147554608903083},
{17153.5420678430855332, 0.1267694505254819, 0.1193653454268653},
{20155.4447348273570242, 0.1306233458952027, 0.1240919174685572},
{23354.8033122850283689, 0.1347785706353585, 0.1279274801777594},
{26759.0636345696802891, 0.1379840577302418, 0.1318570358154138} },
{ {368.4552773411048747, 0.0343297438659499, 0.0317173394178395},
{1345.4377262854256969, 0.0577979115917244, 0.0429894199503936},
{2681.8347307360181730, 0.0663833578363781, 0.0604745605156713},
{4330.8462508688762682, 0.0790153853036561, 0.0695147061682297},
{6294.1580318663800426, 0.0860612857184611, 0.0794590196023597},
{8525.3789269993394555, 0.0943885233895954, 0.0861652995023599},
{11039.4104432065269066, 0.0998874699162157, 0.0931688651184854},
{13803.4020165281235677, 0.1060166304670258, 0.0983774288885883},
{16830.5788355083714123, 0.1103992097347536, 0.1037564540258385},
{20096.6407248136747512, 0.1151800301583455, 0.1079725367103692},
{23612.3807031948927033, 0.1187633203989098, 0.1123129647457130},
{27358.9946438369006501, 0.1226332254557084, 0.1158329708980787},
{31345.2874427433525852, 0.1256277672728281, 0.1194524742339318} },
{ {427.7663474421634078, 0.0310911458811381, 0.0287209896847480},
{1563.5224672079855281, 0.0523942307803561, 0.0388430241053696},
{3116.5626365178577544, 0.0602113759778913, 0.0547648672545904},
{5033.4451106868700663, 0.0717471632677693, 0.0629618189827235},
{7315.2967678315681042, 0.0782032304553281, 0.0720331409281820},
{9908.4818618064127804, 0.0858521790892524, 0.0781406215145868},
{12830.0051304136450199, 0.0909192249228215, 0.0845485011977978},
{16041.7168652680302330, 0.0965791160193964, 0.0893092141630443},
{19558.8968926529341843, 0.1006381348017950, 0.0942455886760228},
{23353.2413705569233571, 0.1050739733739834, 0.0981117145454128},
{27437.3073176009420422, 0.1084079828672216, 0.1021062366550552},
{31789.1417013984100777, 0.1120139999077900, 0.1053435627567753},
{36419.0411296295205830, 0.1148117600946166, 0.1086832852880103} },
{ {493.0619002785167595, 0.0282918186891413, 0.0261317565647541},
{1803.6994666175241946, 0.0477147282707190, 0.0352728576520064},
{3595.3182824071973300, 0.0548606033996758, 0.0498287303585405},
{5807.2322556083809104, 0.0654325428671166, 0.0572937968473220},
{8439.8765701436877862, 0.0713662436338310, 0.0655988289155130},
{11431.6672197528205288, 0.0784112779904983, 0.0711816708297666},
{14801.8803838487056055, 0.0830909910614900, 0.0770630204768566},
{18506.5426406780534307, 0.0883280234357182, 0.0814281672523688},
{22563.1681851284047298, 0.0920933992104451, 0.0859705283576591},
{26939.0482010155901662, 0.0962148648943143, 0.0895252654848743},
{31648.6637290193903027, 0.0993200990270860, 0.0932098435720548},
{36666.5940745981279179, 0.1026830997739634, 0.0961939410824736},
{42004.7660544968166505, 0.1052983524442228, 0.0992814250118996} },
{ {564.6268743474294070, 0.0258555154652576, 0.0238788329819462},
{2067.0199442148623348, 0.0436355890874899, 0.0321764282556087},
{4120.1972800072890095, 0.0501918691529003, 0.0455324202357887},
{6655.5964761802524663, 0.0599127034828274, 0.0523585081838062},
{9672.8229877666981338, 0.0653822973397929, 0.0599879384374331},
{13101.6072340277187322, 0.0718883245461903, 0.0651091154347976},
{16963.6734666101037874, 0.0762202128089836, 0.0705235497075416},
{21208.6751321457304584, 0.0810759588245186, 0.0745382383552377},
{25856.5490075254310796, 0.0845749642397731, 0.0787291351708788},
{30869.7611361621529795, 0.0884102416261373, 0.0820062686108899},
{36264.8838427750160918, 0.0913059668633249, 0.0854128239074570},
{42012.6951354302218533, 0.0944457503651237, 0.0881698721901277},
{48126.8965768483103602, 0.0968923644124381, 0.0910299138734218} },
{ {642.7462129366363115, 0.0237218872784061, 0.0219062083885272},
{2354.5352023047908006, 0.0400583465287400, 0.0294731116335155},
{4693.2953131724516425, 0.0460941973550566, 0.0417698761496649},
{7581.9265652276671972, 0.0550603343042543, 0.0480350431214596},
{11019.0613629358067556, 0.0601162207223734, 0.0550662711668577},
{14924.9735566902818391, 0.0661398871344551, 0.0597794559343367},
{19324.0205587694654241, 0.0701588732785602, 0.0647784471266522},
{24158.9083655262838874, 0.0746703965680046, 0.0684816312501756},
{29452.1931376378197456, 0.0779276884062984, 0.0723582472597962},
{35161.0766970230542938, 0.0815024662488301, 0.0753873207114739},
{41304.3972912025710684, 0.0842065190666183, 0.0785440819875775},
{47848.7830630958196707, 0.0871415536566175, 0.0810972499159101},
{54809.8610437634852133, 0.0894326983646647, 0.0837520309000239} },
{ {727.7048631122434017, 0.0218426559811228, 0.0201690982010269},
{2667.2966119727111618, 0.0369039102709237, 0.0270987780583880},
{5316.7081315846044163, 0.0424782451759210, 0.0384561253479440},
{8589.6113410571051645, 0.0507724798585374, 0.0442263694961223},
{12483.5169111886061728, 0.0554583670450918, 0.0507257426208108},
{16908.4374333777413995, 0.0610491643237888, 0.0550768672756728},
{21891.5570423091703560, 0.0647860850094723, 0.0597049635259508},
{27368.0350268474685436, 0.0689863480793306, 0.0631303469708332},
{33363.2523925948771648, 0.0720241058178668, 0.0667251875070678},
{39828.6886999445050606, 0.0753616473471952, 0.0695320319572370},
{46785.6302338390451041, 0.0778903821647178, 0.0724639202322267},
{54196.1917282161302865, 0.0806377060026750, 0.0748336680470846},
{62078.0827005395185552, 0.0827856998137152, 0.0773029574714503} },
{ {819.7877749573789288, 0.0201788224092726, 0.0186313416232821},
{3006.3556017431860710, 0.0341081701551950, 0.0250018978157000},
{5992.5315437592389571, 0.0392714480527651, 0.0355224525111586},
{9682.0396595970687486, 0.0469651938636953, 0.0408539141748041},
{14071.1147740743726899, 0.0513190967919396, 0.0468785572531754},
{19058.6698254368857306, 0.0565203434711216, 0.0509070968575941},
{24674.9177076978776313, 0.0600024306647516, 0.0552029640088489},
{30846.8467784055865195, 0.0639207203644892, 0.0583797826858159},
{37602.8770551692941808, 0.0667588371932166, 0.0617212991328429},
{44888.2888033145718509, 0.0698801257157417, 0.0643285199827421},
{52727.0060072959313402, 0.0722484444964877, 0.0670575288705657},
{61076.2514367898984347, 0.0748236622915831, 0.0692619128505009},
{69955.9804924282361753, 0.0768399125470796, 0.0715632968609608} },
{ {919.2799009830367822, 0.0186986012533090, 0.0172634783896751},
{3372.7636482796524433, 0.0316187191248772, 0.0231406787309002},
{6722.8614102296660349, 0.0364143853773969, 0.0329128092617191},
{10862.6004198310729407, 0.0435695029121111, 0.0378535185289742},
{15786.7800538228439109, 0.0476245881439549, 0.0434528315381855},
{21382.3414957880158909, 0.0524742846785230, 0.0471928575025943},
{27682.7369046812236775, 0.0557256090662403, 0.0511901611621920},
{34606.1344956255634315, 0.0593879478563983, 0.0541438494851508},
{42184.2162011304098996, 0.0620442406891360, 0.0572569911203904},
{50355.5669375278157531, 0.0649681601856481, 0.0596844028759875},
{59146.9456493744364707, 0.0671895874912055, 0.0622299541341480},
{68510.2895471674564760, 0.0696069232539241, 0.0642849147840650},
{78467.9697522282513091, 0.0715019223625734, 0.0664340279941316} },
{ {1026.4661956770801226, 0.0173758738748023, 0.0160413089321668},
{3767.5722687625998333, 0.0293923818874320, 0.0214809349133025},
{7509.7936371981122647, 0.0338580297031990, 0.0305811140599379},
{12134.6825652754450857, 0.0405283289891881, 0.0351723864978798},
{17635.4378360087066540, 0.0443136266254466, 0.0403892721456200},
{23886.1230696350758080, 0.0488451921638259, 0.0438703169739037},
{30923.6486530146539735, 0.0518870610153829, 0.0475984652955274},
{38656.6884456566185690, 0.0553165882549789, 0.0503512199124127},
{47120.4179516922667972, 0.0578070077107881, 0.0532579141274545},
{56246.2116433115297696, 0.0605505345191437, 0.0555229209945919},
{66063.8683206002024235, 0.0626373154078007, 0.0579021864552102},
{76519.6309775865520351, 0.0649096956575254, 0.0598218113714890},
{87638.4627809512458043, 0.0666930510119224, 0.0618325570552846} },
{ {1141.6316151483486010, 0.0161890155313199, 0.0149448036898741},
{4191.8330146007720032, 0.0273933310097857, 0.0199944808847448},
{8355.4241708430345170, 0.0315616431519721, 0.0284891994868067},
{13501.6750830989731185, 0.0377941210997488, 0.0327667578941339},
{19622.0132042150435154, 0.0413351236874491, 0.0376386308116638},
{26576.6850775069033261, 0.0455780269858706, 0.0408863992162874},
{34406.2867243135406170, 0.0484293354924532, 0.0443711669233467},
{43009.2984226606931770, 0.0516466522775428, 0.0469424216887325},
{52424.6296688488509972, 0.0539854823551687, 0.0496619879617407},
{62575.9103386015340220, 0.0565638741868670, 0.0517799132753613},
{73496.1916429858247284, 0.0585270810787012, 0.0540081007592163},
{85125.5986219112965045, 0.0606662324342316, 0.0558048592038981},
{97491.8693337025615619, 0.0623467153247113, 0.0576896118040035} },
{ {1265.0611168450386685, 0.0151199967777079, 0.0139572678915673},
{4646.5974662622447795, 0.0255916363565643, 0.0186579081445657},
{9261.8489922666649363, 0.0294911532053183, 0.0266052357971632},
{14966.9670019808891084, 0.0353270164842354, 0.0306001174157622},
{21751.4312493002726114, 0.0386461828053464, 0.0351597366435510},
{29460.6979859271486930, 0.0426264807292204, 0.0381966919108746},
{38139.2847029436452431, 0.0453040192193744, 0.0414607443293761},
{47674.7538510057056556, 0.0483274954908500, 0.0438675698790877},
{58109.9981074554816587, 0.0505275378978457, 0.0464170746983378},
{69360.3495297038898570, 0.0529545127041327, 0.0484014441233399},
{81462.3319727101043100, 0.0548041529056693, 0.0504920483347312},
{94349.5136898587661562, 0.0568207041480439, 0.0521769977287561},
{108052.5970243442861829, 0.0584063081908614, 0.0539467852648813} },
{ {1397.0396593321568162, 0.0141536880566103, 0.0130646955328188},
{5132.9172289677653680, 0.0239621377216407, 0.0174516435000450},
{10231.1641130795960635, 0.0276178873022303, 0.0249025087343499},
{16533.9473893239664903, 0.0330934010739342, 0.0286418034674000},
{24028.6170750950986985, 0.0362105803852604, 0.0329179618082551},
{32544.8322192381274363, 0.0399513769258372, 0.0357638109036881},
{42131.2760315461564460, 0.0424700978992827, 0.0388271444060106},
{52663.8438643787594629, 0.0453161432970517, 0.0410845843217675},
{64189.6695343154424336, 0.0473888844823025, 0.0434791446151295},
{76615.2149790407274850, 0.0496767855040336, 0.0453419286772216},
{89980.7046199757169234, 0.0514219046187965, 0.0473069497335429},
{104212.6959853049047524, 0.0533254872920125, 0.0488899158835418},
{119345.0516617187968222, 0.0548234903762353, 0.0505545820958989} },
{ {1537.8522021080368631, 0.0132773165437927, 0.0122552650315339},
{5651.8439291200747903, 0.0224835609823517, 0.0163592173381711},
{11265.4655715687604243, 0.0259175786375626, 0.0233584635854159},
{18206.0053481471550185, 0.0310647736111558, 0.0268659182938475},
{26458.4958016535383649, 0.0339975635575886, 0.0308840154534368},
{35835.7581752499172580, 0.0375194010903752, 0.0335561118330333},
{46390.8940452541282866, 0.0398926505279772, 0.0364364246605853},
{57987.3573668251337949, 0.0425759518132984, 0.0385577786070061},
{70676.7898211261199322, 0.0445317125196761, 0.0408108208592676},
{84356.1918390142964199, 0.0466916581430127, 0.0425626422705530},
{99069.7240270901966142, 0.0483404368164423, 0.0444127751910307},
{114736.4641343361727195, 0.0501397809799055, 0.0459025087684294},
{131393.6375288700219244, 0.0515568070346872, 0.0474708560261761} },
{ {1687.7837054628541864, 0.0124800382574051, 0.0115189420795963},
{6204.4292113208230148, 0.0211378195091436, 0.0153666899690363},
{12366.8494293315325194, 0.0243695788419417, 0.0219539512695356},
{19986.5300140397739597, 0.0292168427029565, 0.0252504672975281},
{29045.9925669063559326, 0.0319808919832529, 0.0290329883779533},
{39340.1462362107049557, 0.0353020850638741, 0.0315466677699121},
{50926.7719976787921041, 0.0375418024478170, 0.0342596729520706},
{63656.0830800941039342, 0.0400755308501930, 0.0362567355921295},
{77584.5045188820222393, 0.0419235989709190, 0.0383802171380157},
{92598.9647593507834245, 0.0439656173436004, 0.0400305279510681},
{108747.8039126591174863, 0.0455254601700709, 0.0417753265593188},
{125942.1357727541617351, 0.0472284835338558, 0.0431796385520412},
{144222.7576150253007654, 0.0485705619220481, 0.0446595531301110} },
{ {1847.1191303535092629, 0.0117525982643948, 0.0108471643833632},
{6791.7247358796939807, 0.0199094577756930, 0.0144621981563552},
{13537.4117684554585139, 0.0229562298497461, 0.0206726291343951},
{21878.9105521376222896, 0.0275288039922840, 0.0237766742025718},
{31796.0325272568406945, 0.0301380689903553, 0.0273435908399976},
{43064.6667767788967467, 0.0332749895520092, 0.0297124522682739},
{55747.5430802037008107, 0.0353918807368043, 0.0322721423798155},
{69680.8095805501798168, 0.0377878726641854, 0.0341554054465338},
{84925.9589163650525734, 0.0395366206699099, 0.0361600036309277},
{101359.2179732814547606, 0.0414697697738504, 0.0377172366911079},
{119033.3573896640737075, 0.0429473860318361, 0.0393652552347334},
{137851.0277001391514204, 0.0445612757179048, 0.0406911340246593},
{157856.8138087735569570, 0.0458338970398849, 0.0420896955644781} },
{ {2016.1434383079772488, 0.0110870588221325, 0.0102325895219230},
{7414.7821766754541386, 0.0187852050423924, 0.0136355938702387},
{14779.2486889257488656, 0.0216623592604727, 0.0195004812365351},
{23886.5361543125000026, 0.0259827576708479, 0.0224284322547222},
{34713.5408574936882360, 0.0284497206804627, 0.0257975402932391},
{47015.9901694170839619, 0.0314170421750654, 0.0280336822182986},
{60861.8404375036116107, 0.0334207288395743, 0.0304525540348258},
{76072.3253281352808699, 0.0356896432344881, 0.0322313776950629},
{92714.2980871436884627, 0.0373466316344932, 0.0341266518045530},
{110652.6353676293947501, 0.0391791058878906, 0.0355983503838610},
{129944.7970614879595814, 0.0405805822646087, 0.0371572658898327},
{150484.4560075211629737, 0.0421118690556258, 0.0384109784092390},
{172320.2070592873496935, 0.0433200367743912, 0.0397345553883447} },
{ {2195.1415913416831245, 0.0104765803456226, 0.0096688918642192},
{8074.6532194097389947, 0.0177536149218722, 0.0128781544112163},
{16094.4563065730762901, 0.0204748722405288, 0.0184254316421959},
{26012.7960365781982546, 0.0245632361345344, 0.0211918614807711},
{37803.4427502258695313, 0.0268990914814368, 0.0243790663742797},
{51200.7867882536666002, 0.0297099983336953, 0.0264932859113267},
{66278.2971785103436559, 0.0316091473028648, 0.0287825316144995},
{82841.4186891270219348, 0.0337606026550651, 0.0304652902069024},
{100962.6669273369916482, 0.0353326709877628, 0.0322598203512366},
{120494.9005395875865361, 0.0370718956073033, 0.0336527492911292},
{141500.5351013401232194, 0.0384027613489745, 0.0351294673421417},
{163863.7361830777372234, 0.0398573866525269, 0.0363166465084999},
{187637.3375110671913717, 0.0410056643475120, 0.0375709794454057} },
{ {2384.3985519004063462, 0.0099152437727032, 0.0091505979178412},
{8772.3895600766390999, 0.0168047724520824, 0.0121823482168927},
{17485.1307511183586030, 0.0193824194448445, 0.0174370306992947},
{28261.0794367085472913, 0.0232568186754547, 0.0200549492457936},
{41070.6634150905520073, 0.0254716319437710, 0.0230745082335975},
{55625.7270117730804486, 0.0281379996754142, 0.0250764698820660},
{72005.5463853747351095, 0.0299404349702665, 0.0272461403000052},
{89998.8779542866104748, 0.0319831286806987, 0.0288403466485121},
{109684.2101852764026262, 0.0334764754677947, 0.0305418531286091},
{130901.6968430956330849, 0.0351291898500922, 0.0318620943188032},
{153718.9833174605155364, 0.0363944749136015, 0.0332628405989660},
{178010.1832000505819451, 0.0377778508723554, 0.0343885609606219},
{203832.6046169881883543, 0.0388704051721738, 0.0355788349326693} },
{ {2584.1992827818876322, 0.0093979056108090, 0.0086729520058697},
{9509.0429036682198785, 0.0159300546076863, 0.0115416444394270},
{18953.3681646323711902, 0.0183751252110323, 0.0165261989579228},
{30634.7756120838203060, 0.0220518153997170, 0.0190072566885858},
{44520.1280778635773459, 0.0241546600674235, 0.0218719850819488},
{60297.4812244721542811, 0.0266872105858756, 0.0237703641816660},
{78052.2211202035978204, 0.0284000106660054, 0.0258295085902110},
{97555.4913531511847395, 0.0303418231369808, 0.0273419203655098},
{118892.0724871871643700, 0.0317620761296450, 0.0289573665078983},
{141888.7074268357828259, 0.0333344074652132, 0.0302104010898576},
{166618.5532072892237920, 0.0345386943005503, 0.0315408007309073},
{192945.1115903863392305, 0.0358557575790061, 0.0326096440016009},
{220930.4072332477953751, 0.0368963969667631, 0.0337405518844902} },
{ {2794.8287471068056220, 0.0089200789620250, 0.0082318060441649},
{10285.6649631027066789, 0.0151219333940995, 0.0109503571761219},
{20501.2646999934513588, 0.0174443638509185, 0.0156850169451940},
{33137.2738377251444035, 0.0209380065475901, 0.0180396776022277},
{48156.7619793052217574, 0.0229370815936933, 0.0207611251552524},
{65222.7198179418483051, 0.0253455174138070, 0.0225637303115912},
{84426.9544303276052233, 0.0269750997227400, 0.0245205165353976},
{105522.0470655953395180, 0.0288231852460945, 0.0259572275348727},
{128599.3983563978981692, 0.0301754631638774, 0.0274929084991873},
{153471.6152657127531711, 0.0316729914138465, 0.0286836878066069},
{180217.6560027346713468, 0.0328204610104094, 0.0299488342793444},
{208689.8355061278562061, 0.0340757208254047, 0.0309649462056485},
{238955.1437010442896280, 0.0350679305875990, 0.0320407438767353} },
{ {3016.5719082713044372, 0.0084778353330798, 0.0078235285952142},
{11103.3074582525623555, 0.0143738130866430, 0.0104035173208677},
{22130.9165198999144195, 0.0165825745605347, 0.0149065516444061},
{35771.9634046260107425, 0.0199064264127071, 0.0171442393271584},
{51985.4903742501555826, 0.0218091578511859, 0.0197328415740458},
{70408.1131915516889421, 0.0241022784093322, 0.0214467195153795},
{91138.3793521696934476, 0.0256544730260887, 0.0233085374227412},
{113909.3332308727840427, 0.0274153392637924, 0.0246750561462230},
{138819.3322306268673856, 0.0287043060978212, 0.0261366758111113},
{165666.1031868454592768, 0.0301321213613601, 0.0272696827409062},
{194534.7027072767959908, 0.0312265941658252, 0.0284741977831048},
{225265.6687712442653719, 0.0324241751246619, 0.0294413375820056},
{257931.2119115964160301, 0.0333711487506644, 0.0304658921701363} },
{ {3249.7137299115120186, 0.0080677231774448, 0.0074449294350316},
{11963.0221150565284916, 0.0136798950124909, 0.0098967665688518},
{23844.4197954752162332, 0.0157831075184671, 0.0141847125320734},
{38542.2336180066122324, 0.0189491833663413, 0.0163139375002133},
{56011.2385305989664630, 0.0207623121629135, 0.0187791460635278},
{75860.3317526647151681, 0.0229481148749694, 0.0204106717678725},
{98195.1289144212059909, 0.0244282288072905, 0.0221842227262043},
{122728.1379551694262773, 0.0261078064130229, 0.0234855402197312},
{149565.0184753476351034, 0.0273377192381748, 0.0248782779165831},
{178487.8538911899959203, 0.0287004724341817, 0.0259575801534718},
{209588.1041283316735644, 0.0297454447043507, 0.0271056660062554},
{242693.9249256253242493, 0.0308891249899617, 0.0280272494281639},
{277883.0093675324460492, 0.0317937923389241, 0.0290040815493302} },
{ {3494.5391758863725045, 0.0076866999853258, 0.0070931966783493},
{12865.8606650267993246, 0.0130350646750262, 0.0094262692988120},
{25643.8707056228486181, 0.0150400953112952, 0.0135141315549285},
{41451.4737961539576645, 0.0180593092646630, 0.0155425982393014},
{60238.9317280947361724, 0.0197889676833607, 0.0178929934089757},
{81586.0459168017550837, 0.0218747359744530, 0.0194479478327196},
{105605.8361405415489571, 0.0232876094022776, 0.0211393222583334},
{131989.2493176174466498, 0.0248913131161006, 0.0223799708646795},
{160849.6013930752524175, 0.0260660642403523, 0.0237085394475707},
{191952.5499712091113906, 0.0273680119219254, 0.0247378357399681},
{225396.2709025266813114, 0.0283666880349001, 0.0258333207756483},
{260995.9172617766307667, 0.0294599334394882, 0.0267124576928512},
{298834.9332295521162450, 0.0303249859848152, 0.0276447784826510} },
{ {3751.3332102231706813, 0.0073320754037944, 0.0067658441252840},
{13812.8748444352149818, 0.0124347970984910, 0.0089886389629490},
{27531.3654360608707066, 0.0143483450350481, 0.0128900626017251},
{44503.0732690804798040, 0.0172306328915502, 0.0148247626813829},
{64673.4952578663287568, 0.0188824109809270, 0.0170681509899235},
{87591.9261073121015215, 0.0208747911625915, 0.0185517883253544},
{113379.1340502850944176, 0.0222248467452476, 0.0201665331073976},
{141703.4553753731888719, 0.0237576291053446, 0.0213506374904217},
{172686.2252354406227823, 0.0248807832792434, 0.0226193339937989},
{206075.8739267612108961, 0.0261258273045977, 0.0236019944894945},
{241977.6135203580197413, 0.0270811484842948, 0.0246483731527133},
{280192.9588567685568705, 0.0281271427506126, 0.0254879004387208},
{320811.3803602691623382, 0.0289550561957765, 0.0263786440799096} },
{ {4020.3807971291344074, 0.0070014633842251, 0.0064606669720902},
{14805.1163938570080063, 0.0118750771083475, 0.0085808763134666},
{29509.0001786690554582, 0.0137032473602226, 0.0123082969319811},
{47700.4213770301066688, 0.0164576731572110, 0.0141555898253092},
{69319.8544210228428710, 0.0180366768090338, 0.0162990888836283},
{93884.6427557454444468, 0.0199417453846642, 0.0177161949390200},
{121523.6556614688888658, 0.0212330315809352, 0.0192593722249212},
{151881.5441665671241935, 0.0226994302349635, 0.0203906938089671},
{185088.0342095076339319, 0.0237742575493001, 0.0216034437508267},
{220873.5081768072850537, 0.0249659802429615, 0.0225425452425865},
{259350.5423429535585456, 0.0258806501248589, 0.0235430130891981},
{300306.3625994283938780, 0.0268823220100385, 0.0243455234412272},
{343836.7473588153370656, 0.0276753765500538, 0.0251973757868755} },
{ {4301.9669008943292283, 0.0066927417545957, 0.0061757043994596},
{15843.6370577033412701, 0.0113523319124849, 0.0082003173369198},
{31578.8711307066041627, 0.0131006995801029, 0.0117650917311969},
{51046.9074704086015117, 0.0157355486094901, 0.0135307744367603},
{74182.9345282300055260, 0.0172464503939940, 0.0155808869146804},
{100470.8663005234266166, 0.0190697731302098, 0.0169358299451978},
{130048.0339907046436565, 0.0203060023398310, 0.0184120685348046},
{162534.3037150590098463, 0.0217101817989609, 0.0194940443157317},
{198068.1724839085363783, 0.0227396868170592, 0.0206544405383736},
{236361.1350713041902054, 0.0238813821766189, 0.0215527973375590},
{277533.4676198944216594, 0.0247578895791323, 0.0225102818400048},
{321357.4412117621395737, 0.0257179370134849, 0.0232781491001500},
{367935.4305933626601472, 0.0264782354913051, 0.0240935729082673} },
{ {4596.3764861783847664, 0.0064040179275084, 0.0059092078433459},
{16929.4885836452704098, 0.0108633738613882, 0.0078445891890102},
{33743.0744947039711406, 0.0125370402399242, 0.0112571095125862},
{54545.9209077779378276, 0.0150599004762526, 0.0129464774055040},
{79267.6608986076171277, 0.0165069842680431, 0.0149091557324208},
{107357.2671880824345862, 0.0182536681633278, 0.0162059308270978},
{138960.9020545219245832, 0.0194382503810957, 0.0176194712253427},
{173672.5220321356318891, 0.0207840389414396, 0.0186552477572497},
{211639.7841947849374264, 0.0217709865383179, 0.0197665845614711},
{252554.4368987262714654, 0.0228656879715508, 0.0206267756079749},
{296544.7995020904345438, 0.0237063272024048, 0.0215439632969706},
{343367.5072695119306445, 0.0246272388739616, 0.0222793647450644},
{393131.8262272624997422, 0.0253567230596756, 0.0230606219686982} },
{ {4903.8945174044411033, 0.0061335997012684, 0.0056596139822481},
{18063.7227231342694722, 0.0104053516676990, 0.0075115727521639},
{36003.7064771216537338, 0.0120089933965770, 0.0107813665240566},
{58200.8510557293338934, 0.0144268269762260, 0.0123992664420365},
{84578.9588594014348928, 0.0158140272278020, 0.0142799695497122},
{114550.5158710375544615, 0.0174887663407250, 0.0155222375025193},
{148270.8928701482072938, 0.0186248389135794, 0.0168769715139174},
{185306.9871184403891675, 0.0199157613671099, 0.0178694347489131},
{225816.0134504287270829, 0.0208626996871337, 0.0189347379626063},
{269469.0958960776915774, 0.0219132047024138, 0.0197591306855754},
{316402.9480544547550380, 0.0227200936877248, 0.0206384911154787},
{366357.8732255701324902, 0.0236041683473485, 0.0213434271381288},
{419450.3302423906861804, 0.0243046335441634, 0.0220925986524741} },
{ {5224.8059593121543003, 0.0058799703071395, 0.0054255216549249},
{19247.3912293562716513, 0.0099757086856067, 0.0071993707019351},
{38362.8632890303269960, 0.0115136209220231, 0.0103351886613284},
{62015.0872882339026546, 0.0138328270544411, 0.0118860653957834},
{90121.7537455182027770, 0.0151637634443034, 0.0136898086148597},
{122057.2828084746142849, 0.0167708793985056, 0.0148809300582089},
{157986.6394550150434952, 0.0178613333920022, 0.0161804356728738},
{197448.4869665593141690, 0.0191006400631024, 0.0171322372256517},
{240610.0043346408638172, 0.0200099209513853, 0.0181542907525042},
{287120.7942533832392655, 0.0210188131692935, 0.0189450621212199},
{337126.3232652564183809, 0.0217939096602288, 0.0197888690748775},
{390349.8514093228732236, 0.0226432734072093, 0.0204651805568118},
{446915.3384585696039721, 0.0233163815673392, 0.0211841836505116} },
{ {5559.3957767072552087, 0.0056417670105762, 0.0052056720688226},
{20481.5458586926797580, 0.0095721471063276, 0.0069062801730144},
{40822.6411446169659030, 0.0110482815508672, 0.0099161736647353},
{65992.0189852118055569, 0.0132747520298070, 0.0114041107916718},
{95900.9708970903884619, 0.0145527601014201, 0.0131355098424362},
{129884.2384663953125710, 0.0160962379663341, 0.0142785752989119},
{168116.7748282141692471, 0.0171437415736532, 0.0155261475035072},
{210107.8095621225074865, 0.0183344341431273, 0.0164397278262340},
{256034.9009096857334953, 0.0192082313595871, 0.0174210971395161},
{305525.2141195565927774, 0.0201779001673893, 0.0181802522764777},
{358733.3350563356070779, 0.0209230162469207, 0.0189906025669799},
{415364.7540552458376624, 0.0217396380251219, 0.0196399862996963},
{475551.2465535091469064, 0.0223869295381950, 0.0203305902142465} },
{ {5907.9489344770699972, 0.0054177626993479, 0.0049989317720432},
{21767.2383695857315615, 0.0091925971297601, 0.0066307692821572},
{43385.1362616527912905, 0.0106105956040502, 0.0095221585956183},
{70135.0355332488252316, 0.0127497639082413, 0.0109509144333177},
{101921.5356637958175270, 0.0139779222265763, 0.0126142243050275},
{138038.0533153786382172, 0.0154614423703333, 0.0137120807143290},
{178669.9320103445788845, 0.0164684617379581, 0.0149107587680112},
{223295.7428849493153393, 0.0176133162528201, 0.0157883676475658},
{272103.8472206967417151, 0.0184536417366963, 0.0167314206255593},
{324698.0376093981903978, 0.0193862998653672, 0.0174608093011274},
{381242.3932893764576875, 0.0201031149518641, 0.0182396394766183},
{441423.8933061584830284, 0.0208888204588457, 0.0188636618390589},
{505382.4500756770721637, 0.0215117247588108, 0.0195275015995969} },
{ {6270.7503975950739914, 0.0052068499907754, 0.0048042779575633},
{23105.5205224829005601, 0.0088351903367134, 0.0063714568979794},
{46052.4448607778031146, 0.0101984145073712, 0.0091511917662875},
{74447.5263238662009826, 0.0122552993342916, 0.0105242311242537},
{108188.3733965316059766, 0.0134364536087498, 0.0121233805200818},
{146525.3978320941387210, 0.0148634200349523, 0.0131786547104425},
{189654.7440234921523370, 0.0158322378256925, 0.0143312463472015},
{237023.0749097313964739, 0.0169338252396292, 0.0151749610754439},
{288829.9872966678231023, 0.0177425436578192, 0.0160818865193025},
{344654.9468054473982193, 0.0186402429263110, 0.0167832178000979},
{404671.9077732103178278, 0.0193303154450215, 0.0175323190151550},
{468548.5812248337897472, 0.0200867996327422, 0.0181324281441942},
{536433.3444589017890394, 0.0206866447516468, 0.0187710168925334} },
{ {6648.0851310435809864, 0.0050080274717814, 0.0046207857401579},
{24497.4440797774404928, 0.0084982366212026, 0.0061270951524918},
{48826.6631654909433564, 0.0098097943741602, 0.0088015084392467},
{78932.8807532335777069, 0.0117890383256927, 0.0101220307215629},
{114706.4094548283901531, 0.0129258228842891, 0.0116606526455718},
{155352.9424982259515673, 0.0142993884930920, 0.0126757721513128},
{201079.8438917815219611, 0.0152321204644060, 0.0137848751011903},
{251300.5936083240376320, 0.0162928250060635, 0.0145966166188692},
{306226.4651513504795730, 0.0170716667881158, 0.0154694407451461},
{365411.6237626027432270, 0.0179363122273518, 0.0161442960301303},
{429040.2882695075240918, 0.0186010901017654, 0.0168653273105185},
{496760.1298016409273259, 0.0193299284263326, 0.0174428639496000},
{568728.3250339259393513, 0.0199079496093601, 0.0180576039616062} },
{ {7040.2380998625694701, 0.0048203877503089, 0.0044476171094035},
{25944.0608053450414445, 0.0081802041473528, 0.0058965542791764},
{51709.8874015684850747, 0.0094429730447925, 0.0084715097283211},
{83594.4882232758536702, 0.0113488770798451, 0.0097424738693416},
{121480.5692016646935372, 0.0124437340258965, 0.0112239328487358},
{164527.3578003974107560, 0.0137668231770026, 0.0122011444165551},
{212953.8646401360747404, 0.0146654330139801, 0.0132691655840530},
{266139.0869482467533089, 0.0156874686433773, 0.0140507128550921},
{324306.4247879405156709, 0.0164380416792314, 0.0148913140127532},
{386983.7505108956829645, 0.0172714042228801, 0.0155411586586904},
{454365.9444977868115529, 0.0179122343180243, 0.0162356587559994},
{526079.8509630706394091, 0.0186148928770554, 0.0167918659437223},
{602291.7870389170711860, 0.0191722403602632, 0.0173840584893862} },
{ {7447.4942692874510612, 0.0046431070506235, 0.0042840113110656},
{27446.4224647702794755, 0.0078797018874191, 0.0056788094280848},
{54704.2137971531265066, 0.0090963500753717, 0.0081597442277343},
{88435.7381380254519172, 0.0109329042586642, 0.0093838908705616},
{128515.7780047692504013, 0.0119881005936006, 0.0108113072338046},
{174055.3142287767259404, 0.0132634292993615, 0.0117526933102288},
{225285.4392973450303543, 0.0141297419104439, 0.0127818659001371},
{281549.3428950291709043, 0.0151151670870103, 0.0135348687368003},
{343083.0101981102488935, 0.0158389672419372, 0.0143449905672138},
{409387.0090588952880353, 0.0166426951464730, 0.0149711842760060},
{480667.2861399344983511, 0.0172608317811450, 0.0156405822814011},
{556529.0565772734116763, 0.0179386764607711, 0.0161766140206889},
{637148.1256285562412813, 0.0184764225011397, 0.0167474682035167} },
{ {7870.1386044472574213, 0.0044754361284461, 0.0041292764490501},
{29005.5808249612782674, 0.0075954643670949, 0.0054729291696945},
{57811.7385824497177964, 0.0087684692507567, 0.0078648919736404},
{93460.0199075929849641, 0.0105393802510223, 0.0090447632376394},
{135816.9612360006431118, 0.0115570232101855, 0.0104210348147236},
{183943.4822786790027749, 0.0127871172408576, 0.0113285282661080},
{238083.2008913507743273, 0.0136228306970868, 0.0123209271069218},
{297542.1494117460679263, 0.0145735616553554, 0.0130469176364692},
{362569.3653643261059187, 0.0152719822365948, 0.0138281808627790},
{432637.0813952800235711, 0.0160476113738208, 0.0144319869804996},
{507962.7228438211022876, 0.0166442240040594, 0.0150776118450426},
{588129.0584606672637165, 0.0172985287434939, 0.0155945408740438},
{673321.7358823100803420, 0.0178176739809573, 0.0161451815694047} },
{ {8308.4560706189404300, 0.0043166923177809, 0.0039827821342693},
{30622.5876540967146866, 0.0073263383049488, 0.0052780654438236},
{61034.5579894733964466, 0.0084580032643342, 0.0075857504053365},
{98670.7229429895814974, 0.0101667189955632, 0.0087237075454505},
{143389.0442714801174589, 0.0111487698085008, 0.0100515291001840},
{194198.5324486455065198, 0.0123359809557315, 0.0109269263819569},
{251355.7824539555585943, 0.0131426772296382, 0.0118844816632124},
{314128.2944600274786353, 0.0140604999333592, 0.0125848845996665},
{382778.6342613969463855, 0.0147348402267097, 0.0133387976073490},
{456749.6494914081413299, 0.0154838033733355, 0.0139213914654691},
{536270.6642270262818784, 0.0160599835375187, 0.0145444805543398},
{620901.1683820589678362, 0.0166919378060637, 0.0150433053248459},
{710837.0128106722841039, 0.0171934170279183, 0.0155747803209337} },
{ {8762.7316332007139863, 0.0041662525513369, 0.0038439530337086},
{32298.4947214762178191, 0.0070712708820882, 0.0050934447496482},
{64374.7682520857779309, 0.0081637402632590, 0.0073212220460037},
{104071.2366627049341332, 0.0098134720070466, 0.0084194612596392},
{151236.9524889566237107, 0.0107617582676041, 0.0097013419268929},
{204827.1352411608386319, 0.0119082789787997, 0.0105463148886507},
{265111.8170178976724856, 0.0126874336208252, 0.0114708244993239},
{331318.5660000133793801, 0.0135740145443009, 0.0121469663624132},
{403723.9608577208709903, 0.0142254875228176, 0.0128749347133355},
{481740.3953029229887761, 0.0149491227579878, 0.0134374111228432},
{565609.5198798540513963, 0.0155058903631319, 0.0140391179151937},
{654866.6980688656913117, 0.0161166059338117, 0.0145207688699399},
{749718.3513639320153743, 0.0166012933038470, 0.0150340553022119} },
{ {9233.2502570721753727, 0.0040235472201999, 0.0037122631956203},
{34034.3537972842896124, 0.0068292994182753, 0.0049183604045153},
{67834.4656052017235197, 0.0078845720038174, 0.0070703036665213},
{109664.9504821516165975, 0.0094783143077453, 0.0081308702716287},
{159365.6112722787947860, 0.0103945411137221, 0.0093691492326111},
{215835.9611619411734864, 0.0115024176835175, 0.0101852557207747},
{279359.9376178876846097, 0.0122554085545838, 0.0110783963512578},
{349123.7519922936335206, 0.0131123044217993, 0.0117315137558678},
{425418.4891100855893455, 0.0137420437183901, 0.0124348487570502},
{507625.0007719453424215, 0.0144416020249595, 0.0129782287548108},
{595997.6993672066600993, 0.0149799110446343, 0.0135596297856884},
{690046.9592087797354907, 0.0155704281380688, 0.0140249750150258},
{789990.1464342494728044, 0.0160391419463409, 0.0145209851730321} },
{ {9720.2969080283892254, 0.0038880547596785, 0.0035872310461896},
{35831.2166529789974447, 0.0065995422651249, 0.0047521657258000},
{71415.7462855990161188, 0.0076194834006893, 0.0068320767314946},
{115455.2538252716913121, 0.0091600320081135, 0.0078568779078378},
{167779.9460062468133401, 0.0100457920102572, 0.0090537385076021},
{227231.6807198380411137, 0.0111169364909853, 0.0098424319057057},
{294108.7772902167635038, 0.0118450516564247, 0.0107057690549860},
{367554.6403898529824801, 0.0126737182521741, 0.0113370161778566},
{447875.3629764204961248, 0.0132827844749255, 0.0120169426124829},
{534419.1478303602198139, 0.0139594366302661, 0.0125421795441780},
{627453.6122329576173797, 0.0144801802755925, 0.0131042806737593},
{726463.2634553065290675, 0.0150514731407162, 0.0135541310210182},
{831676.7928645978681743, 0.0155049801224733, 0.0140337175972710} },
{ {10224.1565512961715285, 0.0037592968644815, 0.0034684149681085},
{37690.1350605920160888, 0.0063811907554784, 0.0045942680113292},
{75120.7065335603547283, 0.0073675432854892, 0.0066056989570499},
{121445.5361141767789377, 0.0088575113196552, 0.0075965152178875},
{176484.8820791688340250, 0.0097142938019748, 0.0087539977019000},
{239020.9644269374548458, 0.0107504947751191, 0.0095166355303183},
{309366.9690728066489100, 0.0114549396512123, 0.0103516325416074},
{386622.0191570405149832, 0.0122567398047043, 0.0109620878588059},
{471107.7264050952508114, 0.0128461262654712, 0.0116197509729279},
{562138.5183938021073118, 0.0135009690966276, 0.0121277359870129},
{659995.6679990446427837, 0.0140049845157980, 0.0126714780711443},
{764136.9224303631344810, 0.0145579665047216, 0.0131065917479374},
{874802.6854499883484095, 0.0149969857727177, 0.0135705525898045} },
{ {10745.1141524671311345, 0.0036368342511261, 0.0033554093848260},
{39612.1607951623082045, 0.0061735020703096, 0.0044441232144353},
{78951.4425857477472164, 0.0071278962175325, 0.0063903968347562},
{127639.1867756557767279, 0.0085697288144065, 0.0073488923739025},
{185485.3448814833827782, 0.0093989279124609, 0.0084689053986033},
{251210.4827981536800507, 0.0104018602452718, 0.0092067570799797},
{325143.1460052336915396, 0.0110837640791127, 0.0100147833126405},
{406336.6762458865414374, 0.0118599749084224, 0.0106054556890537},
{495128.7233414240763523, 0.0124306128262754, 0.0112419275155223},
{590798.7943550649797544, 0.0130646748959188, 0.0117334945334064},
{693642.2761701745912433, 0.0135527474504651, 0.0122597585605149},
{803089.2477234977995977, 0.0140882756394979, 0.0126808453257404},
{919392.2189441339578480, 0.0145134822684029, 0.0131299277435201} },
{ {11283.4546764536007686, 0.0035202628970713, 0.0032478412857759},
{41598.3456277664445224, 0.0059757929065119, 0.0043012312220899},
{82910.0506837058346719, 0.0068997552119323, 0.0061854589973761},
{134039.5952381429087836, 0.0082957427720716, 0.0071131910383742},
{194786.2598075857968070, 0.0090986649227745, 0.0081975220905570},
{263806.9063487951643765, 0.0100698986190387, 0.0089117759737263},
{341445.9411287488765083, 0.0107303203727509, 0.0096941142057352},
{426709.3996148091391660, 0.0114821398678178, 0.0102659484080473},
{519951.4977266335044987, 0.0120349031021238, 0.0108822334996278},
{620415.6576410017441958, 0.0126491498839703, 0.0113581637189616},
{728411.8462337665259838, 0.0131220170448939, 0.0118677754706638},
{843341.5509080380434170, 0.0136408964474490, 0.0122755004200054},
{965469.7880611085565761, 0.0140529247449020, 0.0127104050985182} },
{ {11839.4630892503737414, 0.0034092106962529, 0.0031453671363725},
{43649.7413339598570019, 0.0057874338438850, 0.0041651316599685},
{86998.6270688564691227, 0.0066823952695138, 0.0059902303198621},
{140650.1509322866622824, 0.0080346854781959, 0.0068886575774564},
{204392.5522515103220940, 0.0088125561832951, 0.0079389824214072},
{276816.9056009291089140, 0.0097535644246684, 0.0086307521446864},
{358283.9874804103164934, 0.0103934981259027, 0.0093886052889329},
{447750.9772201699088328, 0.0111220511390313, 0.0099424869841849},
{545589.1934978179633617, 0.0116577605003892, 0.0105395276188198},
{651004.7901281112572178, 0.0122530990960817, 0.0110005536001158},
{764322.7876624515047297, 0.0127114539980938, 0.0114942878857392},
{884915.1435182563727722, 0.0132144414100772, 0.0118892748931288},
{1013059.7874797495314851, 0.0136138879053517, 0.0123106594484318} },
{ {12413.4243562041228870, 0.0033033344791246, 0.0030476701248526},
{45767.3996889573027147, 0.0056078443245623, 0.0040354001578987},
{91219.2679838024923811, 0.0064751476090806, 0.0058041066635922},
{147474.2432884634181391, 0.0077857563559259, 0.0066745970146146},
{214309.1476116080884822, 0.0085397263315018, 0.0076924882705253},
{290247.1510757803916931, 0.0094518927936202, 0.0083628185360221},
{375665.9181180170271546, 0.0100722724080265, 0.0090973157427927},
{469472.1970176390605047, 0.0107786161124858, 0.0096340760379246},
{572054.9545898208161816, 0.0112980432942688, 0.0102127569516253},
{682581.8736910533625633, 0.0118753267378025, 0.0106595663320845},
{801393.5099143469706178, 0.0123198214256680, 0.0111381508418000},
{927831.3370821463176981, 0.0128076289397616, 0.0115209856892694},
{1062186.6118457338307053, 0.0131950551173625, 0.0119294679070074} },
{ {13005.6234427074668929, 0.0032023173524987, 0.0029544577048738},
{47952.3724684449261986, 0.0054364881699206, 0.0039116450192098},
{95574.0696719524712535, 0.0062773945162302, 0.0056265301853590},
{154515.2617424774507526, 0.0075482158296715, 0.0064703676336329},
{224540.9712856141268276, 0.0082793666053726, 0.0074573025792806},
{304104.3132955743931234, 0.0091639921238138, 0.0081071744002287},
{393600.3660756398458034, 0.0097656959980019, 0.0088193766103755},
{491883.8469634712091647, 0.0104508248685262, 0.0093397961806752},
{599361.9249332966282964, 0.0109546960371349, 0.0099009488772255},
{715162.5902146353619173, 0.0115147272278326, 0.0103341877508965},
{839642.4224356820341200, 0.0119459756249399, 0.0107983065663196},
{972111.4430892762029544, 0.0124192738459267, 0.0111695397945083},
{1112874.6557763204909861, 0.0127952086489305, 0.0115657005823677} },
{ {13616.3453144229515601, 0.0031058663204679, 0.0028654593976458},
{50205.7114487452636240, 0.0052728695702658, 0.0037935042440490},
{100065.1283779447257984, 0.0060885647345634, 0.0054569851422806},
{161776.5957289986254182, 0.0073213798331627, 0.0062753761522826},
{235092.9486785506014712, 0.0080307288577936, 0.0072327438282684},
{318395.0627880793181248, 0.0088890375095735, 0.0078630793052304},
{412095.9644010870251805, 0.0094728924278005, 0.0085539843105396},
{514996.7150138342985883, 0.0101377427908835, 0.0090587971597756},
{627523.2484567794017494, 0.0106267418683094, 0.0096032038403833},
{748762.6215689838863909, 0.0111702771687807, 0.0100234798386459},
{879087.9346594617236406, 0.0115888577950519, 0.0104737766360911},
{1017776.7730336039094254, 0.0120482787849883, 0.0108339261438949},
{1165148.3138596725184470, 0.0124132209096377, 0.0112183122273089} },
{ {14245.8749368882072304, 0.0030137101530607, 0.0027804248232395},
{52528.4684066930494737, 0.0051165294910560, 0.0036806428650264},
{104694.5403454573242925, 0.0059081293348937, 0.0052949941339211},
{169261.6346849613764789, 0.0071046148855286, 0.0060890733989261},
{245970.0051900871039834, 0.0077931201884370, 0.0070181810891479},
{333126.0700819185585715, 0.0086262648481592, 0.0076298477629300},
{431161.3461428778828122, 0.0091930497412347, 0.0083003948238880},
{538821.5891239388147369, 0.0098385039374653, 0.0087902917138724},
{656552.0690870461985469, 0.0103132756062375, 0.0093186888650553},
{783397.6496152146719396, 0.0108410281377758, 0.0097265739682147},
{919748.4560098384972662, 0.0112474866007511, 0.0101636549450312},
{1064848.6383670167997479, 0.0116936265799083, 0.0105132083635378},
{1219031.9806591903325170, 0.0120480465817644, 0.0108863347494199} },
{ {14894.4972757978630398, 0.0029255974734768, 0.0026991219336604},
{54921.6951194270368433, 0.0049670424469588, 0.0035727505577543},
{109464.4018222153390525, 0.0057355980067680, 0.0051401147301036},
{176973.7680485666205641, 0.0068973336693530, 0.0059109504329454},
{257177.0662254077324178, 0.0075658981223099, 0.0068130295831205},
{348304.0057026736321859, 0.0083749655445025, 0.0074068444079216},
{450805.1443496109568514, 0.0089254148847443, 0.0080579184731593},
{563369.2572509207529947, 0.0095523050810057, 0.0085335500558902},
{686461.5307481860509142, 0.0100134575384136, 0.0090466317299702},
{819083.3562064132420346, 0.0105261002026625, 0.0094426648365654},
{961642.3958998135058209, 0.0109209514836370, 0.0098671013878825},
{1113348.3505434752441943, 0.0113543733098930, 0.0102065182511586},
{1274550.0507140429690480, 0.0116987155384409, 0.0105688704832163} },
{ {15562.4972960890754621, 0.0028412950387143, 0.0026213354248485},
{57386.4433648283375078, 0.0048240136013693, 0.0034695394946380},
{114376.8090536527306540, 0.0055705157234972, 0.0049919364400009},
{184916.3852594257914461, 0.0066989910530630, 0.0057405350576213},
{268719.0571917841443792, 0.0073484662715512, 0.0066167466877841},
{363935.5401871096109971, 0.0081344817456686, 0.0071934796625411},
{471035.9920716277556494, 0.0086692886584809, 0.0078259152295810},
{588650.5073504310566932, 0.0092784003438693, 0.0082878949115302},
{717264.7773623103275895, 0.0097265078290418, 0.0087863157294362},
{855835.4232047728728503, 0.0102246760822486, 0.0091710050070691},
{1004788.1637342481408268, 0.0106084066360626, 0.0095833361778458},
{1163297.2210023512598127, 0.0110296420835731, 0.0099130499103160},
{1331726.9185444838367403, 0.0113643264615957, 0.0102650861355749} },
{ {16250.1599640681870369, 0.0027605861914468, 0.0025468653066857},
{59923.7649209514347604, 0.0046870761545705, 0.0033707424133253},
{119433.8582867592485854, 0.0054124597385781, 0.0048500779832845},
{193092.8757579580124002, 0.0065090805074204, 0.0055773886807318},
{280600.9034926146268845, 0.0071402704265153, 0.0064288283415686},
{380027.3440613995771855, 0.0079042020457445, 0.0069892058331325},
{491862.5223580335732549, 0.0084240211649870, 0.0076037904849245},
{614676.1273785375524312, 0.0090160963598712, 0.0080526970501370},
{748974.9528494304977357, 0.0094517014753398, 0.0085370749538569},
{893669.5324514224193990, 0.0099359958777607, 0.0089108999932584},
{1049204.1689091930165887, 0.0103090655640135, 0.0093116347264764},
{1214716.5611615960951895, 0.0107186174198925, 0.0096320544645716},
{1390586.9786473650019616, 0.0110440410813161, 0.0099742073295047} },
{ {16957.7702450916003727, 0.0026832694640877, 0.0024755256137501},
{62534.7115661788193393, 0.0045558889873083, 0.0032761108771938},
{124637.6457687594374875, 0.0052610368759712, 0.0047141848298459},
{201506.6289867908635642, 0.0063271308720434, 0.0054211034837821},
{292827.5305405915132724, 0.0069407950278983, 0.0062488058008335},
{396586.0878652714891359, 0.0076835576083425, 0.0067935135891488},
{513293.3682653058785945, 0.0081890076992791, 0.0073909912382756},
{641456.9052924429997802, 0.0087647479050837, 0.0078273712529980},
{781605.2011290142545477, 0.0091883637518010, 0.0082982900312934},
{932601.3657969852210954, 0.0096593523134151, 0.0086617038220897},
{1094908.8208130733110011, 0.0100221961743879, 0.0090513230230551},
{1267627.6824167950544506, 0.0104205401690836, 0.0093628352874115},
{1451154.6255031831096858, 0.0107370789687390, 0.0096955136793064} },
{ {17685.6131042471934052, 0.0026091573180138, 0.0024071432409007},
{65220.3350795703590848, 0.0044301345332199, 0.0031854137038430},
{129990.2677482722792774, 0.0051158810823764, 0.0045839269776748},
{210161.0343876676342916, 0.0061527034332672, 0.0052712998660025},
{305403.8637444931082428, 0.0067495599782584, 0.0060762427108022},
{413618.4421344220172614, 0.0074720186615410, 0.0066059287828554},
{535337.1628418145701289, 0.0079636850332882, 0.0071870026507178},
{669003.6290480159223080, 0.0085237539465543, 0.0076113726708680},
{815168.6661187281133607, 0.0089358660895213, 0.0080693842798366},
{972646.6050823128316551, 0.0093940864303362, 0.0084228150252514},
{1141920.5288276628125459, 0.0097471163294444, 0.0088017734588752},
{1322051.8961806497536600, 0.0101347029171160, 0.0091047436902962},
{1513454.2535454118624330, 0.0104427128228030, 0.0094283343389488} },
{ {18433.9735087900080543, 0.0025380750033001, 0.0023415568902246},
{67981.6872432523523457, 0.0043095168533674, 0.0030984355458181},
{135493.8204737477353774, 0.0049766512124793, 0.0044589969435445},
{219059.4814051018620376, 0.0059853892804128, 0.0051276241319132},
{318334.8285142673994415, 0.0065661177561016, 0.0059107324562060},
{431131.0773980739177205, 0.0072690913244994, 0.0064260095724187},
{558002.5391424101544544, 0.0077475280513594, 0.0069913449281937},
{697327.0865926617989317, 0.0082925540636746, 0.0074041935281832},
{849678.4917324725538492, 0.0086936223434591, 0.0078498202258153},
{1013820.9321430575801060, 0.0091395836855736, 0.0081936730110287},
{1190257.7023275224491954, 0.0094831898186848, 0.0085624010479127},
{1378010.5138342506252229, 0.0098604458195345, 0.0088571750206745},
{1577510.2573035524692386, 0.0101602641978658, 0.0091720439726086} },
{ {19203.1364227148515056, 0.0024698595258397, 0.0022786161172114},
{70819.8198274095339002, 0.0041937598922896, 0.0030149756052589},
{141150.4001944056653883, 0.0048430290224108, 0.0043391079430382},
{228205.3594835214898922, 0.0058248069089593, 0.0049897463981741},
{331625.3502628349233419, 0.0063900508001700, 0.0057518957611924},
{449130.6642033667885698, 0.0070743147306974, 0.0062533438157043},
{581298.1302211604779586, 0.0075400466995131, 0.0068035704977524},
{726438.0659130329731852, 0.0080706252028094, 0.0072053601370809},
{885147.8218924143584445, 0.0084610854068816, 0.0076390964486988},
{1056140.0288154587615281, 0.0088952704127761, 0.0079737547767750},
{1239938.7506949917878956, 0.0092298227039400, 0.0083326600015731},
{1435524.8467541185673326, 0.0095971528215915, 0.0086195651248500},
{1643347.0311567392200232, 0.0098890996261891, 0.0089260591019706} },
{ {19993.3868122181520448, 0.0024043587106937, 0.0022181804656470},
{73735.7846211572468746, 0.0040826058959412, 0.0029348464689377},
{146962.1031597345718183, 0.0047147173493630, 0.0042239922400033},
{237602.0580686978937592, 0.0056706000464214, 0.0048573586942898},
{345280.3544029542827047, 0.0062209691353876, 0.0055993785124229},
{467623.8730825992533937, 0.0068872584164431, 0.0060875467062304},
{605232.5691357509931549, 0.0073407832157709, 0.0066232614453151},
{756347.3549360224278644, 0.0078574787301250, 0.0070144301880681},
{921589.8005021220305935, 0.0082377441363754, 0.0074367447182921},
{1099619.5769310498144478, 0.0086606106072886, 0.0077625719254866},
{1290982.0832596542313695, 0.0089864599987671, 0.0081120406201974},
{1494616.2063085569534451, 0.0093442482232167, 0.0083913871377330},
{1710988.9694817638956010, 0.0096286270938346, 0.0086898347908865} },
{ {20805.0096431285492145, 0.0023414303512210, 0.0021601186820479},
{76730.6334014596504858, 0.0039758139745350, 0.0028578730507476},
{152931.0256192541273776, 0.0045914384575227, 0.0041133996478299},
{247252.9666066037025303, 0.0055224356762669, 0.0047301732383271},
{359304.7663479430484585, 0.0060585082153430, 0.0054528497816885},
{486617.3745797797455452, 0.0067075199470010, 0.0059282586256978},
{629814.4889391523320228, 0.0071493096119745, 0.0064500271883285},
{787065.7416229310911149, 0.0076526577518050, 0.0068309902886225},
{959017.5714813083177432, 0.0080231205553886, 0.0072423273934941},
{1144275.2583198498468846, 0.0084351030018955, 0.0075596679548938},
{1343406.1094036153517663, 0.0087525826476748, 0.0079000664682324},
{1555305.9038554984144866, 0.0091011935536298, 0.0081721485656097},
{1780460.4667448436375707, 0.0093782928339809, 0.0084628616321211} },
{ {21638.2898809133330360, 0.0022809414354024, 0.0021043080010687},
{79805.4179484630731167, 0.0038731587954119, 0.0027838916304808},
{159059.2638241440290585, 0.0044729325340708, 0.0040070961659718},
{257161.4745452323986683, 0.0053800022402142, 0.0046079208686565},
{373703.5115130658377893, 0.0059023269590116, 0.0053120000274522},
{506117.8392353200470097, 0.0065347227561472, 0.0057751431908995},
{655052.5226903533330187, 0.0069652253818214, 0.0062835023583908},
{818604.0139521533856168, 0.0074557346743607, 0.0066546537242115},
{997444.2787394262850285, 0.0078167673079813, 0.0070554350558057},
{1190122.7548073972575366, 0.0082182784037330, 0.0073646157911268},
{1397229.2384756640531123, 0.0085277047748028, 0.0076962918038203},
{1617615.2507464776281267, 0.0088674847244098, 0.0079613886313660},
{1851785.9173169583082199, 0.0091375784053509, 0.0082446630051670} },
{ {22493.5124894578984822, 0.0022227674413437, 0.0020506334951569},
{82961.1900432331894990, 0.0037744293926105, 0.0027127489789357},
{165348.9140219073160551, 0.0043589563185333, 0.0039048627392268},
{267330.9713317519053817, 0.0052430080005033, 0.0044903496162136},
{388481.5153131475672126, 0.0057521059620400, 0.0051765394572445},
{526131.9375896224519238, 0.0063685141773138, 0.0056278854749979},
{680955.3034457599278539, 0.0067881554117474, 0.0061233448731823},
{850972.9598596560535952, 0.0072663089805867, 0.0064850584188210},
{1036883.0661894109798595, 0.0076182653373370, 0.0068756843535682},
{1237177.7482195070479065, 0.0080096972660281, 0.0071770155417659},
{1452469.8798285396769643, 0.0083113711748185, 0.0075002992372767},
{1681565.5583257055841386, 0.0086426494327653, 0.0077586758558816},
{1924989.7155872425064445, 0.0089059980271836, 0.0080347925777046} },
{ {23370.9624395558203105, 0.0021667916948896, 0.0019989874811841},
{86199.0014674638659926, 0.0036794280813217, 0.0026443015608624},
{171802.0724720766593236, 0.0042492818534516, 0.0038064941262673},
{277764.8464147853082977, 0.0051111795461113, 0.0043772234027449},
{403643.7031653262674809, 0.0056075458648958, 0.0050461965343162},
{546666.3401970209088176, 0.0062085636473242, 0.0054861903857118},
{707531.4642630461603403, 0.0066177480747194, 0.0059692341778032},
{884183.3673115676501766, 0.0070840051998625, 0.0063218650750774},
{1077347.0777396531775594, 0.0074272217669928, 0.0067027160359627},
{1285455.9203766060527414, 0.0078089474713155, 0.0069964924466749},
{1509146.4428044783417135, 0.0081031550221231, 0.0073116975948883},
{1747178.1379279925022274, 0.0084262447904355, 0.0075636058517207},
{2000096.2559247957542539, 0.0086830961454564, 0.0078328320262617} },
{ {24270.9246921522499179, 0.0021129047833811, 0.0019492689793578},
{89519.9039991075696889, 0.0035879694669433, 0.0025784148074898},
{178420.8354171893442981, 0.0041436953434456, 0.0037117978667913},
{288466.4892406724393368, 0.0049842604287177, 0.0042683208528486},
{419195.0004864985239692, 0.0054683658627412, 0.0049207166146306},
{567727.7175736667122692, 0.0060545610655337, 0.0053497811846711},
{734789.6382027133367956, 0.0064536734887695, 0.0058208696394504},
{918246.0242650298168883, 0.0069084710536831, 0.0061647554761914},
{1118849.4573048555757850, 0.0072432679644273, 0.0065361931577327},
{1334972.9531079914886504, 0.0076156423053683, 0.0068226950066203},
{1567277.3367426285985857, 0.0079026557768862, 0.0071301199684569},
{1814474.3008876452222466, 0.0082178551560303, 0.0073757993081487},
{2077129.9326926562935114, 0.0084684452077215, 0.0076383889543660} },
{ {25193.6842143554677023, 0.0020610040198271, 0.0019013832182515},
{92924.9494161995098693, 0.0034998795393305, 0.0025149624521587},
{185207.2991119644138962, 0.0040419961123596, 0.0036205933379573},
{299439.2892641361104324, 0.0048620099155248, 0.0041634342082538},
{435140.3326936035882682, 0.0053343023427318, 0.0047998607010364},
{589322.7402912694960833, 0.0059062152933441, 0.0052183981343676},
{762738.4583225516835228, 0.0062956219241966, 0.0056779690799520},
{953171.7186801714124158, 0.0067393757591450, 0.0060134309336571},
{1161403.3487886225339025, 0.0070660577696754, 0.0063757994380090},
{1385744.5282110739499331, 0.0074294186035478, 0.0066552932724835},
{1626880.9709808311890811, 0.0077094972689148, 0.0069552219316915},
{1883475.3585255891084671, 0.0080170901511645, 0.0071949001487935},
{2156115.1402496956288815, 0.0082616436262962, 0.0074510949889234} },
{ {26139.5259713708837808, 0.0020109929526074, 0.0018552411823523},
{96415.1895116704545217, 0.0034149948439289, 0.0024538259226873},
{192163.5598082118085586, 0.0039439956489394, 0.0035327108909181},
{310686.6359310048865154, 0.0047442018478519, 0.0040623683344631},
{451484.6252058630343527, 0.0052051076366201, 0.0046834043032218},
{611458.0788822539616376, 0.0057632527803613, 0.0050917972597337},
{791386.5576820367714390, 0.0061433023451728, 0.0055402674327473},
{988971.2385167971951887, 0.0065764084754271, 0.0058676108667511},
{1205021.8960978633258492, 0.0068952658729372, 0.0062212377581309},
{1437786.3275226627010852, 0.0072499350525895, 0.0064939772793707},
{1687975.7548517079558223, 0.0075233259421783, 0.0067866799078933},
{1954202.6221640298608691, 0.0078235828428934, 0.0070205738451905},
{2237076.2729256600141525, 0.0080623139119313, 0.0072706040376668} },
{ {27108.7349309278943110, 0.0019627809165006, 0.0018107591972925},
{99991.6760580827394733, 0.0033331617225717, 0.0023948937850310},
{199291.7137584833544679, 0.0038495167324570, 0.0034479910601320},
{322211.9186996856005862, 0.0046306235951791, 0.0039649398112324},
{468232.8034418643219396, 0.0050805488777898, 0.0045711363936989},
{634140.4038708347361535, 0.0056254163052679, 0.0049697492140680},
{820742.5693436900619417, 0.0059964410728530, 0.0054075155126620},
{1025655.3717266733292490, 0.0064192768794999, 0.0057270315010217},
{1249718.2431471857707947, 0.0067305863270052, 0.0060722287855626},
{1491114.0328525686636567, 0.0070768706336359, 0.0063384556118989},
{1750580.0977130155079067, 0.0073438092454104, 0.0066241896736668},
{2026677.4031127372290939, 0.0076369880765505, 0.0068525058714166},
{2320037.7250731261447072, 0.0078701009626243, 0.0070965906914611} },
{ {28101.5960560868479661, 0.0019162826209137, 0.0017678585498653},
{103655.4608390591311036, 0.0032542356171315, 0.0023380612333435},
{206593.8572146284568589, 0.0037583926308851, 0.0033662838382745},
{334018.5270130207063630, 0.0045210750957980, 0.0038709760982269},
{485389.7928211133694276, 0.0049604069525674, 0.0044628584501991},
{657376.3858809693483636, 0.0054924638205897, 0.0048520382387470},
{850815.1263679714174941, 0.0058547805588026, 0.0052794788876402},
{1063234.9062666515819728, 0.0062677058591108, 0.0055914446747210},
{1295505.5338443750515580, 0.0065717311819473, 0.0059285097115071},
{1545743.3260129960253835, 0.0069099231928544, 0.0061884540881909},
{1814712.4088360455352813, 0.0071706341534158, 0.0064674649874799},
{2100921.0126865012571216, 0.0074569809451424, 0.0066904002862965},
{2405023.8910187585279346, 0.0076846704901512, 0.0069287487606882} },
{ {29118.3943141502059007, 0.0018714177719037, 0.0017264651389267},
{107407.5956362444412662, 0.0031780804302937, 0.0022832296221613},
{214072.0864295653591398, 0.0036704663646376, 0.0032874480106579},
{346109.8503214747761376, 0.0044153679758208, 0.0037803147695355},
{502960.5187626809347421, 0.0048444755372916, 0.0043583835766977},
{681172.6953817225294188, 0.0053641673916174, 0.0047384612084307},
{881612.8618125378852710, 0.0057180782580556, 0.0051559368428213},
{1101720.6301212343387306, 0.0061214363119471, 0.0054606167428994},
{1342396.9121038401499391, 0.0064184292306763, 0.0057898330919551},
{1601689.8888177641201764, 0.0067488081281146, 0.0060437145514713},
{1880391.0975605302955955, 0.0070035058097519, 0.0063162363287837},
{2176954.7621821435168386, 0.0072832553826016, 0.0065339784317168},
{2492059.1650828192941844, 0.0075057075768131, 0.0067667899281569} },
{ {30159.4146703457336116, 0.0018281107249182, 0.0016865091549690},
{111249.1322318957973039, 0.0031045679381928, 0.0022303060364314},
{221728.4976614382176194, 0.0035855900299573, 0.0032113505438484},
{358489.2780752466642298, 0.0043133247394267, 0.0036928028101745},
{520949.9066894175484776, 0.0047325602129775, 0.0042575356956739},
{705536.0029490048764274, 0.0052403122208689, 0.0046288267535952},
{913144.4087131032720208, 0.0055861055931494, 0.0050366814284686},
{1141123.3312205660622567, 0.0059802240414041, 0.0053343275702177},
{1390405.5217900313436985, 0.0062704248557204, 0.0056559657819181},
{1658969.4030819209292531, 0.0065932571808721, 0.0059039937593488},
{1947634.5732367674354464, 0.0068421462770615, 0.0061702497402567},
{2254799.9629061273299158, 0.0071155228695660, 0.0063829777356110},
{2581167.9415773060172796, 0.0073329153453168, 0.0066104425129981} },
{ {31224.9420903706995887, 0.0017862901653418, 0.0016479247849274},
{115181.1224078643717803, 0.0030335772499259, 0.0021792028965437},
{229565.1871479736291803, 0.0035036241764620, 0.0031378660231542},
{371160.1997318912763149, 0.0042147780237556, 0.0036082959689938},
{539362.8820206237724051, 0.0046244776507772, 0.0041601488051484},
{730472.9791164096677676, 0.0051206957502590, 0.0045229544532283},
{945418.4002226383890957, 0.0054586470004306, 0.0049215165842746},
{1181453.7975416854023933, 0.0058438387401647, 0.0052123696043421},
{1439544.5068550878204405, 0.0061274769660786, 0.0055266879563326},
{1717597.5506025061476976, 0.0064430173239946, 0.0057690623618815},
{2016461.2451766992453486, 0.0066862933865232, 0.0060292657619373},
{2334477.9261489664204419, 0.0069535112415496, 0.0062371506110053},
{2672374.6148219499737024, 0.0071660137346033, 0.0064594503321661} },
{ {32315.2615407581324689, 0.0017458888143147, 0.0016106499408778},
{119204.6179469046910526, 0.0029649943099835, 0.0021298375944425},
{237584.2511576914985199, 0.0034244372338816, 0.0030668761348114},
{384126.0047564897686243, 0.0041195699127599, 0.0035266581630752},
{558204.3701777670066804, 0.0045200548616087, 0.0040660662945025},
{755990.2944539750460535, 0.0050051268340608, 0.0044206740917018},
{978443.4693105939077213, 0.0053354990515705, 0.0048102573328010},
{1222722.8170339609496295, 0.0057120630535316, 0.0050945470226068},
{1489827.0112199576105922, 0.0059893580196878, 0.0054017922062866},
{1777590.0132004548795521, 0.0062978497367000, 0.0056387039603312},
{2086889.5226904924493283, 0.0065356996781414, 0.0058930584493676},
{2416009.9632082870230079, 0.0067969635904257, 0.0060962634414800},
{2765703.5791122047230601, 0.0070047383713075, 0.0063135716520899} },
{ {33430.6579862137950840, 0.0017068431577096, 0.0015746260096669},
{123320.6706306818377925, 0.0028987114396620, 0.0020821321586246},
{245787.7859388549695723, 0.0033479049836736, 0.0029982691887999},
{397390.0826096245436929, 0.0040275513045451, 0.0034477609286574},
{577479.2965844339923933, 0.0044191285043820, 0.0039751403139818},
{782094.6195142075885087, 0.0048934249762393, 0.0043218249737077},
{1012228.2490722865331918, 0.0052164696432142, 0.0047027290361005},
{1264941.1776601958554238, 0.0055846917157189, 0.0049806749455499},
{1541266.1787151501048356, 0.0058558531201339, 0.0052810827075401},
{1838962.4726737190503627, 0.0061575288590417, 0.0055127142395994},
{2158937.8150965319946408, 0.0063901314221134, 0.0057614144689578},
{2499417.3853675713762641, 0.0066456372508800, 0.0059600956462309},
{2861179.2287509716115892, 0.0068488395284069, 0.0061725782218730} },
{ {34571.4163929326750804, 0.0016690931960746, 0.0015397976220600},
{127530.3322421221091645, 0.0028346269137457, 0.0020360129451392},
{254177.8877454706525896, 0.0032739100715603, 0.0029319396786076},
{410955.8226819207193330, 0.0039385813275772, 0.0033714829147796},
{597192.5866650124080479, 0.0043215442476256, 0.0038872311929434},
{808792.6248262715525925, 0.0047854196266589, 0.0042262552922252},
{1046781.3725705880206078, 0.0051013772496259, 0.0045987667096558},
{1308119.6673858885187656, 0.0054615307524740, 0.0048705787110821},
{1593875.1533081412781030, 0.0057267591834611, 0.0051643744526415},
{1901730.6108492289204150, 0.0060218415188978, 0.0053909001675640},
{2232624.5317049096338451, 0.0062493677159505, 0.0056341322627931},
{2584721.5039182836189866, 0.0064993028644880, 0.0058284388170576},
{2958825.9580001696012914, 0.0066980811635852, 0.0060362543812888} },
{ {35737.8217259711673250, 0.0016325822137270, 0.0015061124397046},
{131834.6545621908153407, 0.0027726445700406, 0.0019914103521678},
{262756.6528308932902291, 0.0032023415571934, 0.0028677878748838},
{424826.6144887149566785, 0.0038525268016536, 0.0032977094157750},
{617349.1658289851620793, 0.0042271561797421, 0.0038022069028599},
{836090.9809496095404029, 0.0046809495308517, 0.0041338215450011},
{1082111.4728692253120244, 0.0049900502311015, 0.0044982143895270},
{1352269.0741410462651402, 0.0053423967443122, 0.0047640932044644},
{1647667.0788902242202312, 0.0056018841680732, 0.0050514925417660},
{1965910.1095126196742058, 0.0058905861252454, 0.0052730792556972},
{2307968.0818277266807854, 0.0061131996498455, 0.0055110212772433},
{2671943.6301385061815381, 0.0063577435146128, 0.0057010959212855},
{3058668.1613418641500175, 0.0065522400302855, 0.0059043962367127} },
{ {36930.1589524446899304, 0.0015972565653499, 0.0014735209580394},
{136234.6893787477456499, 0.0027126734479757, 0.0019482585566650},
{271526.1774491086252965, 0.0031330944982657, 0.0028057194496111},
{439005.8474673188175075, 0.0037692617393993, 0.0032263319390918},
{637953.9595250120619312, 0.0041358262639900, 0.0037199425607335},
{863996.3584350144956261, 0.0045798621288640, 0.0040443879951733},
{1118227.1830315643455833, 0.0048823261958532, 0.0044009245463487},
{1397400.1859602488111705, 0.0052271161453454, 0.0046610622386991},
{1702655.0993622629903257, 0.0054810463626333, 0.0049422715274277},
{2031516.6504756854847074, 0.0057635719220054, 0.0051590788755554},
{2384986.8747976776212454, 0.0059814295344723, 0.0053919012501313},
{2761105.0753065608441830, 0.0062207539262266, 0.0055778805650274},
{3160730.2326416517607868, 0.0064111048553650, 0.0057768108993673} },
{ {38148.7130403827759437, 0.0015630654786498, 0.0014419763242136},
{140731.4884671044128481, 0.0026546274546827, 0.0019064952706062},
{280488.5578550826758146, 0.0030660695652059, 0.0027456451291031},
{453496.9110758073511533, 0.0036886668852318, 0.0031572478057357},
{659011.8931422240566462, 0.0040474238339296, 0.0036403199701459},
{892515.4278354593552649, 0.0044820129989593, 0.0039578261724006},
{1155137.1361241459380835, 0.0047780514084307, 0.0043067575441492},
{1443523.7907332270406187, 0.0051155246526159, 0.0045613379816076},
{1758852.3586621389258653, 0.0053640737268480, 0.0048365548080388},
{2098565.9155556727200747, 0.0056406182973303, 0.0050487356264193},
{2463699.3198840031400323, 0.0058538701863966, 0.0052766015509690},
{2852227.1506987605243921, 0.0060881397249767, 0.0054586163113593},
{3265036.5665394398383796, 0.0062744755776688, 0.0056533157804322} },
{ {39393.7689470623372472, 0.0015299608716483, 0.0014114341682097},
{145326.1036147605627775, 0.0025984250556688, 0.0018660615154340},
{289645.8903016630792990, 0.0030011726850335, 0.0026874803720086},
{468303.1947614281089045, 0.0036106292883133, 0.0030903597802945},
{680527.8921221916098148, 0.0039618251263891, 0.0035632271958997},
{921654.8597015213454142, 0.0043872653423443, 0.0038740144108988},
{1192849.9652096196077764, 0.0046770802422557, 0.0042155811388293},
{1490650.6764448080211878, 0.0050074666220261, 0.0044647804251611},
{1816272.0006361259147525, 0.0052508032809764, 0.0047341940662464},
{2167073.5865174578502774, 0.0055215541436434, 0.0049418947497593},
{2544123.8264145767316222, 0.0057303442657585, 0.0051649605700407},
{2945331.1675930097699165, 0.0059597167506748, 0.0053431360489902},
{3371611.5572636635042727, 0.0061421626424278, 0.0055337379381733} },
{ {40665.6116460354605806, 0.0014978971833235, 0.0013818524467962},
{150019.5866036353108939, 0.0025439889881279, 0.0018269014131907},
{299000.2710443568648770, 0.0029383147108969, 0.0026311450715363},
{483428.0879817796521820, 0.0035350419068071, 0.0030255757281751},
{702506.8819341236958280, 0.0038789128484819, 0.0034885581702861},
{951421.3245787413325161, 0.0042954895056411, 0.0037928374214508},
{1231374.3033536982256919, 0.0045792746719665, 0.0041272700134151},
{1538791.6310531890485436, 0.0049027945268588, 0.0043712568937291},
{1874927.1691800151020288, 0.0051410805390092, 0.0046350487492335},
{2237055.3451957642100751, 0.0054062172642528, 0.0048384095865130},
{2626278.8036467176862061, 0.0056106836622151, 0.0050568251522807},
{3040438.4372561867348850, 0.0058353104206777, 0.0052312814071032},
{3480479.5990050067193806, 0.0060139863468480, 0.0054179134729854} },
{ {41964.5261003646519384, 0.0014668312166080, 0.0013531912984700},
{154812.9892166253994219, 0.0024912459952019, 0.0017889619927763},
{308553.7963375986437313, 0.0028774111157644, 0.0025765632791204},
{498874.9801930576213636, 0.0034618032408441, 0.0029628082976832},
{724953.7879271861165762, 0.0037985757760001, 0.0034162123276847},
{981821.4930358451092616, 0.0042065625377447, 0.0037141858945002},
{1270718.7836209700908512, 0.0044845038024875, 0.0040417053472810},
{1587957.4425235860981047, 0.0048013684552686, 0.0042806415881482},
{1934831.0082639248576015, 0.0050347589844716, 0.0045389855850157},
{2308526.8733975486829877, 0.0052944538227292, 0.0047381410737844},
{2710182.6609705546870828, 0.0054947289250040, 0.0049520500720345},
{3137570.2709524012170732, 0.0057147551388862, 0.0051229022126271},
{3591665.0860447259619832, 0.0058897762326919, 0.0053056869660826} },
{ {43290.7972774592708447, 0.0014367219927659, 0.0013254129089187},
{159707.3632376057212241, 0.0024401265792786, 0.0017521930102126},
{318308.5624364153482020, 0.0028183817080482, 0.0025236629479762},
{514647.2608504914678633, 0.0033908169917896, 0.0029019746251209},
{747873.5355377357918769, 0.0037207083805400, 0.0033460942655951},
{1012862.0356240685796365, 0.0041203677785657, 0.0036379561317168},
{1310892.0390622494742274, 0.0043926434317376, 0.0039587744161017},
{1638158.8988131738733500, 0.0047030556437734, 0.0041928151623740},
{1995996.6617561890743673, 0.0049316995814237, 0.0044458781354725},
{2381503.8529164632782340, 0.0051861178315311, 0.0046409572776394},
{2795853.8076197369955480, 0.0053823287336613, 0.0048504975457112},
{3236747.9799642455764115, 0.0055978937470092, 0.0050178559863194},
{3705192.4126582806929946, 0.0057693705220966, 0.0051969109584776} },
{ {44644.7101361318855197, 0.0014075306159550, 0.0012984813862514},
{164703.7604452000814490, 0.0023905647727894, 0.0017165467819127},
{328266.6655910499393940, 0.0027611503674995, 0.0024723756949311},
{530748.3194095928920433, 0.0033219917456443, 0.0028429960608448},
{771271.0502535850973800, 0.0036452104831524, 0.0032781134296754},
{1044549.6228270403807983, 0.0040367944767247, 0.0035640497038851},
{1351902.7027206539642066, 0.0043035756443551, 0.0038783702203180},
{1689406.7878586214501411, 0.0046077300436744, 0.0041076643303341},
{2058437.2735672038979828, 0.0048317703216441, 0.0043556063796445},
{2456001.9655310958623886, 0.0050810706767189, 0.0045467329589828},
{2883310.6529327328316867, 0.0052733394060526, 0.0047520367787104},
{3337992.8755386625416577, 0.0054845770143026, 0.0049160074747174},
{3821085.9730655462481081, 0.0056526155931377, 0.0050914454668704} },
{ {46026.5496576517034555, 0.0013792201474961, 0.0012723626447300},
{169803.2326471744163428, 0.0023424979252220, 0.0016819780294654},
{338430.2020711326622404, 0.0027056447996305, 0.0024226365790902},
{547181.5453238608315587, 0.0032552406787876, 0.0027857979145618},
{795151.2573926673503593, 0.0035719869322903, 0.0032121838210926},
{1076890.9253443405032158, 0.0039557374342578, 0.0034923731327405},
{1393759.4078314926009625, 0.0042171884337642, 0.0038003911397202},
{1741711.8976389658637345, 0.0045152719178134, 0.0040250815003179},
{2122165.9875871865078807, 0.0047348458027948, 0.0042680563270818},
{2532036.8930693110451102, 0.0049791806758066, 0.0044553491701547},
{2972571.6061722110025585, 0.0051676244407201, 0.0046565435444776},
{3441326.2689550109207630, 0.0053746631630907, 0.0048172282148707},
{3939370.1616319492459297, 0.0055393654918129, 0.0049891575335653} },
{ {47436.6007929964107461, 0.0013517554887894, 0.0012470242969127},
{175006.8315745493746363, 0.0022958665049326, 0.0016484437354852},
{348801.2680955015239306, 0.0026517963072867, 0.0023743838959368},
{563950.3280584488529712, 0.0031904812841540, 0.0027303092184851},
{819519.0824805896263570, 0.0035009473040515, 0.0031482237241397},
{1109892.6136077593546361, 0.0038770966758574, 0.0034228375951203},
{1436470.7872547949664295, 0.0041333753506539, 0.0037247406122334},
{1795085.0162232362199575, 0.0044255674654106, 0.0039449644350434},
{2187195.9477717014960945, 0.0046408068359575, 0.0041831196577581},
{2609624.3173257717862725, 0.0048803226664010, 0.0043666928793367},
{3063655.0766844325698912, 0.0050650540909079, 0.0045638997926810},
{3546769.4714445113204420, 0.0052680174268628, 0.0047213961294310},
{4060069.3725487664341927, 0.0054294814778596, 0.0048899208077583} },
{ {48875.1485133120586397, 0.0013251032724314, 0.0012224355528897},
{180315.6090561180608347, 0.0022506139146219, 0.0016159030094074},
{359381.9599407239002176, 0.0025995395779471, 0.0023275589855979},
{581058.0570589659037068, 0.0031276351162983, 0.0026764625068419},
{844379.4509048183681443, 0.0034320056232690, 0.0030861554528205},
{1143561.3582077024038881, 0.0038007771408256, 0.0033553586475045},
{1480045.4741345460060984, 0.0040520351754430, 0.0036513268346609},
{1849536.9314115634188056, 0.0043385084724605, 0.0038672159349618},
{2253540.2979464074596763, 0.0045495400800959, 0.0041006933869982},
{2688779.9200906492769718, 0.0047843776228336, 0.0042806566211818},
{3156579.4737624535337090, 0.0049655049677611, 0.0044739932844233},
{3654343.7942960490472615, 0.0051645116387541, 0.0046283951495686},
{4183208.0000406643375754, 0.0053228316015637, 0.0047936151553835} },
{ {50342.4777837985893711, 0.0012992317607439, 0.0011985671268296},
{185730.6168633747147396, 0.0022066863195140, 0.0015843169624322},
{370174.3739177300012670, 0.0025488124855640, 0.0022821060543252},
{598508.1218106069136411, 0.0030666275537330, 0.0026241936105211},
{869737.2880947040393949, 0.0033650801033128, 0.0030259051146107},
{1177903.8296974212862551, 0.0037266883962727, 0.0032898559693410},
{1524492.1015339184086770, 0.0039730716131337, 0.0035800624840542},
{1905078.4312279019504786, 0.0042539919860571, 0.0037917435433060},
{2321212.1820248579606414, 0.0044609377012976, 0.0040206795532085},
{2769519.3831578213721514, 0.0046912322988982, 0.0041971381709615},
{3251363.2066578962840140, 0.0048688596704083, 0.0043867172523820},
{3764070.5487901973538101, 0.0050640238478330, 0.0045381148635914},
{4308810.4385567381978035, 0.0052192903091671, 0.0047001262958766} },
{ {51838.8735708293679636, 0.0012741107510827, 0.0011753911495593},
{191252.9067743455816526, 0.0021640324870540, 0.0015536485910017},
{381180.6061982756364159, 0.0024995559058215, 0.0022379720080268},
{616303.9117061324650422, 0.0030073875775644, 0.0025734414654958},
{895597.5194741163868457, 0.0033000929038094, 0.0029674023903279},
{1212926.6986328975763172, 0.0036547443692330, 0.0032262531238372},
{1569819.3025231477804482, 0.0038963930086719, 0.0035108644578197},
{1961720.3036374731454998, 0.0041719200105583, 0.0037184592710597},
{2390224.7439586673863232, 0.0043748970549824, 0.0039429849267811},
{2851858.3883706750348210, 0.0046007788945729, 0.0041160402409436},
{3348024.6846916377544403, 0.0047750064408801, 0.0043019700838008},
{3875971.0460229888558388, 0.0049664379612197, 0.0044504501894398},
{4436901.0818342072889209, 0.0051187380748729, 0.0046093454632354} },
{ {53364.6208360862583504, 0.0012497114876874, 0.0011528810870556},
{196883.5305899973027408, 0.0021226036373834, 0.0015238626681886},
{392402.7530665989033878, 0.0024517135437554, 0.0021951062969655},
{634448.8162027498474345, 0.0029498475646872, 0.0025241479343408},
{921965.0704707553377375, 0.0032369699042738, 0.0029105803287354},
{1248636.6355677845422179, 0.0035848630971150, 0.0031644773346666},
{1616035.7101678424514830, 0.0038219120813539, 0.0034436536313500},
{2019473.3365983855910599, 0.0040921992239974, 0.0036472793404191},
{2460591.1276245131157339, 0.0042913203892744, 0.0038675207383615},
{2935812.6174656222574413, 0.0045129147449989, 0.0040372701970130},
{3446582.3172755646519363, 0.0046838388420234, 0.0042196550248409},
{3990066.5975151504389942, 0.0048716434098436, 0.0043653010688614},
{4567504.3246421432122588, 0.0050210610571295, 0.0045211690899114} },
{ {54920.0045618931835634, 0.0012260065791730, 0.0011310116644740},
{202623.5400571221543942, 0.0020823533038134, 0.0014949256421750},
{403842.9107921904069372, 0.0024052317727846, 0.0021534607708372},
{652946.2249119259649888, 0.0028939430949023, 0.0024762576393338},
{948844.8665080888895318, 0.0031756404928885, 0.0028553751547666},
{1285040.3110601399093866, 0.0035169664943360, 0.0031044592776955},
{1663149.9575346545316279, 0.0037495456767882, 0.0033783546315796},
{2078348.3180719099473208, 0.0040147407133841, 0.0035781239451565},
{2532324.4769306601956487, 0.0042101145679550, 0.0037942024253055},
{3021397.7522735092788935, 0.0044275420298761, 0.0039607397941767},
{3547054.5133374510332942, 0.0045952554566690, 0.0041396799044374},
{4106378.5143148549832404, 0.0047795348363716, 0.0042825721820893},
{4700644.5610922640189528, 0.0049261507776888, 0.0044354985114700} },
{ {56505.3096859959041467, 0.0012029699216500, 0.0011097587950067},
{208473.9869933519512415, 0.0020432372024011, 0.0014668055415709},
{415503.1756303309812211, 0.0023600594844005, 0.0021129895432855},
{671799.5270971126155928, 0.0028396127704985, 0.0024297178067677},
{976241.8330183199141175, 0.0031160373690715, 0.0028017260903932},
{1322144.3956615363713354, 0.0034509801344361, 0.0030461328862265},
{1711170.6776941490825266, 0.0036792145349693, 0.0033148956256731},
{2138356.0360255911946297, 0.0039394597270925, 0.0035109170268587},
{2605437.9357729665935040, 0.0041311908115649, 0.0037229493946959},
{3108629.4745861836709082, 0.0043445675019140, 0.0038863649292726},
{3649459.6825643037445843, 0.0045091596062936, 0.0040619568762575},
{4224928.1077099619433284, 0.0046900118034611, 0.0042021726810371},
{4836346.1852047685533762, 0.0048339038214979, 0.0043522396907662} },
{ {58120.8211949129763525, 0.0011805766267051, 0.0010890995137551},
{214435.9231711243337486, 0.0020052131100082, 0.0014394718868617},
{427385.6437512590782717, 0.0023161479475807, 0.0020736488654009},
{691012.1123130774358287, 0.0027867980477715, 0.0023844781213734},
{1004160.8954160804860294, 0.0030580963589660, 0.0027495751871434},
{1359955.5599344312213361, 0.0033868330462887, 0.0029894351690261},
{1760106.5037070182152092, 0.0036108430738830, 0.0032532081230915},
{2199507.2784168650396168, 0.0038662754435100, 0.0034455860656781},
{2679944.6480547585524619, 0.0040544644552030, 0.0036536848017915},
{3197523.4661988005973399, 0.0042639022328525, 0.0038140654099785},
{3753816.2340836948715150, 0.0044254590880744, 0.0039864021775579},
{4345736.6890855319797993, 0.0046029785209106, 0.0041240159399570},
{4974633.5914727291092277, 0.0047442215559917, 0.0042713029599924} },
{ {59766.8240495076825027, 0.0011588029541206, 0.0010690119151433},
{220510.4003761564090382, 0.0019682407503251, 0.0014128956075927},
{439492.4115639119409025, 0.0022734506772727, 0.0020353970072640},
{710587.3699953368632123, 0.0027354430792781, 0.0023404905900726},
{1032606.9791340528754517, 0.0030017562427233, 0.0026988671695127},
{1398480.4744154992513359, 0.0033244575234312, 0.0029343060400865},
{1809966.0686604569200426, 0.0035443591861102, 0.0031932267911020},
{2261812.8332075932994485, 0.0037951107544311, 0.0033820618845781},
{2755857.7576824543066323, 0.0039798547220077, 0.0035863353428054},
{3288095.4089045627042651, 0.0041854613757608, 0.0037437647385506},
{3860142.5772051196545362, 0.0043440659287612, 0.0039129359034866},
{4468825.5695298882201314, 0.0045183435896476, 0.0040480193225438},
{5115531.1740785902366042, 0.0046570098684023, 0.0041926027793928} },
{ {61443.6032147722653463, 0.0011376262489007, 0.0010494750951832},
{226698.4703898340521846, 0.0019322816870721, 0.0013870489648932},
{451825.5752175496309064, 0.0022319233113480, 0.0019981941471772},
{730528.6896071011433378, 0.0026854945661703, 0.0022977094145425},
{1061585.0095825584139675, 0.0029469585928572, 0.0026495492883134},
{1437725.8096528504975140, 0.0032637889455325, 0.0028806881593368},
{1860758.0056205664295703, 0.0034796940494604, 0.0031348892812753},
{2325283.4883756586350501, 0.0037258920623049, 0.0033202784660923},
{2833190.4085547574795783, 0.0039072845108498, 0.0035208310608790},
{3380360.9845079584047198, 0.0041091639424687, 0.0036753899094754},
{3968457.1212232140824199, 0.0042648961540194, 0.0038414817958712},
{4594216.0602487456053495, 0.0044360197637569, 0.0039741039626355},
{5259063.3273411169648170, 0.0045721789196447, 0.0041160575112679} },
{ {63151.4436554731146316, 0.0011170248822638, 0.0010304690968282},
{233001.1850148661469575, 0.0018972992241130, 0.0013619054788745},
{464387.2309966406901367, 0.0021915234952825, 0.0019620022678194},
{750839.4606039032805711, 0.0026369016199574, 0.0022560908718403},
{1091099.9123272609431297, 0.0028936476228271, 0.0026015711833346},
{1477698.2362350402399898, 0.0032047656111673, 0.0028285267833551},
{1912490.9475842765532434, 0.0034167819488331, 0.0030781360674832},
{2389930.0318625546060503, 0.0036585490902650, 0.0032601727808756},
{2911955.7445660117082298, 0.0038366801977034, 0.0034571051644332},
{3474335.8747941711917520, 0.0040349325950237, 0.0036088712199205},
{4078778.2754162326455116, 0.0041878695725443, 0.0037719670453056},
{4721929.4726396510377526, 0.0043559237250917, 0.0039021945606180},
{5405254.4450379824265838, 0.0044896429138524, 0.0040415892084570} },
{ {64890.6303403962228913, 0.0010969781963102, 0.0010119748590143},
{239419.5959491097310092, 0.0018632583116649, 0.0013374398606886},
{477179.4751545203616843, 0.0021522107741235, 0.0019267850590719},
{771523.0724411292467266, 0.0025896156328306, 0.0022155932025627},
{1121156.6125395847484469, 0.0028417700450926, 0.0025548847547034},
{1518404.4246490278746933, 0.0031473285811226, 0.0027777696254808},
{1965173.5277148585300893, 0.0033555601094624, 0.0030229102941075},
{2455763.2516060429625213, 0.0035930147040189, 0.0032016846266100},
{2992166.9096231064759195, 0.0037679714491287, 0.0033950938567901},
{3570035.7615635590627789, 0.0039626934501140, 0.0035441420920535},
{4191124.4490860491059721, 0.0041129095734475, 0.0037043221057116},
{4851987.1178876366466284, 0.0042779758737497, 0.0038322191911666},
{5554128.9219343960285187, 0.0044093198820493, 0.0039691234158475} },
{ {66661.4482228343549650, 0.0010774664522256, 0.0009939741687610},
{245954.7551167552010156, 0.0018301254585105, 0.0013136279488004},
{490204.4039464217494242, 0.0021139464911510, 0.0018925078267755},
{792582.9145800272235647, 0.0025435901561578, 0.0021761765058513},
{1151760.0358326763380319, 0.0027912749379691, 0.0025094440421865},
{1559851.0456385419238359, 0.0030914215312754, 0.0027283667248054},
{2018814.3790744312573224, 0.0032959685405668, 0.0029691576333281},
{2522793.9356566513888538, 0.0035292247447207, 0.0031447564774092},
{3073837.0475865211337805, 0.0037010910474552, 0.0033347361764993},
{3667476.3266145237721503, 0.0038923758957062, 0.0034811389063783},
{4305514.0515115661546588, 0.0040399429363076, 0.0036384805203807},
{4984410.3072332488372922, 0.0042021001306530, 0.0037641091235574},
{5705711.1520519386976957, 0.0043311314793589, 0.0038985889844848} },
{ {68464.1823002060409635, 0.0010584707816645, 0.0009764496163228},
{252607.7142044176289346, 0.0017978686494970, 0.0012904466491099},
{503464.1136296142940409, 0.0020766936928151, 0.0018591374071845},
{814022.3765177190070972, 0.0024987807863837, 0.0021378026411093},
{1182915.1075851682107896, 0.0027421136207743, 0.0024652051119630},
{1602044.7696163065265864, 0.0030369906146676, 0.0026802703229869},
{2073422.1346687937621027, 0.0032379498884676, 0.0029168261516238},
{2591032.8718099878169596, 0.0034671178723043, 0.0030893333422214},
{3156979.3026084536686540, 0.0036359747268260, 0.0032759738474587},
{3766673.2517039105296135, 0.0038239124188299, 0.0034198008454834},
{4421965.4919534036889672, 0.0039688996528017, 0.0035743787588132},
{5119220.3519261628389359, 0.0041282237525901, 0.0036977986529029},
{5860025.5296183358877897, 0.0042550027945681, 0.0038299178968459} },
{ {70299.1175070876925020, 0.0010399731409568, 0.0009593845530830},
{259379.5250207709032111, 0.0017664572682929, 0.0012678738788036},
{516960.7004419826553203, 0.0020404170394603, 0.0018266420866338},
{835844.8476289815735072, 0.0024551450579461, 0.0021004351355569},
{1214626.7532252077944577, 0.0026942395364974, 0.0024221259503567},
{1644992.2670137763489038, 0.0029839843319410, 0.0026334347487436},
{2129005.4276237208396196, 0.0031814492988267, 0.0028658661841480},
{2660490.8483837042003870, 0.0034066354181728, 0.0030353626318963},
{3241606.8181800418533385, 0.0035725610188933, 0.0032187511382822},
{3867642.2188961636275053, 0.0037572384439270, 0.0033600697471941},
{4540497.1798107670620084, 0.0038997127592063, 0.0035119560634185},
{5256438.5631908113136888, 0.0040562771582831, 0.0036332249417322},
{6017096.4488884890452027, 0.0041808621712418, 0.0037630451030395} },
{ {72166.5388192662067013, 0.0010219562684100, 0.0009427630518471},
{266271.2393389316857792, 0.0017358620242950, 0.0012458885141328},
{530696.2606849416624755, 0.0020050827214887, 0.0017949915261048},
{858053.7173347255447879, 0.0024126423426406, 0.0020640390974427},
{1246899.8981850517448038, 0.0026476081415559, 0.0023801663639637},
{1688700.2087052704300731, 0.0029323534096340, 0.0025878163094209},
{2185572.8909430387429893, 0.0031264142870311, 0.0028162302168650},
{2731178.6528799943625927, 0.0033477212468814, 0.0029827940344972},
{3327732.7384777204133570, 0.0035107911081403, 0.0031630147299699},
{3970398.9094055872410536, 0.0036922921808228, 0.0033018899666852},
{4661127.5242406716570258, 0.0038323181789533, 0.0034511543055802},
{5396086.2520877309143543, 0.0039861937649772, 0.0035703278711312},
{6176948.3041389277204871, 0.0041086410396614, 0.0036979083667699} },
{ {74066.7312030861794483, 0.0010044036437981, 0.0009265698697178},
{273283.9090087239164859, 0.0017060548849795, 0.0012244703398492},
{544672.8905864665284753, 0.0019706583805355, 0.0017641566903579},
{880652.3753000824945047, 0.0023712337551040, 0.0020285811345018},
{1279739.4678848825860769, 0.0026021768023588, 0.0023392878858403},
{1733175.2649801208171993, 0.0028820506857070, 0.0025433731889414},
{2243133.1577384653501213, 0.0030727946165398, 0.0027678727757169},
{2803107.0735448412597179, 0.0032903216259439, 0.0029315793978969},
{3415370.2073297449387610, 0.0034506086957278, 0.0031087135918285},
{4074959.0055215540342033, 0.0036290144819465, 0.0032452082468364},
{4783874.9345861775800586, 0.0037666545747005, 0.0033919178503095},
{5538184.7307601999491453, 0.0039179098349144, 0.0035090499009044},
{6339605.4895643843337893, 0.0040382737587519, 0.0036344481205512} },
{ {75999.9796239327843068, 0.0009872994506033, 0.0009107904131548},
{280418.5856596627854742, 0.0016770090109665, 0.0012036000036852},
{558892.6864292843965814, 0.0019371130353722, 0.0017341097812028},
{903644.2108136157039553, 0.0023308820638001, 0.0019940292772747},
{1313150.3877604547888041, 0.0025579046977113, 0.0022994536872890},
{1778424.1064819556195289, 0.0028330310019731, 0.0025000653519535},
{2301694.8610932445153594, 0.0030205421842527, 0.0027207503224439},
{2876286.8982807216234505, 0.0032343851037049, 0.0028816726195286},
{3504532.3686381787993014, 0.0033919598714022, 0.0030557988645742},
{4181338.1888885470107198, 0.0035673487079412, 0.0031899735964060},
{4908757.8201266946271062, 0.0037026632090029, 0.0033341934290315},
{5682755.3089426010847092, 0.0038513643306946, 0.0034493359378549},
{6505092.3989174868911505, 0.0039696974673339, 0.0035726073296090} },
{ {77966.5690476871095598, 0.0009706285402581, 0.0008954107048973},
{287676.3212506811250933, 0.0016486986959489, 0.0011832589719668},
{573357.7443590019829571, 0.0019044170121467, 0.0017048241748401},
{927032.6133768287254497, 0.0022915516073920, 0.0019603529069768},
{1347137.5832465395797044, 0.0025147527272584, 0.0022606284947294},
{1824453.4037729722913355, 0.0027852511027625, 0.0024578544534745},
{2361266.6340650473721325, 0.0029696109128256, 0.0026748211564869},
{2950728.9150411379523575, 0.0031798623940399, 0.0028330295425232},
{3595232.3663231870159507, 0.0033347929929664, 0.0030042237503632},
{4289552.1412986274808645, 0.0035072406010733, 0.0031361371752884},
{5035794.5901442663744092, 0.0036402878133526, 0.0032779300198213},
{5829819.2989599835127592, 0.0037864987793072, 0.0033911332121424},
{6673433.4272646466270089, 0.0039028519441902, 0.0035123313637374} },
{ {79966.7844405958458083, 0.0009543763985983, 0.0008804173532267},
{295058.1674707488855347, 0.0016210993096324, 0.0011634294885590},
{588070.1608649484114721, 0.0018725418787212, 0.0016762743627274},
{950820.9724504525074735, 0.0022532082158967, 0.0019275226875568},
{1381705.9797534756362438, 0.0024726834251791, 0.0022227785116781},
{1871269.8273991148453206, 0.0027386695396128, 0.0024167037541468},
{2421857.1097154817543924, 0.0029199566491310, 0.0026300453227279},
{3026443.9118005102500319, 0.0031267062679461, 0.0027856078580415},
{3687483.3442600271664560, 0.0032790585728369, 0.0029539434093860},
{4399616.5445563308894634, 0.0034486381662854, 0.0030836521865136},
{5165003.6539362948387861, 0.0035794744647232, 0.0032230787346089},
{5979398.0117487590759993, 0.0037232571440503, 0.0033343911605590},
{6844652.9683014061301947, 0.0038376794762082, 0.0034535678767552} },
{ {82000.9107677678694017, 0.0009385291142758, 0.0008657975225722},
{302565.1761223843786865, 0.0015941872441807, 0.0011440945360081},
{603032.0318930626381189, 0.0018414603828821, 0.0016484358960387},
{975012.6774905695347115, 0.0022158191365328, 0.0018955105017418},
{1416860.5026853936724365, 0.0024316608788310, 0.0021858713450312},
{1918880.0479357696603984, 0.0026932465813950, 0.0023765780400092},
{2483474.9211205164901912, 0.0028715370686720, 0.0025863845244745},
{3103442.6765172434970737, 0.0030748714514731, 0.0027393670132165},
{3781298.4463678118772805, 0.0032247091710551, 0.0029049148622825},
{4511547.0804431876167655, 0.0033914915588429, 0.0030324737745907},
{5296403.4207742763683200, 0.0035201714692587, 0.0031695927130577},
{6131512.7585673546418548, 0.0036615857036537, 0.0032790613168199},
{7018775.4163527609780431, 0.0037741247341847, 0.0033962666927765} },
{ {84069.2329984290263383, 0.0009230733488899, 0.0008515389060323},
{310198.3989919448504224, 0.0015679398636918, 0.0011252377989405},
{618245.4539277047151700, 0.0018111463940472, 0.0016212853332890},
{999611.1179578256560490, 0.0021793529636992, 0.0018642893907709},
{1452606.0775484144687653, 0.0023916506521522, 0.0021498759357582},
{1967290.7359233305323869, 0.0026489441296880, 0.0023374435473075},
{2546128.7013430213555694, 0.0028243115854926, 0.0025438020416506},
{3181735.9971499461680651, 0.0030243145293793, 0.0026942681244058},
{3876690.8165394370444119, 0.0031716992945083, 0.0028570968983370},
{4625359.4307565903291106, 0.0033357529785753, 0.0029825589296111},
{5430012.2999441279098392, 0.0034623292527113, 0.0031174270224258},
{6286184.8506268234923482, 0.0036014329386941, 0.0032250972081148},
{7195825.1656540641561151, 0.0037121346556925, 0.0033403796991838} },
{ {86172.0360907409194624, 0.0009079963089935, 0.0008376296996130},
{317958.8878635013825260, 0.0015423354565871, 0.0011068436295188},
{633712.5231413941364735, 0.0017815748481576, 0.0015948001909354},
{1024619.6833103973185644, 0.0021437795730986, 0.0018338334975522},
{1488947.6296422358136624, 0.0023526197134151, 0.0021147624935027},
{2016508.5619286315049976, 0.0026057256389213, 0.0022992678912651},
{2609827.0834604888223112, 0.0027782412671789, 0.0025022626533914},
{3261334.6616552476771176, 0.0029749938544024, 0.0026502738954245},
{3973673.5986829428002238, 0.0031199853019406, 0.0028104499888079},
{4741069.2772977519780397, 0.0032813765701699, 0.0029338663968796},
{5565848.7007397897541523, 0.0034059002570797, 0.0030665385632159},
{6443435.5991862611845136, 0.0035427494240817, 0.0031724542574021},
{7375826.6104034855961800, 0.0036516583346503, 0.0032858607457262} },
{ {88309.6050184066407382, 0.0008932857194667, 0.0008240585775804},
{325847.6945196032756940, 0.0015173531905786, 0.0010888970151758},
{649435.3358182657975703, 0.0017527216964814, 0.0015689588962657},
{1050041.7630040601361543, 0.0021090700594535, 0.0018041180130854},
{1525890.0847664873581380, 0.0023145363670788, 0.0020805024348757},
{2066540.1965040217619389, 0.0025635560410556, 0.0022620199991127},
{2674578.7005474162288010, 0.0027332887547346, 0.0024617325653208},
{3342249.4580425228923559, 0.0029268694615623, 0.0026073485403976},
{4072259.9367143157869577, 0.0030695253142011, 0.0027649362053007},
{4858692.3018405409529805, 0.0032283183289432, 0.0028863565917348},
{5703931.0324375852942467, 0.0033508388429698, 0.0030168859802377},
{6603286.3154730219393969, 0.0034854877278894, 0.0031210896914702},
{7558804.1448193592950702, 0.0035926469170530, 0.0032326655491871} },
{ {90482.2247422550572082, 0.0008789297986450, 0.0008108146695257},
{333865.8707438627607189, 0.0014929730704708, 0.0010713835474707},
{665415.9882258292054757, 0.0017245638555569, 0.0015437407446452},
{1075880.7464531932491809, 0.0020751966778486, 0.0017751191259054},
{1563438.3677637816872448, 0.0022773701894581, 0.0020470683252300},
{2117392.3102083597332239, 0.0025224016745162, 0.0022256700467548},
{2740392.1856480445712805, 0.0026894181868821, 0.0024221793407124},
{3424491.1742211412638426, 0.0028799029872181, 0.0025654577109877},
{4172462.9744796333834529, 0.0030202791295798, 0.0027205191425629},
{4978244.1861337386071682, 0.0031765360120635, 0.0028399915189117},
{5844277.7043069899082184, 0.0032971011976020, 0.0029684295784225},
{6765758.3107161093503237, 0.0034296023156897, 0.0030709624540631},
{7744782.1631016489118338, 0.0035350535026153, 0.0031807516035247} },
{ {92690.1802264705474954, 0.0008649172345698, 0.0007978875384605},
{342014.4683182212756947, 0.0014691758979341, 0.0010542893935316},
{681656.5766781697748229, 0.0016970791620772, 0.0015191258567369},
{1102140.0231750928796828, 0.0020421327882810, 0.0017468139742627},
{1601597.4043896202929318, 0.0022410919679293, 0.0020144338237222},
{2169071.5736298817209899, 0.0024822302169870, 0.0021901893991482},
{2807276.1718042376451194, 0.0026465951285810, 0.0023835718355277},
{3508070.5981255401857197, 0.0028340575929120, 0.0025245684275677},
{4274295.8560454910621047, 0.0029722081439251, 0.0026771638456910},
{5099740.6121394000947475, 0.0031259890545374, 0.0027947346967685},
{5986907.1256425390020013, 0.0032446452477610, 0.0029211312436195},
{6930872.8961479328572750, 0.0033750494602745, 0.0030220331237886},
{7933785.0595018826425076, 0.0034788330518076, 0.0031300780949515} },
{ {94933.7564414542866871, 0.0008512371628211, 0.0007852671603010},
{350294.5390434663859196, 0.0014459432337676, 0.0010376012685133},
{698159.1971846550004557, 0.0016702463288105, 0.0014950951399083},
{1128822.9827515231445432, 0.0020098528032440, 0.0017191806010235},
{1640372.1201049725059420, 0.0022056736436340, 0.0019825736313660},
{2221584.6572574083693326, 0.0024430106220608, 0.0021555505537630},
{2875239.2922328566201031, 0.0026047865035820, 0.0023458801371614},
{3592998.5179149438627064, 0.0027892978929967, 0.0024846490144231},
{4377771.7249431228265166, 0.0029252752750422, 0.0026348367414185},
{5223197.2613960346207023, 0.0030766384899692, 0.0027505510853532},
{6131837.7057402618229389, 0.0031934305776558, 0.0028749543676040},
{7098651.3830003617331386, 0.0033217871562776, 0.0029742638366538},
{8125837.2281963331624866, 0.0034239422979809, 0.0030806058217532} },
{ {97213.2383480831049383, 0.0008378791453259, 0.0007729439044884},
{358707.1346840576734394, 0.0014232573622419, 0.0010213064096086},
{714925.9462469252757728, 0.0016440449033354, 0.0014716302510405},
{1155933.0144150482956320, 0.0019783321380734, 0.0016921979113245},
{1679767.4402814172208309, 0.0021710882571262, 0.0019514634419953},
{2274938.2317294958047569, 0.0024047130593373, 0.0021217270871315},
{2944290.1798163629136980, 0.0025639605305419, 0.0023090755064101},
{3679285.7213194794021547, 0.0027455898865999, 0.0024456690382289},
{4482903.7256906228139997, 0.0028794448912800, 0.0025935055729502},
{5348629.8159328652545810, 0.0030284468756088, 0.0027074070186630},
{6279087.8538792841136456, 0.0031434183512202, 0.0028298637772605},
{7269115.0825262600556016, 0.0032697750395220, 0.0029276182128326},
{8320963.0634240182116628, 0.0033703396644370, 0.0030322971185259} },
{ {99528.9109311026113573, 0.0008248331505206, 0.0007609085153992},
{367253.3070316271623597, 0.0014011012570869, 0.0010053925519889},
{731958.9200452539371327, 0.0016184552291724, 0.0014487135615094},
{1183473.5077193595934659, 0.0019475471646286, 0.0016658456316153},
{1719788.2903435393236578, 0.0021373098971909, 0.0019210798958523},
{2329138.9675178257748485, 0.0023673088575053, 0.0020886936047537},
{3014437.4677703366614878, 0.0025240866627208, 0.0022731303227042},
{3766942.9961214796639979, 0.0027029008930261, 0.0024075992501181},
{4589705.0014507044106722, 0.0028346827440318, 0.0025531393386582},
{5476053.9573579514399171, 0.0029813782213331, 0.0026652701404837},
{6428675.9794538524001837, 0.0030945712386171, 0.0027858256675752},
{7442285.3059249268844724, 0.0032189743107059, 0.0028820612873820},
{8519186.9593305625021458, 0.0033179851857905, 0.0029851157844260} },
{ {101881.0591220081405481, 0.0008120895344490, 0.0007491520951136},
{375934.1078704469255172, 0.0013794585497286, 0.0009898479052448},
{749260.2148364745080471, 0.0015934584088593, 0.0014263281240159},
{1211447.8521205601282418, 0.0019174751664468, 0.0016401042722558},
{1760439.5957245859317482, 0.0021043136522487, 0.0018914005356810},
{2384193.5353212072513998, 0.0023307704509776, 0.0020564256931011},
{3085689.7890818323940039, 0.0024851355308835, 0.0022380180321802},
{3855981.1313154911622405, 0.0026611994907840, 0.0023704115307928},
{4698188.6969936201348901, 0.0027909559038076, 0.0025137082338245},
{5605485.3677300279960036, 0.0029353979227061, 0.0026241093436949},
{6580620.4913231525570154, 0.0030468533466910, 0.0027428075382929},
{7618183.3639603592455387, 0.0031693476632344, 0.0028375594447290},
{8720533.3102060668170452, 0.0032668404336738, 0.0029390270154609} },
{ {104269.9679282777360640, 0.0007996390229133, 0.0007376660869945},
{384750.5889834773843177, 0.0013583134989905, 0.0009746611317755},
{766831.9268790214555338, 0.0015690362690307, 0.0014044576412803},
{1239859.4370880082715303, 0.0018880942971207, 0.0016149550909970},
{1801726.2818533892277628, 0.0020720755644671, 0.0018624037652413},
{2440108.6053759646601975, 0.0022950713289903, 0.0020248998745883},
{3158055.7768371524289250, 0.0024470788892517, 0.0022037130987552},
{3946410.9133082902990282, 0.0026204554598147, 0.0023340788384941},
{4808367.9548888280987740, 0.0027482326997961, 0.0024751835956406},
{5736939.7287483774125576, 0.0028904726974337, 0.0025838947128446},
{6734939.7990000825375319, 0.0030002301532496, 0.0027007781339798},
{7796830.5693169757723808, 0.0031208592148250, 0.0027940803566715},
{8925026.5102308560162783, 0.0032168684464203, 0.0028939973401946} },
{ {106695.9222807287587784, 0.0007874726946337, 0.0007264422598955},
{393703.8021488533122465, 0.0013376509624257, 0.0009598213259876},
{784676.1524322773329914, 0.0015451713274013, 0.0013830864362437},
{1268711.6520660782698542, 0.0018593835406204, 0.0015903800590447},
{1843653.2741573767270893, 0.0020405725862634, 0.0018340688100149},
{2496890.8485653339885175, 0.0022601859875556, 0.0019940935648813},
{3231544.0641155759803951, 0.0024098895643335, 0.0021701909575252},
{4038243.1314192921854556, 0.0025806397268122, 0.0022985751599447},
{4920255.9203234314918518, 0.0027064826625074, 0.0024375378511248},
{5870432.7216497287154198, 0.0028465705251671, 0.0025445974698511},
{6891652.3125756755471230, 0.0029546684445957, 0.0026597073871704},
{7978248.2318494832143188, 0.0030734744428170, 0.0027515929236609},
{9132690.9537626057863235, 0.0031680336624330, 0.0028499945591244} },
{ {109159.2071572146087419, 0.0007755819652418, 0.0007154726936300},
{402794.7991771756787784, 0.0013174563691773, 0.0009453179946920},
{802794.9877299517393112, 0.0015218467614674, 0.0013621994240945},
{1298007.8865397486370057, 0.0018313226736195, 0.0015663618285858},
{1886225.4980723389890045, 0.0020097825391582, 0.0018063756798322},
{2554546.9355573807843029, 0.0022260898839841, 0.0019639850323459},
{3306163.2839833968318999, 0.0023735414064797, 0.0021374279709071},
{4131488.5723806163296103, 0.0025417243134213, 0.0022638754636552},
{5033865.7372026145458221, 0.0026656764696479, 0.0024007444677778},
{6005980.0287790177389979, 0.0028036605906327, 0.0025061899224330},
{7050776.4398839985951781, 0.0029101362566333, 0.0026195663646886},
{8162457.6631434932351112, 0.0030271601228430, 0.0027100672192498},
{9343551.0347565207630396, 0.0031203018570556, 0.0028069876870509} },
{ {111660.1075331025349442, 0.0007639585720700, 0.0007047497648457},
{412024.6317784428829327, 0.0012977156943620, 0.0009311410385682},
{821190.5290916174417362, 0.0014990463788251, 0.0013417820854469},
{1327751.5299360135104507, 0.0018038922299318, 0.0015428837022333},
{1929447.8790176943875849, 0.0019796840747082, 0.0017793051337951},
{2613083.5365629093721509, 0.0021927593936590, 0.0019345533597297},
{3381922.0695154727436602, 0.0023380092439301, 0.0021054013868317},
{4226158.0254145273938775, 0.0025036822871150, 0.0022299556558687},
{5149210.5485418951138854, 0.0026257858946153, 0.0023647779067726},
{6143597.3322174204513431, 0.0027617132296043, 0.0024686454153227},
{7212330.5911246137693524, 0.0028666028188103, 0.0025803272166469},
{8349480.1744680991396308, 0.0029818842707244, 0.0026694744373237},
{9557631.1475241724401712, 0.0030736400827250, 0.0027649468985665} },
{ {114198.9083342597441515, 0.0007525945599020, 0.0006942661337365},
{421394.3518511684378609, 0.0012784154345864, 0.0009172807344549},
{839864.8727440193761140, 0.0014767545891069, 0.0013218204412629},
{1357945.9717274061404169, 0.0017770734667718, 0.0015199296039098},
{1973325.3424310223199427, 0.0019502566375165, 0.0017528386466663},
{2672507.3222816684283316, 0.0021601717692438, 0.0019057784078074},
{3458829.0537694804370403, 0.0023032688392668, 0.0020740892993213},
{4322262.2775546489283442, 0.0024664877146686, 0.0021967925386976},
{5266303.4982057949528098, 0.0025867837577164, 0.0023296135786886},
{6283300.3131836894899607, 0.0027206998776752, 0.0024319382840717},
{7376333.1753582777455449, 0.0028240385010679, 0.0025419631281349},
{8539337.0770855210721493, 0.0029376160873191, 0.0026297868421241},
{9774955.6862797606736422, 0.0030280166122901, 0.0027238434763114} },
{ {116775.8945911312912358, 0.0007414822672071, 0.0006840147317082},
{430905.0111416259896941, 0.0012595425849595, 0.0009037277187885},
{858820.1148968549678102, 0.0014549563772582, 0.0013023010287873},
{1388594.6013350524008274, 0.0017508483326835, 0.0014974840514020},
{2017862.8137497173156589, 0.0019214804301776, 0.0017269583772505},
{2732824.9632679619826376, 0.0021283051017716, 0.0018776407809256},
{3536892.8698143563233316, 0.0022692968481394, 0.0020434706109190},
{4419812.1172420755028725, 0.0024301156180003, 0.0021643637704015},
{5385157.7309982227161527, 0.0025486438800434, 0.0022952278014221},
{6425104.6536490982398391, 0.0026805930218005, 0.0023960438111518},
{7542802.6018758453428745, 0.0027824147635541, 0.0025044482733996},
{8732049.6822053547948599, 0.0028943259062911, 0.0025909777210275},
{9995549.0452416520565748, 0.0029834008851637, 0.0026836497620198} },
{ {119391.3512223450234160, 0.0007306143133448, 0.0006739887491579},
{440557.6613825123058632, 0.0012410846170968, 0.0008904729716213},
{878058.3517039765138179, 0.0014336372783070, 0.0012832108788399},
{1419700.8083495213650167, 0.0017251994372308, 0.0014755321300489},
{2063065.2183929502498358, 0.0018933363799459, 0.0017016471382455},
{2794043.1300930259749293, 0.0020971382839274, 0.0018501217942683},
{3616122.1508116922341287, 0.0022360707800264, 0.0020135249972490},
{4518818.3323220359161496, 0.0023945419322845, 0.0021326478277943},
{5505786.3903586799278855, 0.0025113410395461, 0.0022615977603381},
{6569026.0354456910863519, 0.0026413661541282, 0.0023609381843030},
{7711757.2799616884440184, 0.0027417041088146, 0.0024677577724240},
{8927639.3010477963835001, 0.0028519851444965, 0.0025530213394388},
{10219435.6185777224600315, 0.0029397634562420, 0.0026443391099649} },
{ {122045.5632080880604917, 0.0007199835862158, 0.0006641816243175},
{450353.3543871118454263, 0.0012230294583420, 0.0008775078016899},
{897581.6796721245627850, 0.0014127833532652, 0.0012645374943580},
{1451267.9822052735835314, 0.0017001100220691, 0.0014540594681942},
{2108937.4817369142547250, 0.0018658061071645, 0.0016768883677402},
{2856168.4932968877255917, 0.0020666509750991, 0.0018232034428486},
{3696525.5295823072083294, 0.0022035689611160, 0.0019842328731789},
{4619291.7107287338003516, 0.0023597434661436, 0.0021016239705410},
{5628202.6202256828546524, 0.0024748509292071, 0.0022287014704895},
{6715080.1403296804055572, 0.0026029937282939, 0.0023265984571099},
{7883215.6188788348808885, 0.0027018800365144, 0.0024318676496158},
{9126127.2448324039578438, 0.0028105662548693, 0.0025158928981867},
{10446639.8005300778895617, 0.0028970759474790, 0.0026058858427878} },
{ {124738.8155129955994198, 0.0007095832306779, 0.0006545870324953},
{460293.1419441251200624, 0.0012053654720320, 0.0008648238321086},
{917392.1949416891438887, 0.0013923811665752, 0.0012462688296469},
{1483299.5120449806563556, 0.0016755639336848, 0.0014330522134278},
{2155484.5293932287022471, 0.0018388718952458, 0.0016526661020343},
{2919207.7234575008042157, 0.0020368235681640, 0.0017968683720117},
{3778111.6395226037129760, 0.0021717704988895, 0.0019555753608135},
{4721243.0404534051194787, 0.0023256978639659, 0.0020712722071554},
{5752419.5644760960713029, 0.0024391501179042, 0.0021965177404508},
{6863282.6500927181914449, 0.0025654511179271, 0.0022930025114283},
{8057196.0279092444106936, 0.0026629170003158, 0.0023967547947022},
{9327534.8248350806534290, 0.0027700426817401, 0.0024795684930622},
{10677185.9853174109011889, 0.0028553110017815, 0.0025682652095714} },
{ {127471.3931028858496575, 0.0006994066374010, 0.0006451988758833},
{470378.0758383343927562, 0.0011880814387062, 0.0008524129868028},
{937491.9937701028538868, 0.0013724177640576, 0.0012283932714055},
{1515798.7877083974890411, 0.0016515455973171, 0.0014124970103056},
{2202711.2865307722240686, 0.0018125166622062, 0.0016289649500015},
{2983167.4911419888958335, 0.0020076371580287, 0.0017710998494342},
{3860889.1136822202242911, 0.0021406552486376, 0.0019275342591045},
{4824683.1094425618648529, 0.0022923835699915, 0.0020415732628382},
{5878450.3670156337320805, 0.0024042160124309, 0.0021650261384461},
{7013649.2465392183512449, 0.0025287145771718, 0.0022601290218054},
{8233716.9163781758397818, 0.0026247903670449, 0.0023623969254790},
{9531883.3522138185799122, 0.0027303888183195, 0.0024440250762721},
{10911098.5671072900295258, 0.0028144422393343, 0.0025314533460715} },
{ {130243.5809433061513118, 0.0006894474324857, 0.0006360112737778},
{480609.2078468241961673, 0.0011711665383234, 0.0008402674776451},
{957883.1724173265974969, 0.0013528806527109, 0.0012108996198301},
{1548769.1987597648985684, 0.0016280399922605, 0.0013923809791604},
{2250622.6787615045905113, 0.0017867239335405, 0.0016057700685093},
{3048054.4668375975452363, 0.0019790735116371, 0.0017458817386241},
{3944866.5848161331377923, 0.0021102037815979, 0.0019000920148762},
{4929622.7056660847738385, 0.0022597797941933, 0.0020125085488933},
{6006308.1717632999643683, 0.0023700268219765, 0.0021342069597667},
{7166195.6114724613726139, 0.0024927612031496, 0.0022279574216122},
{8412796.6934828348457813, 0.0025874763777375, 0.0023287725524769},
{9739194.1382757425308228, 0.0026915799663240, 0.0024092404199510},
{11148401.9401367865502834, 0.0027744442159883, 0.0024954272368271} },
{ {133055.6639984187204391, 0.0006796994673736, 0.0006270185534610},
{490987.5897651334526017, 0.0011546103332782, 0.0008283797922527},
{978567.8271244106581435, 0.0013337577810052, 0.0011937770711868},
{1582214.1341868285089731, 0.0016050326285012, 0.0013726916957450},
{2299223.6314639006741345, 0.0017614778165128, 0.0015830671392197},
{3113875.3212104770354927, 0.0019511150395918, 0.0017211984735458},
{4030052.6860946863889694, 0.0020803973545436, 0.0018732316953814},
{5036072.6170688942074776, 0.0022278664799394, 0.0019840601335664},
{6136006.1225878614932299, 0.0023365615242170, 0.0021040411959572},
{7320937.4266529353335500, 0.0024575689003692, 0.0021964678709044},
{8594453.7684526350349188, 0.0025509521107662, 0.0022958609453891},
{9949488.4942161608487368, 0.0026535922976085, 0.0023751930813524},
{11389120.4985515605658293, 0.0027352923837308, 0.0024601646792305} },
{ {135907.9272423017246183, 0.0006701568093650, 0.0006182152413962},
{501514.2733799691195600, 0.0011384027523105, 0.0008167426822993},
{999548.0542046736227348, 0.0013150375202907, 0.0011770152012214},
{1616136.9834927704650909, 0.0015825095240915, 0.0013534171722690},
{2348519.0700738723389804, 0.0017367629756880, 0.0015608423464920},
{3180636.7248551975935698, 0.0019237447689719, 0.0016970350345491},
{4116456.0504534081555903, 0.0020512178810864, 0.0018469369621270},
{5144043.6316601615399122, 0.0021966242731116, 0.0019562107143620},
{6267557.3634070586413145, 0.0023037998329045, 0.0020745105055790},
{7477890.3739258730784059, 0.0024231163468082, 0.0021656412257560},
{8778706.5507640559226274, 0.0025151954466567, 0.0022636421010311},
{10162787.7311547473073006, 0.0026164028176278, 0.0023418623698609},
{11633278.6364548895508051, 0.0026969630531304, 0.0024256442492872} },
{ {138800.6556306824204512, 0.0006608137326270, 0.0006095960548643},
{512190.3104660014505498, 0.0011225340751925, 0.0008053491525047},
{1020825.9498247401788831, 0.0012967086471518, 0.0011606039491327},
{1650541.1365444806870073, 0.0015604571842291, 0.0013345458389271},
{2398513.9200040264986455, 0.0017125646096299, 0.0015390823563193},
{3248345.3481981642544270, 0.0018969463177362, 0.0016733769256329},
{4204085.3116855286061764, 0.0020226479041576, 0.0018211920461428},
{5253546.5373654933646321, 0.0021660344928800, 0.0019289435918581},
{6400975.0381087688729167, 0.0022717221673308, 0.0020455971861533},
{7637070.1348080579191446, 0.0023893829616656, 0.0021354590092948},
{8965573.4495323840528727, 0.0024801850346683, 0.0022320967130646},
{10379113.1602782532572746, 0.0025799893307300, 0.0023092283154830},
{11880900.7484025098383427, 0.0026594333575584, 0.0023918452690621} },
{ {141734.1341312976437621, 0.0006516647095659, 0.0006011558941049},
{523016.7527822588454001, 0.0011069949181350, 0.0007941924500311},
{1042403.6103246328420937, 0.0012787603265320, 0.0011445336025659},
{1685429.9825890124775469, 0.0015388625808840, 0.0013160665266155},
{2449213.1067368206568062, 0.0016888684287022, 0.0015177742963103},
{3317007.8617663201875985, 0.0018707038699314, 0.0016502101527582},
{4292949.1021964754909277, 0.0019946705699749, 0.0017959817241579},
{5364592.1221703514456749, 0.0021360791036600, 0.0019022426445118},
{6536272.2908003982156515, 0.0022403096229151, 0.0020172841478274},
{7798492.3917245985940099, 0.0023563488746439, 0.0021059033838618},
{9155072.8744490966200829, 0.0024459002609359, 0.0022012061429790},
{10598486.0935165546834469, 0.0025443304070522, 0.0022772716390237},
{12132011.2285470198839903, 0.0026226812191120, 0.0023587477758357} },
{ {144708.6477131554856896, 0.0006427044026761, 0.0005928898346517},
{533994.6521600113483146, 0.0010917762199489, 0.0007832660545006},
{1064283.1317099316511303, 0.0012611820957683, 0.0011287947831923},
{1720806.9110910699237138, 0.0015177131335639, 0.0012979684503610},
{2500621.5556059582158923, 0.0016656606340943, 0.0014969057366965},
{3386630.9362979806028306, 0.0018450021528223, 0.0016275212027394},
{4383056.0551822362467647, 0.0019672696031814, 0.0017712912961833},
{5477191.1740247206762433, 0.0021067406887006, 0.0018760923049107},
{6673462.2649765117093921, 0.0022095439435339, 0.0019895548880206},
{7962172.8256401224061847, 0.0023239948967520, 0.0020769571247206},
{9347223.2347740735858679, 0.0024123212182011, 0.0021709523925858},
{10820927.8412001859396696, 0.0025094053511002, 0.0022459737235650},
{12386634.4702687729150057, 0.0025866853162717, 0.0023263324924797} },
{ {147724.4813413673255127, 0.0006339276566986, 0.0005847931203483},
{545125.0603665281087160, 0.0010768692286504, 0.0007725636685865},
{1086466.6107970767188817, 0.0012439638493044, 0.0011133784330116},
{1756675.3114996249787509, 0.0014969966910198, 0.0012802411936443},
{2552744.1921953945420682, 0.0016429278976228, 0.0014764646721325},
{3457221.2418500953353941, 0.0018198264142832, 0.0016052970240375},
{4474414.8037328105419874, 0.0019404292832365, 0.0017471065640733},
{5591354.4809440430253744, 0.0020780024247029, 0.0018504775370278},
{6812558.1044838372617960, 0.0021794074949471, 0.0019623934673552},
{8128127.1191670252010226, 0.0022923024924748, 0.0020486035950104},
{9542042.9373801238834858, 0.0023794286769422, 0.0021413180778745},
{11046459.7143252398818731, 0.0024751941717301, 0.0022153165873409},
{12644794.8685286454856396, 0.0025514250529990, 0.0022945807995139} },
{ {150781.9199808543489780, 0.0006253294912632, 0.0005768611562193},
{556409.0292482834775001, 0.0010622654892185, 0.0007620792086520},
{1108956.1433440954424441, 0.0012270958241691, 0.0010982758013247},
{1793038.5732890048529953, 0.0014767015137706, 0.0012628746933056},
{2605585.9416406941600144, 0.0016206573426353, 0.0014564395044691},
{3528785.4496421930380166, 0.0017951624018125, 0.0015835250076476},
{4567033.9809542177245021, 0.0019141344216697, 0.0017234138111771},
{5707092.8306152168661356, 0.0020498480577462, 0.0018253838146368},
{6953572.9542501419782639, 0.0021498832395487, 0.0019357844867239},
{8296370.9538031378760934, 0.0022612537532235, 0.0020208267216846},
{9739550.3956831730902195, 0.0023472040578985, 0.0021122864038771},
{11275103.0252388529479504, 0.0024416775536096, 0.0021852828579488},
{12906516.8184514008462429, 0.0025168805292974, 0.0022634747082428} },
{ {153881.2485980928177014, 0.0006169050937428, 0.0005690895022306},
{567847.6102958885021508, 0.0010479568312708, 0.0007518067962730},
{1131753.8257663156837225, 0.0012105685861498, 0.0010834784322755},
{1829900.0859151547774673, 0.0014568162575070, 0.0012458592252558},
{2659151.7296048519201577, 0.0015988365258077, 0.0014368190262390},
{3601330.2302434230223298, 0.0017709963423133, 0.0015621929693443},
{4660922.2198879253119230, 0.0018883703410531, 0.0017001997826199},
{5824417.0115619972348213, 0.0020222618802999, 0.0018007971007100},
{7096519.9567313157021999, 0.0021209547123808, 0.0019097130655732},
{8466920.0097855199128389, 0.0022308313720995, 0.0019936109729434},
{9939764.0162322316318750, 0.0023156294059270, 0.0020838411409966},
{11506879.0860486850142479, 0.0024088368299966, 0.0021558557477299},
{13171824.7114289272576571, 0.0024830325132872, 0.0022329968354619} },
{ {157022.7521581352048088, 0.0006086498126125, 0.0005614738668362},
{579441.8557232191087678, 0.0010339353577921, 0.0007417407500282},
{1154861.7543109112884849, 0.0011943730165900, 0.0010689781530944},
{1867263.2388327571097761, 0.0014373299571787, 0.0012291853909678},
{2713446.4821652984246612, 0.0015774534197156, 0.0014175924050303},
{3674862.2537845252081752, 0.0017473149229004, 0.0015412891325714},
{4756088.1536960890516639, 0.0018631228539714, 0.0016774516671484},
{5943337.8111246963962913, 0.0019952287093956, 0.0017767038277163},
{7241412.2567312987521291, 0.0020926059981244, 0.0018841648210100},
{8639789.9719576127827168, 0.0022010186198423, 0.0019669413363253},
{10142702.2082626000046730, 0.0022846873647011, 0.0020559666024908},
{11741809.2043412141501904, 0.0023766539568045, 0.0021270190302833},
{13440742.9440866224467754, 0.0024498624143439, 0.0022031303789899} },
{ {160206.7156218781601638, 0.0006005591509446, 0.0005540101012525},
{591192.8170453528873622, 0.0010201934341900, 0.0007318755775085},
{1178282.0252415405120701, 0.0011785002997772, 0.0010547670626577},
{1905131.4215127204079181, 0.0014182320119099, 0.0012128441043805},
{2768475.1230857451446354, 0.0015564963963066, 0.0013987491684518},
{3749388.1909788767807186, 0.0017241052725806, 0.0015208021122383},
{4852540.4152346439659595, 0.0018383782440169, 0.0016551570790549},
{6063866.0175791895017028, 0.0019687338657080, 0.0017530908790194},
{7388262.9979763841256499, 0.0020648217092830, 0.0018591258479163},
{8814996.5208272356539965, 0.0021717993217661, 0.0019408032980840},
{10348383.3838814217597246, 0.0022543611535390, 0.0020286476222953},
{11979914.6953842025250196, 0.0023451114877722, 0.0020987570181144},
{13713295.9114240929484367, 0.0024173522577423, 0.0021738590946622} },
{ {163433.4239339850901160, 0.0005926287603174, 0.0005466941935933},
{603101.5461710355011746, 0.0010067236779189, 0.0007222059678825},
{1202016.7346660655457526, 0.0011629419109786, 0.0010408375208616},
{1943508.0233999427873641, 0.0013995121706521, 0.0011968265795013},
{2824242.5791858965530992, 0.0015359542110801, 0.0013802791899451},
{3824914.7122696125879884, 0.0017013549448522, 0.0015007208990294},
{4950287.6378100700676441, 0.0018141232467846, 0.0016333040415949},
{6186012.4190599489957094, 0.0019427631536465, 0.0017299455708840},
{7537085.3242946416139603, 0.0020375869652805, 0.0018345827001812},
{8992555.3366062249988317, 0.0021431578360530, 0.0019151828233610},
{10556825.9475195370614529, 0.0022246345440651, 0.0020018695350366},
{12221216.8673329409211874, 0.0023141925510004, 0.0020710545413389},
{13989508.0049914922565222, 0.0023854846602867, 0.0021451672741117} },
{ {166703.1621617642813362, 0.0005848544349368, 0.0005395222637129},
{615169.0949935502139851, 0.0009935189485335, 0.0007127267847400},
{1226067.9792894960846752, 0.0011476896050047, 0.0010271821382380},
{1982396.4339260540436953, 0.0013811605182808, 0.0011811243186901},
{2880753.7754872213117778, 0.0015158159879967, 0.0013621726751235},
{3901448.4885831824503839, 0.0016790519007319, 0.0014810348449572},
{5049338.4545121639966965, 0.0017903450323424, 0.0016118809707765},
{6309787.8034507380798459, 0.0019173028423428, 0.0017072556355630},
{7687892.3796387249603868, 0.0020108873726015, 0.0018105223724665},
{9172482.1021185740828514, 0.0021150790321875, 0.0018900663378176},
{10768048.3131833225488663, 0.0021954918387983, 0.0019756181559356},
{12465737.0323158353567123, 0.0022838808263188, 0.0020438969272383},
{14269403.6206939611583948, 0.0023542428071162, 0.0021170397238367} },
{ {170016.2151473039702978, 0.0005772321061329, 0.0005324905578118},
{627396.5148060958599672, 0.0009805723382604, 0.0007034330592115},
{1250437.8549143497366458, 0.0011327354052367, 0.0010137937662415},
{2021800.0426306731533259, 0.0013631674625733, 0.0011657291012824},
{2938013.6374242207966745, 0.0014960712050901, 0.0013444201487679},
{3978996.1896568052470684, 0.0016571844931859, 0.0014617336488953},
{5149701.4978230008855462, 0.0017670311881426, 0.0015908766600163},
{6435202.9584385817870498, 0.0018923396475609, 0.0016850092049269},
{7840697.3078846139833331, 0.0019847090057534, 0.0017869322831072},
{9354792.5000973306596279, 0.0020875482722407, 0.0018654407086507},
{10982068.8875225652009249, 0.0021669178504492, 0.0019498797621988},
{12713496.5008807983249426, 0.0022541605238069, 0.0020172699809315},
{14553007.1518814694136381, 0.0023236104295903, 0.0020894617449754} },
{ {173372.8679113059188239, 0.0005697578369251, 0.0005255954437101},
{639784.8579265979351476, 0.0009678771629321, 0.0006943199835134},
{1275128.4579576910473406, 0.0011180715932151, 0.0010006654877979},
{2061722.2388544154819101, 0.0013455237216646, 0.0011506329728428},
{2996027.0904349531047046, 0.0014767096807267, 0.0013270124424724},
{4057564.4873031000606716, 0.0016357414517383, 0.0014428073432666},
{5251385.4010480130091310, 0.0017441697029283, 0.0015702802655038},
{6562268.6721697673201561, 0.0018678607143255, 0.0016631947950027},
{7995513.2529633119702339, 0.0019590383891681, 0.0017638002575973},
{9539502.2122704666107893, 0.0020605513905800, 0.0018412932283703},
{11198906.0802163369953632, 0.0021388978821126, 0.0019246410750317},
{12964516.5875480007380247, 0.0022250163632590, 0.0019911599667460},
{14840342.9930883981287479, 0.0022935717840787, 0.0020624191141542} },
{ {176773.4054123469977640, 0.0005624278170074, 0.0005188334061161},
{652335.1758797396905720, 0.0009554269533414, 0.0006853829047056},
{1300141.8846699106507003, 0.0011036906986911, 0.0009877906084194},
{2102166.4122928399592638, 0.0013282203120262, 0.0011358282349556},
{3054799.0599386463873088, 0.0014577215605123, 0.0013099406827628},
{4137160.0507546905428171, 0.0016147118679717, 0.0014242462811265},
{5354398.7988050524145365, 0.0017217489511995, 0.0015500812921902},
{6690995.7325923312455416, 0.0018438536004762, 0.0016418012911758},
{8152353.3586547691375017, 0.0019338624797970, 0.0017411145128698},
{9726626.9194436408579350, 0.0020340746763065, 0.0018176115980462},
{11418578.3008757624775171, 0.0021114177084223, 0.0018998892424873},
{13218818.6003852263092995, 0.0021964335546051, 0.0019655535905646},
{15131435.5385554376989603, 0.0022641116318642, 0.0020358980651434} },
{ {180218.1126167567272205, 0.0005552383578896, 0.0005122010420523},
{665048.5205694289179519, 0.0009432154470453, 0.0006766173186804},
{1325480.2313152926508337, 0.0010895854901016, 0.0009751626476627},
{2143135.9525002399459481, 0.0013112485371037, 0.0011213074352791},
{3114334.4713390669785440, 0.0014390973046689, 0.0012931962797916},
{4217789.5523110125213861, 0.0015940851817951, 0.0014060411237929},
{5458750.3227158496156335, 0.0016997576784735, 0.0015302695803654},
{6821394.9276762176305056, 0.0018203062607194, 0.0016208179339374},
{8311230.7691735755652189, 0.0019091686506960, 0.0017188636423433},
{9916182.3034439180046320, 0.0020081048556965, 0.0017943839115628},
{11641103.9586996082216501, 0.0020844635575183, 0.0018756118231577},
{13476423.8514373172074556, 0.0021683977791523, 0.0019404379828790},
{15426309.1823959760367870, 0.0022352152197853, 0.0020098852713672} },
{ {183707.2744946462335065, 0.0005481858881907, 0.0005056950566638},
{677925.9435730775585398, 0.0009312365804439, 0.0006680188645861},
{1351145.5941360341385007, 0.0010757489654758, 0.0009627753309363},
{2184634.2481129826046526, 0.0012945999763108, 0.0011070633582427},
{3174638.2501696608960629, 0.0014208276761598, 0.0012767709165453},
{4299459.6609028326347470, 0.0015738511679350, 0.0013881828293621},
{5564448.6048091221600771, 0.0016781849871708, 0.0015108352929652},
{6953477.0453968802466989, 0.0017972070316894, 0.0016002343054297},
{8472158.6278690751641989, 0.0018849446750506, 0.0016970366015521},
{10108184.0463148094713688, 0.0019826290756338, 0.0017715986408569},
{11866501.4628900438547134, 0.0020580220937855, 0.0018517967705282},
{13737353.6519731432199478, 0.0021408951716587, 0.0019158006825779},
{15724988.3188843019306660, 0.0022068682617664, 0.0019843678291054} },
{ {187241.1760049711447209, 0.0005412669493425, 0.0004993122590835},
{690968.4968266430078074, 0.0009194844812720, 0.0006595833193382},
{1377140.0694145592860878, 0.0010621743437164, 0.0009506225817264},
{2226664.6891945074312389, 0.0012782664746411, 0.0010930890161311},
{3235715.3216251148842275, 0.0014029037291727, 0.0012606565384655},
{4382177.0471150018274784, 0.0015539999234442, 0.0013706626410874},
{5671502.2797462632879615, 0.0016570203230984, 0.0014917689032817},
{7087252.8737175632268190, 0.0017745446174211, 0.0015800403164715},
{8635150.0789363738149405, 0.0018611787111914, 0.0016756226943935},
{10302647.8298655692487955, 0.0019576348876698, 0.0017492446214072},
{12094789.2227656170725822, 0.0020320804014142, 0.0018284324179788},
{14001629.3132303487509489, 0.0021139123032501, 0.0018916296215783},
{16027497.3421803135424852, 0.0021790569212614, 0.0019593332415557} },
{ {190820.1021152120374609, 0.0005344781911491, 0.0004930495584934},
{704177.2321689733071253, 0.0009079534613988, 0.0006513065923998},
{1403465.7533610383979976, 0.0010488550563512, 0.0009386985141462},
{2269230.6654902966693044, 0.0012622401326639, 0.0010793776403346},
{3297570.6115038343705237, 0.0013853167981871, 0.0012448453436382},
{4465948.3823961568996310, 0.0015345218556346, 0.0013534720769469},
{5779919.9803443923592567, 0.0016362534625835, 0.0014730611833556},
{7222733.2005902212113142, 0.0017523080755848, 0.0015602261942426},
{8800218.2664703913033009, 0.0018378592880110, 0.0016546115601147},
{10499589.3359580039978027, 0.0019331102328723, 0.0017273110386923},
{12325985.6475961375981569, 0.0020066259686469, 0.0018055074645754},
{14269272.1463869456201792, 0.0020874361650075, 0.0018679131099886},
{16333860.6463989038020372, 0.0021517677943895, 0.0019347694034750} },
{ {194444.3377969723660499, 0.0005278163679053, 0.0004869039605201},
{717553.2014391639968380, 0.0008966380099250, 0.0006431847208619},
{1430124.7422381872311234, 0.0010357847395274, 0.0009269974257846},
{2312335.5660200589336455, 0.0012465132969667, 0.0010659226733168},
{3360209.0449514677748084, 0.0013680584874987, 0.0012293297732976},
{4550780.3369451081380248, 0.0015154076703458, 0.0013366029192209},
{5889710.3395380349829793, 0.0016158745000996, 0.0014547031926999},
{7359928.8139543216675520, 0.0017304868042876, 0.0015407824703611},
{8967376.3349228743463755, 0.0018149752912090, 0.0016339931607128},
{10699024.2462860401719809, 0.0019090434273258, 0.0017057874150686},
{12560109.1466296855360270, 0.0019816466726768, 0.0017830109613534},
{14540303.4627661518752575, 0.0020614541523711, 0.0018446398219751},
{16644102.6259012017399073, 0.0021249878937168, 0.0019106645866072} },
{ {198114.1680099455697928, 0.0005212783343922, 0.0004808725634186},
{731097.4561981816077605, 0.0008855327866036, 0.0006352138647417},
{1457119.1323293882887810, 0.0010229572264211, 0.0009155137908949},
{2355982.7803929713554680, 0.0012310785510215, 0.0010527177606653},
{3423635.5476308055222034, 0.0013511206612268, 0.0012141025027566},
{4636679.5813139760866761, 0.0014966483609642, 0.0013200472047371},
{6000881.9903581142425537, 0.0015958738365105, 0.0014366862677433},
{7498850.5018511274829507, 0.0017090705293929, 0.0015216999696878},
{9136637.4264629520475864, 0.0017925159501204, 0.0016137577689313},
{10900968.2427203189581633, 0.0018854231481338, 0.0016846635972268},
{12797178.1292221564799547, 0.0019571307652435, 0.0017609322982360},
{14814744.5733862798660994, 0.0020359540501044, 0.0018217987822782},
{16958247.6749281734228134, 0.0020987046329158, 0.0018870074256469} },
{ {201829.8777221342024859, 0.0005148610422532, 0.0004749525548416},
{744811.0483400356024504, 0.0008746326155757, 0.0006273903022693},
{1484451.0200683400034904, 0.0010103665399944, 0.0009042422538551},
{2400175.6980222449637949, 0.0012159287064561, 0.0010397567436773},
{3487855.0446471483446658, 0.0013344954336419, 0.0011991564328681},
{4723652.7860771855339408, 0.0014782351977643, 0.0013037972154212},
{6113443.5659126397222281, 0.0015762421676837, 0.0014190020115127},
{7639509.0523605765774846, 0.0016880492924822, 0.0015029697994182},
{9308014.6874353289604187, 0.0017704708248816, 0.0015938959568644},
{11105437.0069721527397633, 0.0018622384202077, 0.0016639297443284},
{13037211.0045418720692396, 0.0019330668588924, 0.0017392611915833},
{15092616.7895912602543831, 0.0020109240179789, 0.0017993793532046},
{17276320.1873966865241528, 0.0020729058120067, 0.0018637869048546} },
{ {205591.7519161928212270, 0.0005085615363693, 0.0004691412085590},
{758695.0296490013133734, 0.0008639324792721, 0.0006197104257563},
{1512122.5014391110744327, 0.0009980068859796, 0.0008931776229395},
{2444917.7083892174996436, 0.0012010567946448, 0.0010270336520725},
{3552872.4612934091128409, 0.0013181751601535, 0.0011844846816364},
{4811706.6217547906562686, 0.0014601597177963, 0.0012878454693339},
{6227403.6992593314498663, 0.0015569704737660, 0.0014016422838661},
{7781915.2527672229334712, 0.0016674134394261, 0.0014845833386611},
{9481521.2586218193173409, 0.0017488297944753, 0.0015743985848662},
{11312446.2208910994231701, 0.0018394786034703, 0.0016435763163494},
{13280226.1820630654692650, 0.0019094439136667, 0.0017179876721506},
{15373941.4230493195354939, 0.0019863525770587, 0.0017773712223242},
{17598344.5583769679069519, 0.0020475796031638, 0.0018409923452221} },
{ {209400.0755042874079663, 0.0005023769515270, 0.0004634358810676},
{772750.4519031215459108, 0.0008534275126847, 0.0006121707372425},
{1540135.6729484293609858, 0.0009858726462592, 0.0008823148643319},
{2490212.2009626119397581, 0.0011864560587515, 0.0010145426971532},
{3618692.7232348117977381, 0.0013021524283147, 0.0011700805763239},
{4900847.7589307343587279, 0.0014424137151386, 0.0012721847119940},
{6342771.0234844340011477, 0.0015380500087994, 0.0013845991921676},
{7926079.8922837488353252, 0.0016471536089352, 0.0014665322287470},
{9657170.2856120485812426, 0.0017275830450016, 0.0015552567911267},
{11522011.5662199258804321, 0.0018171333807318, 0.0016235940632790},
{13526242.0707100480794907, 0.0018862512245556, 0.0016971020736535},
{15658739.7838106267154217, 0.0019622285965284, 0.0017557643905888},
{17924345.1801771521568298, 0.0020227145372005, 0.0018186133922190} },
{ {213255.1335080645221751, 0.0004963045090471, 0.0004578340089437},
{786978.3668945658719167, 0.0008431129977867, 0.0006047678446566},
{1568492.6302517789881676, 0.0009739583724376, 0.0008716490963674},
{2536062.5652024014852941, 0.0011721199460606, 0.0010022782652109},
{3685320.7573533747345209, 0.0012864200495978, 0.0011559376458957},
{4991082.8680378366261721, 0.0014249892315988, 0.0012568079082164},
{6459554.1716528609395027, 0.0015194722907547, 0.0013678650822481},
{8072013.7569508487358689, 0.0016272607223397, 0.0014488083634725},
{9834974.9128965586423874, 0.0017067210586709, 0.0015364619815625},
{11734148.7247728612273932, 0.0017951927459012, 0.0016039740145586},
{13775277.0800933670252562, 0.0018634784092744, 0.0016765950217878},
{15947033.1843765750527382, 0.0019385412811095, 0.0017345491608821},
{18254346.4481269270181656, 0.0019982994906356, 0.0017966400040117} },
{ {217157.2108719897805713, 0.0004903415138090, 0.0004523331057300},
{801379.8264347790973261, 0.0008329843583149, 0.0005974984578768},
{1597195.4703331987839192, 0.0009622587797819, 0.0008611755840889},
{2582472.1905368650332093, 0.0011580421003736, 0.0009902349114012},
{3752761.4863996482454240, 0.0012709710512032, 0.0011420496137108},
{5082418.6197241665795445, 0.0014078785477671, 0.0012417082341548},
{6577761.7769879112020135, 0.0015012290920998, 0.0013514325298581},
{8219727.6377381309866905, 0.0016077259732173, 0.0014314038801003},
{10014948.2832549884915352, 0.0016862346030551, 0.0015180058201587},
{11948873.3784106262028217, 0.0017736469928956, 0.0015847074689423},
{14027349.6194419041275978, 0.0018411153967886, 0.0016564574237076},
{16238842.9350630324333906, 0.0019152801590419, 0.0017137161272873},
{18588372.7578812055289745, 0.0019743236732145, 0.0017750624402589} },
{ {221106.5925608453690074, 0.0004844853511304, 0.0004469307591463},
{815955.8822217513807118, 0.0008230371546151, 0.0005903593851346},
{1626246.2889049211516976, 0.0009507687412732, 0.0008508897339617},
{2629444.4665266289375722, 0.0011442163555117, 0.0009784073532363},
{3821019.8380438904277980, 0.0012557986683793, 0.0011284103905793},
{5174861.6852923240512609, 0.0013910741745274, 0.0012268790697197},
{6697402.4721058979630470, 0.0014833124306799, 0.0013352943323818},
{8369232.3181659504771233, 0.0015885408176959, 0.0014143111506701},
{10197103.5417920909821987, 0.0016661147209633, 0.0014998802197309},
{12166201.2088558953255415, 0.0017524867048592, 0.0015657859849592},
{14282478.0975117757916451, 0.0018191524160929, 0.0016366804579690},
{16534190.3464833144098520, 0.0018924350704689, 0.0016932561645792},
{18926448.5003247261047363, 0.0019507766160464, 0.0017538712511792} },
{ {225103.5635526217229199, 0.0004787334839653, 0.0004416246286343},
{830707.5860804449766874, 0.0008132670787977, 0.0005833475295393},
{1655647.1827901753131300, 0.0009394832820703, 0.0008407870888433},
{2676982.7824602592736483, 0.0011306367280007, 0.0009667904652735},
{3890100.7354875127784908, 0.0012408963370427, 0.0011150140680506},
{5268418.7342780781909823, 0.0013745688448121, 0.0012123139913524},
{6818484.8904633009806275, 0.0014657145609261, 0.0013194435010412},
{8520538.5914308484643698, 0.0015696969651429, 0.0013975227735656},
{10381453.8325899187475443, 0.0016463527205771, 0.0014820773331081},
{12386147.8979674614965916, 0.0017317027439391, 0.0015472013715762},
{14540680.9245340190827847, 0.0017975799856837, 0.0016172555647158},
{16833096.7306542545557022, 0.0018699961563950, 0.0016731604182367},
{19268598.0756533890962601, 0.0019276481601615, 0.0017330572673998} },
{ {229148.4087975159927737, 0.0004730834501514, 0.0004364124426962},
{845635.9899891872191802, 0.0008036699501005, 0.0005764598856305},
{1685400.2480369475670159, 0.0009283975740898, 0.0008308633231586},
{2725090.5279436730779707, 0.0011172974108809, 0.0009553792732911},
{3960009.1061415220610797, 0.0012262576866227, 0.0011018549120611},
{5363096.4365062266588211, 0.0013583555057576, 0.0011980067650421},
{6941017.6654596831649542, 0.0014484279655804, 0.0013038732531491},
{8673657.2434102687984705, 0.0015511863691839, 0.0013810315655953},
{10568012.2984176333993673, 0.0016269401661394, 0.0014645895445799},
{12608729.1272662151604891, 0.0017112862413555, 0.0015289456794200},
{14801976.5092992968857288, 0.0017763889028629, 0.0015981744368650},
{17135583.4031275026500225, 0.0018479538480379, 0.0016534202948523},
{19614845.8667687140405178, 0.0019049284456239, 0.0017126115898574} },
{ {233241.4132574522227515, 0.0004675328597459, 0.0004312919964324},
{860742.1452358121750876, 0.0007942417103503, 0.0005696935362261},
{1715507.5802578113507479, 0.0009175069308449, 0.0008211142382286},
{2773771.0923815765418112, 0.0011041927674726, 0.0009441689490625},
{4030749.8747534705325961, 0.0012118765333352, 0.0010889273567227},
{5458901.4651809576898813, 0.0013424273112015, 0.0011839513396493},
{7065009.4283480383455753, 0.0014314453476152, 0.0012885770050821},
{8828599.0582721941173077, 0.0015330012191351, 0.0013648305541915},
{10756792.0827573053538799, 0.0016078688689586, 0.0014474094617246},
{12833960.5784156396985054, 0.0016912285879857, 0.0015110111921710},
{15066383.2608321961015463, 0.0017555702348455, 0.0015794290105088},
{17441671.6691430062055588, 0.0018262988566632, 0.0016340274529953},
{19965216.2812169119715691, 0.0018826079009637, 0.0016925255803832} },
{ {237382.8619349227810744, 0.0004620793925141, 0.0004262611493204},
{876027.1043877898482606, 0.0007849784197650, 0.0005630456492733},
{1745971.2771884473040700, 0.0009068068024756, 0.0008115357578971},
{2823027.8652505800127983, 0.0010913173253412, 0.0009331548051745},
{4102327.9667292982339859, 0.0011977468735465, 0.0010762259985542},
{5555840.4875216968357563, 0.0013267776144277, 0.0011701418405098},
{7190468.8152465578168631, 0.0014147596225539, 0.0012735483650956},
{8985374.8295250050723553, 0.0015151339317510, 0.0013489129701068},
{10947806.3303881827741861, 0.0015891308786611, 0.0014305299077197},
{13061857.9329512398689985, 0.0016715214253332, 0.0014933904184797},
{15333919.5899074096232653, 0.0017351153084494, 0.0015610114570182},
{17751382.8387673832476139, 0.0018050221637767, 0.0016149737943495},
{20319733.7049433588981628, 0.0018606772331526, 0.0016727908526162} },
{ {241573.0397468201990705, 0.0004567207955328, 0.0004213178228348},
{891491.9183392806444317, 0.0007758762527454, 0.0005565134749344},
{1776793.4337065555155277, 0.0008962927710127, 0.0008021239241731},
{2872864.2360018161125481, 0.0010786657707052, 0.0009223322902135},
{4174748.3075525285676122, 0.0011838628776653, 0.0010637455906032},
{5653920.1772416839376092, 0.0013113999612351, 0.0011565725633610},
{7317404.4569974169135094, 0.0013983639110629, 0.0012587811268081},
{9143995.3422697298228741, 0.0014975771433677, 0.0013332722403485},
{11141068.1850984059274197, 0.0015707184751310, 0.0014139439136968},
{13292436.8756183590739965, 0.0016521566367637, 0.0014760760841098},
{15604603.9025373272597790, 0.0017150157015608, 0.0015429141745996},
{18064738.2281383834779263, 0.0017841150117767, 0.0015962514552529},
{20678422.5341044738888741, 0.0018391274179652, 0.0016533992631163} },
{ {245812.2316816753009334, 0.0004514548807930, 0.0004164599984225},
{907137.6394619466736913, 0.0007669314939768, 0.0005500943427018},
{1807976.1461117805447429, 0.0008859605458402, 0.0007928748932573},
{2923283.5941103282384574, 0.0010662329428147, 0.0009116969839722},
{4248015.8226165557280183, 0.0011702188837590, 0.0010514810375149},
{5753147.2030921112746000, 0.0012962880833163, 0.0011432379682683},
{7445824.9872067915275693, 0.0013822515319174, 0.0012442692627244},
{9304471.3852937184274197, 0.0014803237022451, 0.0013179019813706},
{11336590.7906344719231129, 0.0015526241603491, 0.0013976447117111},
{13525713.0851650666445494, 0.0016331263391845, 0.0014590611244894},
{15878454.6102440040558577, 0.0016952632343307, 0.0015251297805050},
{18381759.1464550048112869, 0.0017635688949647, 0.0015778527986857},
{21041307.1635786108672619, 0.0018179496906325, 0.0016343429031042} },
{ {250100.7226700062747113, 0.0004462795230345, 0.0004116857153794},
{922965.3193799202563241, 0.0007581405346197, 0.0005437856586896},
{1839521.5111824334599078, 0.0008758059592493, 0.0007837849314895},
{2974289.3288817545399070, 0.0010540138288200, 0.0009012445929611},
{4322135.4373601889237761, 0.0011568093924724, 0.0010394273895763},
{5853528.2360277324914932, 0.0012814358918334, 0.0011301326741955},
{7575739.0390313016250730, 0.0013664159951270, 0.0012300069180390},
{9466813.7462546583265066, 0.0014633666613641, 0.0013027959925502},
{11534387.2910257503390312, 0.0015348406508683, 0.0013816257276539},
{13761702.2427869569510221, 0.0016144228750432, 0.0014423386774183},
{16155490.1222504228353500, 0.0016758499608450, 0.0015076511034790},
{18702466.9049800075590611, 0.0017433755508522, 0.0015597704063338},
{21408411.9864858873188496, 0.0017971355370272, 0.0016156140903083} },
{ {254438.7977111744694412, 0.0004411926575488, 0.0004069930689878},
{938976.0098845714237541, 0.0007494998686740, 0.0005375849030104},
{1871431.6254986908752471, 0.0008658249622996, 0.0007748504116855},
{3025884.8301567747257650, 0.0010420035586116, 0.0008909709461328},
{4397112.0771960010752082, 0.0011436290607682, 0.0010275798383798},
{5955069.9465917656198144, 0.0012668374713157, 0.0011172514534377},
{7707155.2454882133752108, 0.0013508509954606, 0.0012159884048706},
{9631033.2133734598755836, 0.0014466992714059, 0.0012879482499439},
{11734470.8301622234284878, 0.0015173608705055, 0.0013658805747298},
{14000420.0293798651546240, 0.0015960388046248, 0.0014259020762334},
{16435728.8478802423924208, 0.0016567681612622, 0.0014904711765331},
{19026882.8150122836232185, 0.0017235269518981, 0.0015419970711774},
{21779761.3974221572279930, 0.0017766766850465, 0.0015972053612647} },
{ {258826.7418776465638075, 0.0004361922781604, 0.0004023802083328},
{955170.7627372237620875, 0.0007410060894988, 0.0005314896272212},
{1903708.5846648125443608, 0.0008560136207953, 0.0007660678094217},
{3078073.4869412849657238, 0.0010301974000318, 0.0008808719906544},
{4472950.6675827987492085, 0.0011306726969743, 0.0010159337117642},
{6057779.0053963130339980, 0.0012524870737362, 0.0011045892263886},
{7840082.2396800043061376, 0.0013355504061502, 0.0012022081963956},
{9797140.5736449323594570, 0.0014303149740289, 0.0012733529002866},
{11936854.5519478060305119, 0.0015001779432575, 0.0013504030471058},
{14241882.1318046804517508, 0.0015779668986061, 0.0014097448430452},
{16719189.1961205229163170, 0.0016380103340065, 0.0014735832300165},
{19355028.1878073662519455, 0.0017040152975610, 0.0015245257903936},
{22155379.7914401814341545, 0.0017565650964824, 0.0015791094638693} },
{ {263264.8400081779691391, 0.0004312764352266, 0.0003978453348158},
{971550.6297815946163610, 0.0007326558863717, 0.0005254974519678},
{1936354.4851614527869970, 0.0008463681112299, 0.0007574336997417},
{3130858.6889951406046748, 0.0010185907541933, 0.0008709437878875},
{4549656.1339209908619523, 0.0011179352556491, 0.0010044844691835},
{6161662.0830101314932108, 0.0012383791128994, 0.0010921410565954},
{7974528.6546775810420513, 0.0013205082728745, 0.0011886609215755},
{9965146.6163450721651316, 0.0014142073954376, 0.0012590042552188},
{12141551.6001911070197821, 0.0014832851865644, 0.0013351871137572},
{14486104.2305562775582075, 0.0015602001310689, 0.0013938606824989},
{17005889.5763188228011131, 0.0016195691884331, 0.0014569806849709},
{19686924.3340896554291248, 0.0016848330065992, 0.0015073497582732},
{22535291.5615440607070923, 0.0017367929590595, 0.0015613193502217} },
{ {267753.3770660858135670, 0.0004264432337192, 0.0003933867000969},
{988116.6626748921116814, 0.0007244460414106, 0.0005196060646054},
{1969371.4232457634061575, 0.0008368847173333, 0.0007489447535360},
{3184243.8257230832241476, 0.0010071791510047, 0.0008611825096182},
{4627233.4016736587509513, 0.0011054118325656, 0.0009932276973575},
{6266725.8498901892453432, 0.0012245081589676, 0.0010799021459863},
{8110503.1235035918653011, 0.0013057188079216, 0.0011753413598098},
{10135062.1285780407488346, 0.0013983703401654, 0.0012448967857532},
{12348575.1189361307770014, 0.0014666761048072, 0.0013202269125748},
{14733102.0030047427862883, 0.0015427316725538, 0.0013782434755386},
{17295848.3976615928113461, 0.0016014376378641, 0.0014406571466199},
{20022592.5661725550889969, 0.0016659727097256, 0.0014904623598046},
{22919521.1018260307610035, 0.0017173526788723, 0.0015438281698041} },
{ {272292.6380272396490909, 0.0004216908314551, 0.0003890026047184},
{1004869.9133735654177144, 0.0007163734263177, 0.0005138132169614},
{2002761.4951724130660295, 0.0008275598264756, 0.0007405977343592},
{3238232.2866363353095949, 0.0009959582448726, 0.0008515844343239},
{4705687.3963465243577957, 0.0010930976600825, 0.0009821591059456},
{6372976.9766795700415969, 0.0012108689332514, 0.0010678678301313},
{8248014.2793286573141813, 0.0012911763847140, 0.0011622444359760},
{10306897.9002856090664864, 0.0013827977851726, 0.0012310251168730},
{12557938.2518843822181225, 0.0014503443830843, 0.0013055167447410},
{14982891.1355464644730091, 0.0015255548835822, 0.0013628872735669},
{17589084.0695333704352379, 0.0015836087926065, 0.0014246063983877},
{20362054.1941605210304260, 0.0016474272425803, 0.0014738571641319},
{23308092.8099514916539192, 0.0016982368731302, 0.0015266292628399} },
{ {276882.9077955629909411, 0.0004170174372537, 0.0003846913962128},
{1021811.4333731831284240, 0.0007084349995085, 0.0005081167232397},
{2036526.7971985228359699, 0.0008183899257746, 0.0007323894958294},
{3292827.4610373112373054, 0.0009849238105217, 0.0008421459436076},
{4785023.0431749932467937, 0.0010809881025783, 0.0009712745235306},
{6480422.1339433621615171, 0.0011974563032360, 0.0010560335738251},
{8387070.7553820619359612, 0.0012768755323586, 0.0011493652155422},
{10480664.7171200159937143, 0.0013674838740603, 0.0012173840225437},
{12769654.1432376764714718, 0.0014342838812371, 0.0012910510693174},
{15235487.3073026295751333, 0.0015086633083274, 0.0013477862927389},
{17885615.0012027248740196, 0.0015660759535611, 0.0014088223957707},
{20705330.5315801464021206, 0.0016291896388828, 0.0014575279185072},
{23701031.0717670433223248, 0.0016794383631604, 0.0015097161539981} },
{ {281524.4715972605627030, 0.0004124213093443, 0.0003804514676956},
{1038942.2752010333351791, 0.0007006278031948, 0.0005025144579194},
{2070669.4255257006734610, 0.0008093715993849, 0.0007243169780076},
{3348032.7384612960740924, 0.0009740717390927, 0.0008328635188438},
{4865245.2678324086591601, 0.0010690786520625, 0.0009605698936411},
{6589067.9921353552490473, 0.0011842652776668, 0.0010443949668733},
{8527681.1839802525937557, 0.0012628109305853, 0.0011366988999090},
{10656373.3654858954250813, 0.0013524229116148, 0.0012039684206806},
{12983735.9366946704685688, 0.0014184886280137, 0.0012768244980128},
{15490906.2008124906569719, 0.0014920506685454, 0.0013329349085407},
{18185459.6019574552774429, 0.0015488326058734, 0.0013932992608497},
{21052442.8868020512163639, 0.0016112531238906, 0.0014414685424370},
{24098360.2867973856627941, 0.0016609501676553, 0.0014930825462611} },
{ {286217.6140595228644088, 0.0004079007536602, 0.0003762812563092},
{1056263.4893153046723455, 0.0006929489606281, 0.0004970043537906},
{2105191.4765905942767859, 0.0008005015249488, 0.0007163772049562},
{3403851.5083608357235789, 0.0009633980342433, 0.0008237337378469},
{4946358.9956335006281734, 0.0010573649240227, 0.0009500412709757},
{6698921.2221852568909526, 0.0011712910019401, 0.0010329477199825},
{8669854.1984412353485823, 0.0012489774047792, 0.0011242408220362},
{10834034.6370478495955467, 0.0013376093584621, 0.0011907733684499},
{13200196.7775672618299723, 0.0014029528155499, 0.0012628317901345},
{15749163.4979655742645264, 0.0014757108577817, 0.0013183276505922},
{18488636.2810534872114658, 0.0015318724128888, 0.0013780312766374},
{21403412.5716548636555672, 0.0015936111081742, 0.0014256731219481},
{24500104.8590467832982540, 0.0016427654961831, 0.0014767223150855} },
{ {290962.6204496703576297, 0.0004034541223340, 0.0003721792417723},
{1073776.1290011329110712, 0.0006853956734648, 0.0004915844000148},
{2140095.0463434960693121, 0.0007917764707259, 0.0007085672817122},
{3460287.1602771002799273, 0.0009528988085384, 0.0008147532716916},
{5028369.1508734170347452, 0.0010458426533659, 0.0009396848177417},
{6809988.4941726224496961, 0.0011585287537077, 0.0010216876606827},
{8813598.4338292013853788, 0.0012353699211508, 0.0011119864420143},
{11013659.3203984480351210, 0.0013230378260470, 0.0011777940577395},
{13419049.8046109173446894, 0.0013876707940920, 0.0012490678478058},
{16010274.8806096483021975, 0.0014596379356670, 0.0013039591974818},
{18795163.4476493597030640, 0.0015151892103292, 0.0013630128819674},
{21758260.8922186270356178, 0.0015762571815202, 0.0014101359041765},
{24906289.1616997942328453, 0.0016248777430039, 0.0014606295027203} },
{ {295759.7753780860803090, 0.0003990798121623, 0.0003681439450894},
{1091481.2450519744306803, 0.0006779652192270, 0.0004862526403198},
{2175382.2315269540995359, 0.0007831932924452, 0.0007008843920138},
{3517343.0836933534592390, 0.0009425702798081, 0.0008059188817761},
{5111280.6618541646748781, 0.0010345076905422, 0.0009294968002365},
{6922276.4789103735238314, 0.0011459739384514, 0.0010106107296790},
{8958922.5204586870968342, 0.0012219835822795, 0.0010999313430108},
{11195258.1997262202203274, 0.0013087030716698, 0.0011650258107342},
{13640308.1670010238885880, 0.0013726370667689, 0.0012355277113089},
{16274256.0304800868034363, 0.0014438261227045, 0.0012898243720395},
{19105059.5112925618886948, 0.0014987770007333, 0.0013482386663070},
{22117009.1718188561499119, 0.0015591851071417, 0.0013948512921599},
{25316937.6062866039574146, 0.0016072804810417, 0.0014447983129216} },
{ {300609.3641707119531929, 0.0003947762631640, 0.0003641739270757},
{1109379.8895369672682136, 0.0006706549488008, 0.0004810071712044},
{2211055.1279731835238636, 0.0007747489306732, 0.0006933257954151},
{3575022.6680294713005424, 0.0009324087678825, 0.0007972274168155},
{5195098.4514947561547160, 0.0010233559978002, 0.0009194735854287},
{7035791.8468635901808739, 0.0011336220854584, 0.0009997129771230},
{9105835.0923664774745703, 0.0012088136226370, 0.0010880712272604},
{11378842.0646936837583780, 0.0012945999938388, 0.0011524640757089},
{13863985.0096558984369040, 0.0013578462847253, 0.0012222065545958},
{16541122.6294156368821859, 0.0014282697948691, 0.0012759181366880},
{19418342.8785838484764099, 0.0014826299480431, 0.0013337033649784},
{22479678.7167756631970406, 0.0015423888160529, 0.0013798138397424},
{25732074.5794617943465710, 0.0015899674561477, 0.0014292231055221} },
{ {305511.6716027013026178, 0.0003905419572133, 0.0003602677872135},
{1127473.1142314053140581, 0.0006634622841327, 0.0004758461402879},
{2247115.8318409966304898, 0.0007664404079829, 0.0006858888250190},
{3633329.3027076730504632, 0.0009224106911976, 0.0007886758100120},
{5279827.4450339144095778, 0.0010123836456084, 0.0009096116376622},
{7150541.2686386769637465, 0.0011214688437992, 0.0009889905590709},
{9254344.7837059982120991, 0.0011958554043315, 0.0010764019122246},
{11564421.7030069306492805, 0.0012807236276655, 0.0011401044230318},
{14090093.4721673354506493, 0.0013432932423449, 0.0012090996810308},
{16810890.3588930405676365, 0.0014129634787559, 0.0012622355888080},
{19735031.9654632583260536, 0.0014667423723960, 0.0013194018543930},
{22846290.8310009762644768, 0.0015258624017174, 0.0013650182467376},
{26151724.4769208133220673, 0.0015729325814986, 0.0014138983915931} },
{ {310466.9826742786099203, 0.0003863754166700, 0.0003564241623142},
{1145761.9709338110405952, 0.0006563847158431, 0.0004707677447184},
{2283566.4393575773574412, 0.0007582648263333, 0.0006785708850535},
{3692266.3768815249204636, 0.0009125725636689, 0.0007802610764718},
{5365472.5702928733080626, 0.0010015868091797, 0.0008999075156575},
{7266531.4147957870736718, 0.0011095099785142, 0.0009784397341065},
{9404460.2224562764167786, 0.0011831044130840, 0.0010649193269585},
{11752007.9033621232956648, 0.0012670691405527, 0.0011279425411451},
{14318646.7003191225230694, 0.0013289728726543, 0.0011962025191880},
{17083574.9009509645402431, 0.0013979018467119, 0.0012487719565802},
{20055145.1745652668178082, 0.0014511087452137, 0.0013053291476036},
{23216866.8369522616267204, 0.0015096001148109, 0.0013504593543166},
{26575911.6929003670811653, 0.0015561699323072, 0.0013988188285201} },
{ {315475.5823624234180897, 0.0003822752031024, 0.0003526417253127},
{1164247.5113954069092870, 0.0006494198014690, 0.0004657702290393},
{2320409.0482747321948409, 0.0007502193645904, 0.0006713694486742},
{3751837.2808705754578114, 0.0009028909916945, 0.0007719803104640},
{5452038.7518988065421581, 0.0009909617651492, 0.0008903578694245},
{7383768.9558769194409251, 0.0010977413668720, 0.0009680568600876},
{9556190.0507022254168987, 0.0011705562541626, 0.0010536195085258},
{11941611.4532851651310921, 0.0012536318279130, 0.0011159742329139},
{14549657.8379350714385509, 0.0013148802429091, 0.0011835106189240},
{17359191.9365631751716137, 0.0013830797122276, 0.0012355225947155},
{20378700.9176544174551964, 0.0014357236843223, 0.0012914803899356},
{23591428.0375182777643204, 0.0014935963582842, 0.0013361321404246},
{27004660.6201500333845615, 0.0015396737406672, 0.0013839792153436} },
{ {320537.7556225203443319, 0.0003782399160712, 0.0003489191842509},
{1182930.7874714981298894, 0.0006425651624661, 0.0004608518849485},
{2357645.7522504711523652, 0.0007423012760282, 0.0006642820556919},
{3812045.4023773400112987, 0.0008933626711795, 0.0007638306829670},
{5539530.9136714376509190, 0.0009805048883488, 0.0008809594373626},
{7502260.5625070491805673, 0.0010861589948888, 0.0009578383910486},
{9709542.8966674897819757, 0.0011582066490344, 0.0010424985983597},
{12133243.1393775362521410, 0.0012404071092535, 0.0011041954119619},
{14783140.0289799254387617, 0.0013010105504448, 0.0011710196474964},
{17637757.1517239436507225, 0.0013684920254198, 0.0012224829804916},
{20705717.6068916395306587, 0.0014205819493373, 0.0012778508547124},
{23969995.7459751963615417, 0.0014778456825245, 0.0013220317154849},
{27437995.6565376371145248, 0.0015234383905736, 0.0013693744883094} },
{ {325653.7874323132564314, 0.0003742681918615, 0.0003452552810707},
{1201812.8508786237798631, 0.0006358184831089, 0.0004560110484543},
{2395278.6494185156188905, 0.0007345078860358, 0.0006573063105586},
{3872894.1315751015208662, 0.0008839843847415, 0.0007558094392630},
{5627953.9809700530022383, 0.0009702126487336, 0.0008717090435603},
{7622012.9050067905336618, 0.0010747589538674, 0.0009477808740417},
{9864527.3895881809294224, 0.0011460514309211, 0.0010315528395332},
{12326913.7526006791740656, 0.0012273905241223, 0.0010926020991313},
{15019106.4168044663965702, 0.0012873591184297, 0.0011587253859149},
{17919286.2208330668509007, 0.0013541338688153, 0.0012096487099210},
{21036213.6431393921375275, 0.0014056784372314, 0.0012644359393674},
{24352591.2715777680277824, 0.0014623427807025, 0.0013081533182845},
{27875941.1931467838585377, 0.0015074584131509, 0.0013549997165342} },
{ {330823.9627521834918298, 0.0003703587024086, 0.0003416487905584},
{1220894.7533926528412849, 0.0006291775080201, 0.0004512460992527},
{2433309.8354460578411818, 0.0007268365898482, 0.0006504398803037},
{3934386.8608565749600530, 0.0008747529989746, 0.0007479138965980},
{5717312.8811475643888116, 0.0009600816084050, 0.0008626035949984},
{7743032.6543683325871825, 0.0010635374371237, 0.0009378809463785},
{10021152.1694102082401514, 0.0011340865420542, 0.0010207785730129},
{12522634.0806488525122404, 0.0012145777284657, 0.0010811904192229},
{15257570.1464721392840147, 0.0012739213920622, 0.0011466237253106},
{18203794.8346407637000084, 0.0013400004531755, 0.0011970154939639},
{21370207.4442443959414959, 0.0013910081780115, 0.0012512311614798},
{24739235.9338802769780159, 0.0014470824843244, 0.0012944923118098},
{28318521.6249139383435249, 0.0014917284820750, 0.0013408500978393} },
{ {336048.5665098094032146, 0.0003665101541589, 0.0003380985193411},
{1240177.5465850105974823, 0.0006226400404531, 0.0004465554590753},
{2471741.4067951436154544, 0.0007192848503222, 0.0006436804925460},
{3996526.9758239816874266, 0.0008656654618520, 0.0007401414418932},
{5807612.5387723408639431, 0.0009501084187121, 0.0008536400790512},
{7865326.4815915236249566, 0.0010524907367454, 0.0009281353326604},
{10179425.8654771298170090, 0.0011223080299033, 0.0010101722348348},
{12720414.9072363376617432, 0.0012019644909282, 0.0010699565976768},
{15498544.3606424313038588, 0.0012606929346871, 0.0011347106635641},
{18491298.6673374101519585, 0.0013260871135007, 0.0011845791549447},
{21707717.4176810905337334, 0.0013765663306050, 0.0012382321550160},
{25129951.0371027365326881, 0.0014320597589401, 0.0012810441795686},
{28765761.3460597731173038, 0.0014762434091149, 0.0013269209547238} },
{ {341327.8837549455929548, 0.0003627212869778, 0.0003346033048645},
{1259662.2830289888661355, 0.0006162039404061, 0.0004419375905777},
{2510575.4595635421574116, 0.0007118501959105, 0.0006370259336695},
{4059317.8674832289107144, 0.0008567188002105, 0.0007324895296973},
{5898857.8792383316904306, 0.0009402898175143, 0.0008448155609897},
{7988901.0557351708412170, 0.0010416152406298, 0.0009185408422148},
{10339357.1097196806222200, 0.0011107120440187, 0.0009997303531785},
{12920267.0231255497783422, 0.0011895466893982, 0.0010588969574883},
{15742042.2033536676317453, 0.0012476694241798, 0.0011229823019459},
{18781813.4009347371757030, 0.0013123893050669, 0.0011723356232886},
{22048761.9677316583693027, 0.0013623481788608, 0.0012254346667652},
{25524757.8879180811345577, 0.0014172696999876, 0.0012678045216902},
{29217684.7503721192479134, 0.0014609981398716, 0.0013132077305876} },
{ {346662.1994402250275016, 0.0003589908731362, 0.0003311620146016},
{1279350.0136886124964803, 0.0006098671229039, 0.0004373909959483},
{2549814.0901232291944325, 0.0007045302185941, 0.0006304740470139},
{4122762.9252226534299552, 0.0008479101172705, 0.0007249556799660},
{5991053.8280505500733852, 0.0009306226264813, 0.0008361271814616},
{8113763.0482451841235161, 0.0010309074294723, 0.0009090943663733},
{10500954.5383315570652485, 0.0010992948328664, 0.0009894495454336},
{13122201.2174460422247648, 0.0011773203076610, 0.0010480079161784},
{15988076.8199654929339886, 0.0012348466494251, 0.0011114348419850},
{19075354.7197422981262207, 0.0012989026000338, 0.0011602809338334},
{22393359.5115009807050228, 0.0013483491276707, 0.0012128345528466},
{25923677.8078048899769783, 0.0014027075287658, 0.0012547690513864},
{29674316.2329793088138103, 0.0014459877496719, 0.0012997059858450} },
{ {352051.7984427556511946, 0.0003553177163116, 0.0003277735448157},
{1299241.7906221733428538, 0.0006036275563287, 0.0004329142156658},
{2589459.3942184341140091, 0.0006973225719369, 0.0006240227310509},
{4186865.5384024954400957, 0.0008392365904118, 0.0007175374761036},
{6084205.3105725748464465, 0.0009211037485802, 0.0008275721543744},
{8239919.1291221594437957, 0.0010203638738953, 0.0008997928760039},
{10664226.7829402163624763, 0.0010880527407102, 0.0009793265156370},
{13326228.2783974073827267, 0.0011652814320884, 0.0010372859828913},
{16236661.3536273837089539, 0.0012222205068683, 0.0011000645823381},
{19371938.3071511201560497, 0.0012856226835356, 0.0011484112229086},
{22741528.4508062861859798, 0.0013345646992980, 0.0012004277753421},
{26326732.1073684133589268, 0.0013883685886156, 0.0012419335915282},
{30135680.1846925467252731, 0.0014312074395776, 0.0012864113945197} },
{ {357496.9657624597311951, 0.0003517006506882, 0.0003244368199788},
{1319338.6656232902314514, 0.0005974832607175, 0.0004285058274314},
{2629513.4690532316453755, 0.0006902249692390, 0.0006176699378063},
{4251629.0965453488752246, 0.0008306954686423, 0.0007102325631601},
{6178317.2522225724533200, 0.0009117301655260, 0.0008191477644712},
{8367375.9699160736054182, 0.0010099812317719, 0.0008906334190886},
{10829182.4767831228673458, 0.0010769822049063, 0.0009693580515519},
{13532358.9904836844652891, 0.0011534262485661, 0.0010267277556623},
{16487808.9472232908010483, 0.0012097869972682, 0.0010888679158212},
{19671579.8432112298905849, 0.0012725453503675, 0.0011367227251960},
{23093287.2038933672010899, 0.0013209905298131, 0.0011882103990208},
{26733942.0862073153257370, 0.0013742483411915, 0.0012292940712159},
{30601801.0073215030133724, 0.0014166525325351, 0.0012733197405855} },
{ {362997.9863633090280928, 0.0003481385398800, 0.0003211507917989},
{1339641.6904771744739264, 0.0005914323063034, 0.0004241644449670},
{2669978.4098874870687723, 0.0006832351816181, 0.0006114136711560},
{4317056.9890924086794257, 0.0008222840708059, 0.0007030386456617},
{6273394.5784789910539985, 0.0009024989354416, 0.0008108513652839},
{8496140.2407400216907263, 0.0009997562454997, 0.0008816131183693},
{10995830.2528144400566816, 0.0010660797528471, 0.0009595410223632},
{13740604.1430204920470715, 0.0011417510394574, 0.0010163299185782},
{16741532.7430061865597963, 0.0011975422225411, 0.0010778413266192},
{19974295.0098818056285381, 0.0012596665017247, 0.0011252117706335},
{23448654.1672015674412251, 0.0013076223655715, 0.0011761785883308},
{27145329.0672185905277729, 0.0013603423628536, 0.0012168465226280},
{31072703.0887280702590942, 0.0014023184697061, 0.0012604269147896} },
{ {368555.1452704531257041, 0.0003446302762159, 0.0003179144383348},
{1360151.9169662557542324, 0.0005854728118750, 0.0004198887169934},
{2710856.3136609159410000, 0.0006763510365011, 0.0006052519851846},
{4383152.6054995004087687, 0.0008139997831751, 0.0006959534860484},
{6369442.2147662583738565, 0.0008934071905151, 0.0008026803770310},
{8626218.6120668277144432, 0.0009896857395400, 0.0008727291690302},
{11164178.7441811971366405, 0.0010553419993674, 0.0009498723761172},
{13950974.5281717684119940, 0.0011302521806810, 0.0010060892393170},
{16997845.8943129964172840, 0.0011854823826681, 0.0010669813874476},
{20280099.4886598177254200, 0.0012469821418529, 0.0011138747816399},
{23807647.7573028951883316, 0.0012944560599545, 0.0011643286042864},
{27560914.3576885983347893, 0.0013466463412434, 0.0012045870778539},
{31548410.8230847418308258, 0.0013882008069100, 0.0012477289113480} },
{ {374168.7274154535843991, 0.0003411747797356, 0.0003147267633172},
{1380870.3968206511344761, 0.0005796029434214, 0.0004156773260614},
{2752149.2764335609972477, 0.0006695704155649, 0.0005991829829297},
{4449919.3352346327155828, 0.0008058400575257, 0.0006889749029044},
{6466465.0865931296721101, 0.0008844521347919, 0.0007946322846016},
{8757617.7544052321463823, 0.0009797666178837, 0.0008639788366143},
{11334236.5839026514440775, 0.0010447656440910, 0.0009403491373468},
{14163480.9264925867319107, 0.0011189261389041, 0.0009960025665631},
{17256761.5329351536929607, 0.0011736037728169, 0.0010562847569414},
{20589008.9616649225354195, 0.0012344883750812, 0.0011027082702730},
{24170286.3886540159583092, 0.0012814875701316, 0.0011526568016067},
{27980719.2689978703856468, 0.0013331560718794, 0.0011925119658903},
{32028948.6034289188683033, 0.0013742952111575, 0.0012352218248526} },
{ {379839.0177638249588199, 0.0003377709974744, 0.0003115867953208},
{1401798.1821274629328400, 0.0005738209126412, 0.0004115289876642},
{2793859.3944543218240142, 0.0006628912534372, 0.0005932048146707},
{4517360.5677499063313007, 0.0007978024091648, 0.0006821007692137},
{6564468.1184459477663040, 0.0008756310420074, 0.0007867046356195},
{8890344.3386882040649652, 0.0009699958616597, 0.0008553594548727},
{11506012.4051424544304609, 0.0010343474688600, 0.0009309684048130},
{14378134.1317837480455637, 0.0011077694688593, 0.0009860668276033},
{17518292.8046464100480080, 0.0011619027803668, 0.0010457481770392},
{20901039.1102318949997425, 0.0012221814027099, 0.0010917088355807},
{24536588.4629499129951000, 0.0012687129539480, 0.0011411596259146},
{28404765.1122371666133404, 0.0013198674550175, 0.0011806175097827},
{32514340.8334237970411777, 0.0013605974573557, 0.0012229018472290} },
{ {385566.3012733576470055, 0.0003344179025535, 0.0003084935870491},
{1422936.3240799866616726, 0.0005681249756598, 0.0004074424491558},
{2835988.7639568853192031, 0.0006563115359673, 0.0005873156766161},
{4585479.6924637593328953, 0.0007898844150749, 0.0006753290107311},
{6663456.2369535341858864, 0.0008669412535455, 0.0007788950385761},
{9024405.0352768916636705, 0.0009603705268895, 0.0008468684237259},
{11679514.8408218938857317, 0.0010240843353678, 0.0009217273492516},
{14594944.9302053544670343, 0.0010967788106488, 0.0009762790259389},
{17782452.8598897680640221, 0.0011503758823485, 0.0010353684705022},
{21216205.6158511973917484, 0.0012100575202714, 0.0010808731609816},
{24906572.3916268646717072, 0.0012561283669712, 0.0011298336109920},
{28833073.1988361999392509, 0.0013067764925518, 0.0011689001237217},
{33004611.9017278626561165, 0.0013471034250694, 0.0012107652649022} },
{ {391350.8629110853653401, 0.0003311144934916, 0.0003054462146342},
{1444285.8748050483409315, 0.0005625134316180, 0.0004034164888685},
{2878539.4812074643559754, 0.0006498292987624, 0.0005815138094914},
{4654280.0989730367437005, 0.0007820837119928, 0.0006686576044916},
{6763434.3681076411157846, 0.0008583801763999, 0.0007712011610469},
{9159806.5143730752170086, 0.0009508877422597, 0.0008385032073045},
{11854752.5241065677255392, 0.0010139731826972, 0.0009126232113299},
{14813924.1096851769834757, 0.0010859508873064, 0.0009666362391214},
{18049254.8391501381993294, 0.0011390196426320, 0.0010251425386558},
{21534524.1630939841270447, 0.0011981131145782, 0.0010701980118252},
{25280256.5840765424072742, 0.0012437300595114, 0.0011186753762265},
{29265664.8396181128919125, 0.0012938792846748, 0.0011573563106976},
{33499786.1892667189240456, 0.0013338090954348, 0.0011988084559786} },
{ {397192.9876518794335425, 0.0003278597934062, 0.0003024437767599},
{1465847.8862866975832731, 0.0005569846214838, 0.0003994499151707},
{2921513.6425058892928064, 0.0006434426256493, 0.0005757974973318},
{4723765.1765934256836772, 0.0007743979947795, 0.0006620845772569},
{6864407.4365168288350105, 0.0008499452812689, 0.0007636207279087},
{9296555.4458978958427906, 0.0009415447069624, 0.0008302613320671},
{12031734.0880570504814386, 0.0010040110251143, 0.0009036532994302},
{15035082.4581891018897295, 0.0010752825023008, 0.0009571356164783},
{18318711.8863584436476231, 0.0011278307094700, 0.0010150673587708},
{21856010.4282627440989017, 0.0011863446611798, 0.0010596802328515},
{25657659.4497469253838062, 0.0012315143739548, 0.0011076816240410},
{29702561.3467800654470921, 0.0012811720276737, 0.0011459826591886},
{33999888.1218148767948151, 0.0013207105481758, 0.0011870278875004} },
{ {403092.9604671705164947, 0.0003246528493018, 0.0002994853943356},
{1487623.4102024650201201, 0.0005515369267819, 0.0003955415656197},
{2964913.3440841371193528, 0.0006371496473527, 0.0005701650661163},
{4793938.3147652782499790, 0.0007668250146265, 0.0006556080040226},
{6966380.3670070702210069, 0.0008416341006564, 0.0007561515196840},
{9434658.5025270152837038, 0.0009323386886175, 0.0008221403849627},
{12210468.1655912417918444, 0.0009941949498403, 0.0008948149878581},
{15258430.7637698594480753, 0.0010647705371925, 0.0009477743770601},
{18590837.1454311683773994, 0.0011168058129710, 0.0010051399821930},
{22180680.0985371321439743, 0.0011747487216743, 0.0010493167460469},
{26038799.3975682035088539, 0.0012194777419671, 0.0010968491374771},
{30143784.0293037891387939, 0.0012686510104210, 0.0011347758413672},
{34504942.0536666214466095, 0.0013078039587667, 0.0011754201128997} },
{ {409051.0661945716128685, 0.0003214927314029, 0.0002965702095906},
{1509613.4978764250408858, 0.0005461687684198, 0.0003916903060531},
{3008740.6822012183256447, 0.0006309485400571, 0.0005646148825581},
{4864802.9034235421568155, 0.0007593625774019, 0.0006492260066555},
{7069358.0854607718065381, 0.0008334442271074, 0.0007487913709407},
{9574122.3535901512950659, 0.0009232670213338, 0.0008141380116666},
{12390963.3900341894477606, 0.0009845221149551, 0.0008861057146431},
{15483979.8139164112508297, 0.0010544119493442, 0.0009385498075658},
{18865643.7602346390485764, 0.0011059417627090, 0.0009953575319551},
{22508548.8537032082676888, 0.0011633219412306, 0.0010391045481900},
{26423694.8371390923857689, 0.0012076166819520, 0.0010861747778295},
{30589354.2010445967316628, 0.0012563126120263, 0.0011237326102726},
{35014972.4177308231592178, 0.0012950855955337, 0.0011639817694063} },
{ {415067.5899301168392412, 0.0003183785324271, 0.0002936973855000},
{1531819.2015897987876087, 0.0005408786055521, 0.0003878950297743},
{3052997.7530992645770311, 0.0006248375241106, 0.0005591453529506},
{4936362.3314647683873773, 0.0007520085421298, 0.0006429367525256},
{7173345.5174371646717191, 0.0008253733114328, 0.0007415381686205},
{9714953.6689065694808960, 0.0009143271036955, 0.0008062519148879},
{12573228.3954919148236513, 0.0009749897472108, 0.0008775229800687},
{15711740.3970889393240213, 0.0010442037697625, 0.0009294592604364},
{19143144.8746017403900623, 0.0010952354454485, 0.0009857172007200},
{22839632.3751964680850506, 0.0011520610461535, 0.0010290407088357},
{26812364.1775336451828480, 0.0011959277964676, 0.0010756554824202},
{31039293.1711613275110722, 0.0012441532991806, 0.0011128497975917},
{35530003.5759675726294518, 0.0012825518170532, 0.0011527095757339} },
{ {421142.8169040851062164, 0.0003153093669602, 0.0002908661052894},
{1554241.5731977142859250, 0.0005356649344861, 0.0003841546568335},
{3097686.6528215119615197, 0.0006188148627704, 0.0005537549219923},
{5008619.9880564603954554, 0.0007447608194017, 0.0006367384531783},
{7278347.5882228361442685, 0.0008174190609807, 0.0007343898507095},
{9857159.1208453979343176, 0.0009055163969540, 0.0007984798526800},
{12757271.8124516587704420, 0.0009655951402265, 0.0008690643445417},
{15941723.3020947445183992, 0.0010341431009057, 0.0009205001518566},
{19423353.6317388564348221, 0.0010846838229131, 0.0009762162488185},
{23173946.3460790663957596, 0.0011409628415600, 0.0010191223681535},
{27204825.8281418234109879, 0.0011844077698703, 0.0010652882622991},
{31493622.2494310550391674, 0.0012321696235534, 0.0011021243112362},
{36050059.9402032718062401, 0.0012701990694897, 0.0011416003295337} },
{ {427277.0316750384517945, 0.0003122843708647, 0.0002880755717294},
{1576881.6641526296734810, 0.0005305262875534, 0.0003804681331515},
{3142809.4786432320252061, 0.0006128788608993, 0.0005484420717034},
{5081579.2645191531628370, 0.0007376173699080, 0.0006306293630769},
{7384369.2231127591803670, 0.0008095792381619, 0.0007273444045379},
{10000745.3786576204001904, 0.0008968324232316, 0.0007908196368946},
{12943102.2765008788555861, 0.0009563356524643, 0.0008607274269955},
{16173939.3148942161351442, 0.0010242271146790, 0.0009116699600362},
{19706283.1781567037105560, 0.0010742839296206, 0.0009668520023217},
{23511506.4427247419953346, 0.0011300242090896, 0.0010093467348492},
{27601098.1981311812996864, 0.0011730533659435, 0.0010550702002609},
{31952362.7566363140940666, 0.0012203582194386, 0.0010915531334044},
{36575165.9016151726245880, 0.0012580238841408, 0.0011306509053453} },
{ {433470.5192303877556697, 0.0003093027005608, 0.0002853250066305},
{1599740.5263670384883881, 0.0005254612321475, 0.0003768344298578},
{3188368.3250135038979352, 0.0006070278638511, 0.0005432053203472},
{5155243.5476585915312171, 0.0007305762029969, 0.0006246077783362},
{7491415.3486301992088556, 0.0008018516587161, 0.0007203998655509},
{10145719.1144402585923672, 0.0008882727637555, 0.0007832691316015},
{13130728.4175975881516933, 0.0009472087053863, 0.0008525099031861},
{16408399.2260103020817041, 0.0010144530503160, 0.0009029662234450},
{19991946.6538288332521915, 0.0010640328707659, 0.0009576218510630},
{23852328.3548973985016346, 0.0011192421047224, 0.0009997110842702},
{28001199.6966910734772682, 0.0011618614255631, 0.0010449984486473},
{32415535.9907221347093582, 0.0012087158013063, 0.0010811333182071},
{37105345.8542077988386154, 0.0012460228749257, 0.0011198582522023} },
{ {439723.5646946027409285, 0.0003063635326002, 0.0002826136502800},
{1622819.2119018645025790, 0.0005204683696841, 0.0003732525425290},
{3234365.2896075113676488, 0.0006012602562719, 0.0005380432214053},
{5229616.2260799547657371, 0.0007236353752989, 0.0006186720355781},
{7599490.8878023354336619, 0.0007942341902332, 0.0007135543158704},
{10292086.9975919555872679, 0.0008798350571368, 0.0007758262516367},
{13320158.8712323866784573, 0.0009382117816548, 0.0008444095041130},
{16645113.8188828080892563, 0.0010048182130136, 0.0008943865386720},
{20280357.2086565233767033, 0.0010539278202797, 0.0009485232468974},
{24196427.7589737810194492, 0.0011086135566725, 0.0009902127564588},
{28405148.7333696298301220, 0.0011508288646207, 0.0010350702274625},
{32883163.2728346697986126, 0.0011972391615860, 0.0010708619897403},
{37640624.1910747289657593, 0.0012341927361257, 0.0011092193916314} },
{ {446036.4529859145404771, 0.0003034660629418, 0.0002799407609723},
{1646118.7711455388925970, 0.0005155463346635, 0.0003697214905018},
{3280802.4682903517968953, 0.0005955744610122, 0.0005329543625309},
{5304700.6933946451172233, 0.0007167929893972, 0.0006128205107424},
{7708600.7682920675724745, 0.0007867247507348, 0.0007068058829538},
{10439855.6948606241494417, 0.0008715169978503, 0.0007684889611283},
{13511402.2706614751368761, 0.0009293424234193, 0.0008364240144107},
{16884093.8860645405948162, 0.0009953199709813, 0.0008859285596538},
{20571527.9783866405487061, 0.0010439660188353, 0.0009395537019150},
{24543820.3422940969467163, 0.0010981356633428, 0.0009808491543870},
{28812963.7165567427873611, 0.0011399526717592, 0.0010252828224266},
{33355265.9090900607407093, 0.0011859251684448, 0.0010607363400822},
{38181025.3072509840130806, 0.0012225302400084, 0.0010987314155506} },
{ {452409.4690535254194401, 0.0003006095065037, 0.0002773056144089},
{1669640.2588160017039627, 0.0005106937936846, 0.0003662403162206},
{3327681.9573058816604316, 0.0005899689380090, 0.0005279373646546},
{5380500.3393753953278065, 0.0007100471924721, 0.0006070516180354},
{7818749.9162019956856966, 0.0007793213072854, 0.0007001527382653},
{10589031.8816625922918320, 0.0008633163345638, 0.0007612552721353},
{13704467.2527697347104549, 0.0009205982306515, 0.0008285512708449},
{17125350.2140719927847385, 0.0009859557547540, 0.0008775899951806},
{20865472.1125197298824787, 0.0010341447719548, 0.0009307107867269},
{24894521.7812682874500751, 0.0010878055913206, 0.0009716177420799},
{29224663.0576138831675053, 0.0011292299064721, 0.0010156335831180},
{33831865.2139196246862411, 0.0011747707636276, 0.0010507536274043},
{38726573.5971629694104195, 0.0012110322346995, 0.0010883914842846} },
{ {458842.8979011859628372, 0.0002977930965466, 0.0002747075032816},
{1693384.7237867105286568, 0.0005059094446227, 0.0003628080845660},
{3375005.8529048874042928, 0.0005844421832940, 0.0005229908810446},
{5457018.5474023232236505, 0.0007033961751287, 0.0006013638087963},
{7929943.2545203166082501, 0.0007720218742719, 0.0006935930963152},
{10739622.2263043075799942, 0.0008552308686855, 0.0007541232431822},
{13899362.4420285280793905, 0.0009119768594359, 0.0008207891608451},
{17368893.5907881557941437, 0.0009767230548475, 0.0008693686077922},
{21162202.7536974921822548, 0.0010244614482164, 0.0009219921288903},
{25248547.7585877701640129, 0.0010776205734889, 0.0009625160429724},
{29640265.1578343175351620, 0.0011186576969756, 0.0010061199211576},
{34312982.4886328503489494, 0.0011637729604080, 0.0010409111739952},
{39277293.4493223503232002, 0.0011996956420434, 0.0010781968246666} },
{ {465337.0244671158143319, 0.0002950160842859, 0.0002721457368240},
{1717353.2194397179409862, 0.0005011920156757, 0.0003594238822332},
{3422776.2513527362607419, 0.0005789927279369, 0.0005181135962905},
{5534258.7131554577499628, 0.0006968381701282, 0.0005957555704616},
{8042185.7097156411036849, 0.0007648245127390, 0.0006871252129700},
{10891633.4079687315970659, 0.0008472584526551, 0.0007470909783486},
{14096096.4734740182757378, 0.0009034760205142, 0.0008131356211075},
{17614734.8035395257174969, 0.0009676194202392, 0.0008612622122214},
{21461733.0451839454472065, 0.0010149134775130, 0.0009133954112162},
{25605913.9535881057381630, 0.0010675779071696, 0.0009535416382310},
{30059788.4452267102897167, 0.0011082332383885, 0.0009967393085449},
{34798639.0572293698787689, 0.0011529288416151, 0.0010312063646523},
{39833209.2713567987084389, 0.0011885174555382, 0.0010681447281073} },
{ {471892.1337334470590577, 0.0002922777382563, 0.0002696196401652},
{1741546.7966199521906674, 0.0004965402645807, 0.0003560868171163},
{3470995.2489130161702633, 0.0005736191371056, 0.0005133042255999},
{5612224.2227537278085947, 0.0006903714512184, 0.0005902254255935},
{8155482.2074037874117494, 0.0007577273284283, 0.0006807473847347},
{11045072.0846161898225546, 0.0008393969890330, 0.0007401566253059},
{14294677.9866687394678593, 0.0008950934777060, 0.0008055886361720},
{17862884.6419239528477192, 0.0009586424567668, 0.0008532686738692},
{21764076.1298171132802963, 0.0010054983492590, 0.0009049183703220},
{25966636.0591761767864227, 0.0010576749523494, 0.0009446921651178},
{30483251.3129493929445744, 0.0010979537907969, 0.0009874892758817},
{35288856.2257643789052963, 0.0011422355576277, 0.0010216366447634},
{40394345.4465355724096298, 0.0011774947382994, 0.0010582325489136} },
{ {478508.5106024064589292, 0.0002895773438940, 0.0002671285542043},
{1765966.5080058926250786, 0.0004919529777179, 0.0003527960177608},
{3519664.9417303502559662, 0.0005683200090940, 0.0005085615137852},
{5690918.4664200702682137, 0.0006839943320262, 0.0005847719309076},
{8269837.6727309944108129, 0.0007507284709454, 0.0006744579474300},
{11199944.9283531494438648, 0.0008316444285788, 0.0007333183748235},
{14495115.6086815763264894, 0.0008868270464160, 0.0007981462371860},
{18113353.8926790282130241, 0.0009497898254962, 0.0008453859074548},
{22069245.1535558551549911, 0.0009962136108548, 0.0008965587950536},
{26330729.7397820875048637, 0.0010479091298465, 0.0009359653155260},
{30910672.1682288162410259, 0.0010878166775355, 0.0009783674107918},
{35783655.3069964051246643, 0.0011316903246045, 0.0010121995187527},
{40960726.3648070096969604, 0.0011666246212531, 0.0010484577024350} },
{ {485186.4401263899635524, 0.0002869142031036, 0.0002646718348555},
{1790613.4041576674208045, 0.0004874289694412, 0.0003495506327107},
{3568787.4262529551051557, 0.0005630939744250, 0.0005038842346029},
{5770344.8337787603959441, 0.0006777051649215, 0.0005793936762540},
{8385257.0312593933194876, 0.0007438261323442, 0.0006682552751467},
{11356258.6165304426103830, 0.0008239987692083, 0.0007265744591382},
{14697417.9746562317013741, 0.0008786745922811, 0.0007908065005025},
{18366153.3427698947489262, 0.0009410592412351, 0.0008376118756406},
{22377253.2698704302310944, 0.0009870568660021, 0.0008883145250884},
{26698210.6969464793801308, 0.0010382779198250, 0.0009273588343114},
{31342069.4321998655796051, 0.0010778192833866, 0.0009693713563930},
{36283057.6102153509855270, 0.0011212904225946, 0.0010028925483883},
{41532376.4319909140467644, 0.0011559043011074, 0.0010388176634262} },
{ {491926.2070931516354904, 0.0002842876337194, 0.0002622488528486},
{1815488.5372887423727661, 0.0004829670811942, 0.0003463498300557},
{3618364.7987022865563631, 0.0005579396949636, 0.0004992711898131},
{5850506.7142388690263033, 0.0006715023399938, 0.0005740892838325},
{8501745.2084041479974985, 0.0007370185461162, 0.0006621377792043},
{11514019.8165128082036972, 0.0008164580546131, 0.0007199231509914},
{14901593.7172716129571199, 0.0008706340297206, 0.0007835675465521},
{18621293.7817393317818642, 0.0009324484710105, 0.0008299445877060},
{22688113.6021130792796612, 0.0009780257731992, 0.0008801834495976},
{27069094.5948265604674816, 0.0010287788599589, 0.0009188705180014},
{31777461.5056841485202312, 0.0010679590528878, 0.0009604988096788},
{36787084.4468743950128555, 0.0011110331937947, 0.0009937133512309},
{42109320.0435109809041023, 0.0011453310386830, 0.0010293099644413} },
{ {498728.0968844904564321, 0.0002816969691534, 0.0002598589932284},
{1840592.9608630752190948, 0.0004785661808169, 0.0003431927968460},
{3668399.1563731646165252, 0.0005528558630564, 0.0004947212085204},
{5931407.4971680957823992, 0.0006653842840064, 0.0005688574071845},
{8619307.1295868530869484, 0.0007303039859149, 0.0006561039070702},
{11673235.1987606883049011, 0.0008090203730398, 0.0007133627624552},
{15107651.4698359947651625, 0.0008627033206483, 0.0007764275385421},
{18878785.9995037391781807, 0.0009239553327541, 0.0008223820982799},
{23001839.3077606149017811, 0.0009691180442465, 0.0008721635057725},
{27443397.1219314895570278, 0.0010194095440178, 0.0009104982132951},
{32216866.8034150935709476, 0.0010582334887516, 0.0009517475201348},
{37295757.1273324266076088, 0.0011009160409098, 0.0009846595990740},
{42691581.5789624080061913, 0.0011349021571012, 0.0010199321942297} },
{ {505592.3940584427327849, 0.0002791415579441, 0.0002575016549612},
{1865927.7247926464769989, 0.0004742251618402, 0.0003400787385407},
{3718892.5923827649094164, 0.0005478412007075, 0.0004902331463787},
{6013050.5720820669084787, 0.0006593494593481, 0.0005636967304466},
{8737947.7201318070292473, 0.0007236807645545, 0.0006501521414929},
{11833911.4338785931468010, 0.0008016838560768, 0.0007068916439079},
{15315599.8652736824005842, 0.0008548804731591, 0.0007693846813654},
{19138640.7796757780015469, 0.0009155776937366, 0.0008149225061143},
{23318443.5241271778941154, 0.0009603314427878, 0.0008642526776161},
{27821133.9677296020090580, 0.0010101676202501, 0.0009022398157224},
{32660303.7315479815006256, 0.0010486401502540, 0.0009431152882700},
{37809096.9606363698840141, 0.0010909364254402, 0.0009757290165524},
{43279185.4547211751341820, 0.0011246150400820, 0.0010106819961815} },
{ {512519.3837076046038419, 0.0002766207633045, 0.0002551762506626},
{1891493.8814215878956020, 0.0004699429427626, 0.0003370068785888},
{3769847.2051822091452777, 0.0005428944587630, 0.0004858058848691},
{6095439.3284726664423943, 0.0006533963631508, 0.0005586059674053},
{8857671.9051895570009947, 0.0007171472329491, 0.0006442809994256},
{11996055.1925299484282732, 0.0007944466774520, 0.0007005081829406},
{15525447.5365535989403725, 0.0008471635403021, 0.0007624372204228},
{19400868.9155239090323448, 0.0009073134694452, 0.0008075639528889},
{23637939.3999577015638351, 0.0009516637828874, 0.0008564489945950},
{28202320.8024748899042606, 0.0010010507899332, 0.0008940932682990},
{33107790.7019687555730343, 0.0010391766516965, 0.0009345999642050},
{38327125.2710870802402496, 0.0010810918661297, 0.0009669193796092},
{43872156.0423061326146126, 0.0011144671303684, 0.0010015570669020} },
{ {519509.3508033096441068, 0.0002741339627872, 0.0002528822060495},
{1917292.4825083287432790, 0.0004657184663981, 0.0003339764578769},
{3821265.0909516462124884, 0.0005380144161319, 0.0004814383306420},
{6178577.1558228805661201, 0.0006475235262874, 0.0005535838608097},
{8978484.6128765996545553, 0.0007107017790310, 0.0006384890311673},
{12159673.1450385544449091, 0.0007873070518611, 0.0006942108034809},
{15737203.1167529970407486, 0.0008395506188468, 0.0007555834405705},
{19665481.1851368620991707, 0.0008991606220926, 0.0008003046220443},
{23960340.0774343535304070, 0.0009431129277060, 0.0008487505305677},
{28586973.3123132362961769, 0.0009920568059854, 0.0008860565602890},
{33559346.1061671003699303, 0.0010298406609556, 0.0009261994464303},
{38849863.3610868081450462, 0.0010713799374128, 0.0009582285141886},
{44470517.7402434498071671, 0.0011044559280746, 0.0009925551547070} },
{ {526562.5803290229523554, 0.0002716805478542, 0.0002506189597659},
{1943324.5798345434013754, 0.0004615506992214, 0.0003309867343164},
{3873148.3483685650862753, 0.0005331998790740, 0.0004771294148094},
{6262467.4434957988560200, 0.0006417295125232, 0.0005486291815630},
{9100390.7621129099279642, 0.0007043428267850, 0.0006327748194781},
{12324771.9622104689478874, 0.0007802632341688, 0.0006879979644797},
{15950875.2399769071489573, 0.0008320398481114, 0.0007488216650187},
{19932488.3900956586003304, 0.0008911171595046, 0.0007931427376389},
{24285658.7138191536068916, 0.0009346767881863, 0.0008411554022935},
{28975107.1793517880141735, 0.0009831834715139, 0.0008781277259303},
{34014988.3931687772274017, 0.0010206298979900, 0.0009179116803730},
{39377332.5355553925037384, 0.0010617982679091, 0.0009496542948583},
{45074294.9498298242688179, 0.0010945789891380, 0.0009836740583286} },
{ {533679.3572216827888042, 0.0002692599235515, 0.0002483859629557},
{1969591.2251833016052842, 0.0004574386307002, 0.0003280369823241},
{3925499.0689765983261168, 0.0005284496804017, 0.0004728780922411},
{6347113.5808492712676525, 0.0006360129176064, 0.0005437407278643},
{9223395.2892123237252235, 0.0006980688353179, 0.0006271369786810},
{12491358.3143906593322754, 0.0007733135177311, 0.0006818681595269},
{16166472.5374189503490925, 0.0008246294088450, 0.0007421502543306},
{20201901.3053529746830463, 0.0008831811338287, 0.0007860765633185},
{24613908.4192781001329422, 0.0009263533217348, 0.0008336617686742},
{29366738.0844610147178173, 0.0009744286385389, 0.0008703048432319},
{34474735.9357764199376106, 0.0010115421335147, 0.0009097346572768},
{39909554.1026802286505699, 0.0010523445390161, 0.0009411946435023},
{45683512.0644726455211639, 0.0010848339239058, 0.0009749116254604} },
{ {540859.9664984917035326, 0.0002668715080908, 0.0002461826789026},
{1996093.4703899561427534, 0.0004533812727241, 0.0003251264924514},
{3978319.3520102500915527, 0.0005237626788205, 0.0004686833410980},
{6432518.9583394657820463, 0.0006303723683757, 0.0005389173246898},
{9347503.1068417578935623, 0.0006918782978615, 0.0006215741538127},
{12659438.8735691457986832, 0.0007664562339554, 0.0006758199154498},
{16384003.6480665020644665, 0.0008173175221015, 0.0007355676054201},
{20473730.7252432666718960, 0.0008753506403510, 0.0007791044012424},
{24945102.3701567649841309, 0.0009181405310296, 0.0008262678293274},
{29761881.7113273926079273, 0.0009657902066434, 0.0008625860328100},
{34938607.1480922400951385, 0.0010025751876277, 0.0009016664128736},
{40446549.3975641205906868, 0.0010430164834774, 0.0009328475280738},
{46298193.4765932559967041, 0.0010752183956184, 0.0009662657515424} },
{ {548104.6930037147831172, 0.0002645147325489, 0.0002440085827616},
{2022832.3672410931903869, 0.0004493776590319, 0.0003222545708970},
{4031611.2917874166741967, 0.0005191377582504, 0.0004645441620466},
{6518686.9657661309465766, 0.0006248065220179, 0.0005341578228530},
{9472719.1520184278488159, 0.0006857697409244, 0.0006160850198311},
{12829020.3067754078656435, 0.0007596897510420, 0.0006698517916276},
{16603477.1904617566615343, 0.0008101024481983, 0.0007290721506286},
{20747987.4394896887242794, 0.0008676238164232, 0.0007722245910050},
{25279253.6986587122082710, 0.0009100364627970, 0.0008189718236202},
{30160553.7398719973862171, 0.0009572661217393, 0.0008549694568105},
{35406620.4615733399987221, 0.0009937269284738, 0.0008937050262823},
{40988339.7011942863464355, 0.0010338118839937, 0.0009246109614200},
{46918363.5825144797563553, 0.0010657301190404, 0.0009577343784824} },
{ {555413.8218414187431335, 0.0002621890405154, 0.0002418631611764},
{2049808.9675985109061003, 0.0004454268445536, 0.0003194205391116},
{4085376.9873782033100724, 0.0005145738271042, 0.0004604595776715},
{6605620.9885439546778798, 0.0006193140652100, 0.0005294610985034},
{9599048.3357567861676216, 0.0006797417233812, 0.0006106682808571},
{13000109.2881719097495079, 0.0007530124729869, 0.0006639623791093},
{16824901.8122580088675022, 0.0008029824856464, 0.0007226623566864},
{21024682.2338466793298721, 0.0008599988402624, 0.0007654355087349},
{25616375.5468796491622925, 0.0009020392066371, 0.0008117720296498},
{30562769.8528159223496914, 0.0009488543748492, 0.0008474533176888},
{35878794.2547335624694824, 0.0009849952710130, 0.0008858486188406},
{41534946.3529586493968964, 0.0010247285719914, 0.0009164830000194},
{47544046.7723510786890984, 0.0010563668590967, 0.0009493154933682} },
{ {562787.6379091454437003, 0.0002598938877203, 0.0002397459121747},
{2077024.3228765160311013, 0.0004415279049150, 0.0003166237334353},
{4139618.5317668234929442, 0.0005100698177155, 0.0004564286320464},
{6693324.4176098126918077, 0.0006138937133718, 0.0005248260523834},
{9726495.6012533735483885, 0.0006737928356388, 0.0006053226693316},
{13172712.4879788216203451, 0.0007464228387578, 0.0006581502997764},
{17048286.1413617618381977, 0.0007959559702020, 0.0007163367239692},
{21303825.8959347344934940, 0.0008524739299420, 0.0007587355660635},
{25956481.0606759525835514, 0.0008941468939890, 0.0008046667631472},
{30968545.7320640310645103, 0.0009405530009022, 0.0008400358573376},
{36355146.9609052315354347, 0.0009763781757958, 0.0008780953529411},
{42086390.6540517061948776, 0.0010157644262685, 0.0009084617428651},
{48175267.4429433494806290, 0.0010471264295827, 0.0009410071273531} },
{ {570226.4262008211808279, 0.0002576287418024, 0.0002376563445302},
{2104479.4852101788856089, 0.0004376799358988, 0.0003138635046690},
{4194338.0247438251972198, 0.0005056246856617, 0.0004524503900304},
{6781800.6501639140769839, 0.0006085442098924, 0.0005202516091536},
{9855065.8631895799189806, 0.0006679216987912, 0.0006000469453210},
{13346836.5704396441578865, 0.0007399193212807, 0.0006524142055672},
{17273638.8111363947391510, 0.0007890212738260, 0.0007100937854327},
{21585429.2150463275611401, 0.0008450473423441, 0.0007521232092857},
{26299583.3837679252028465, 0.0008863576969035, 0.0007976543765886},
{31377897.0572426468133926, 0.0009323600776190, 0.0008327153558789},
{36835696.9816984906792641, 0.0009678736477488, 0.0008704434310657},
{42642693.9010384008288383, 0.0010069173717839, 0.0009005453303996},
{48812049.9847557321190834, 0.0010380066918078, 0.0009328073544660} },
{ {577730.4716806256910786, 0.0002553930819329, 0.0002355939778134},
{2132175.5063450979068875, 0.0004338820528948, 0.0003111392176743},
{4249537.5588876595720649, 0.0005012374091801, 0.0004485239367636},
{6871053.0629609869793057, 0.0006032643254513, 0.0005157367168543},
{9984764.0462169293314219, 0.0006621269638504, 0.0005948398958263},
{13522488.2183977868407965, 0.0007335004266068, 0.0006467527776177},
{17500968.4542734287679195, 0.0007821768038028, 0.0007039321059359},
{21869502.9763928316533566, 0.0008377173721089, 0.0007455969183591},
{26645695.6599358506500721, 0.0008786698270636, 0.0007907332581484},
{31790839.5144128836691380, 0.0009242737243749, 0.0008254901307565},
{37320462.7249319627881050, 0.0009594797350350, 0.0008628910946592},
{43203877.4166529327630997, 0.0009981853784601, 0.0008927319433529},
{49454418.8081662654876709, 0.0010290055534990, 0.0009247142905025} },
{ {585300.0591714077163488, 0.0002531863985119, 0.0002335583419196},
{2160113.4374309792183340, 0.0004301333904013, 0.0003084502510918},
{4305219.2319214269518852, 0.0004969069885381, 0.0004446483772156},
{6961085.0543700838461518, 0.0005980528572364, 0.0005112803462130},
{10115595.0792244728654623, 0.0006564073109760, 0.0005897003340241},
{13699674.0920008681714535, 0.0007271646930154, 0.0006411647256125},
{17730283.7043923325836658, 0.0007754210017967, 0.0006978502812854},
{22156057.9701405167579651, 0.0008304823507026, 0.0007391552060496},
{26994831.0336546413600445, 0.0008710815347708, 0.0007839018308294},
{32207388.7806210741400719, 0.0009162921011225, 0.0008183585357712},
{37809462.6019896045327187, 0.0009511945279451, 0.0008554366231614},
{43769962.5180938318371773, 0.0009895664600407, 0.0008850198017870},
{50102398.2766176387667656, 0.0010201209674288, 0.0009167260919726} },
{ {592935.4738298929296434, 0.0002510081929473, 0.0002315489768055},
{2188294.3312217174097896, 0.0004264331015150, 0.0003057959968836},
{4361385.1449704160913825, 0.0004926324456509, 0.0004408228354450},
{7051900.0112560866400599, 0.0005929086283114, 0.0005068814901332},
{10247563.8868011794984341, 0.0006507614486511, 0.0005846270986319},
{13878400.8642941378057003, 0.0007209106902331, 0.0006356487869609},
{17961593.1940347775816917, 0.0007687523429361, 0.0006918469375199},
{22445104.9871984049677849, 0.0008233406454719, 0.0007327966171858},
{27347002.6484878398478031, 0.0008635911079244, 0.0007771585515476},
{32627560.5376902744174004, 0.0009084134073391, 0.0008113189600996},
{38302715.0231242999434471, 0.0009430161577731, 0.0008480783329678},
{44340970.5073411315679550, 0.0009810586728978, 0.0008774071639988},
{50756012.8119250759482384, 0.0010113509304141, 0.0009088409549919} },
{ {600637.0008074587676674, 0.0002488579772735, 0.0002295654324393},
{2216719.2393506113439798, 0.0004227803575045, 0.0003031758600811},
{4418037.3868291871622205, 0.0004884128231705, 0.0004370464545037},
{7143501.3231514748185873, 0.0005878304869544, 0.0005025391630831},
{10380675.3934586979448795, 0.0006451881130681, 0.0005796190532597},
{14058675.2105554211884737, 0.0007147370186231, 0.0006302037261026},
{18194905.5568760223686695, 0.0007621693350296, 0.0006859207301029},
{22736654.8123825527727604, 0.0008162906586829, 0.0007265197277133},
{27702223.6475160308182240, 0.0008561968710505, 0.0007705019102701},
{33051370.4800970479846001, 0.0009006358810637, 0.0008043698274141},
{38800238.3916773498058319, 0.0009349427958117, 0.0008408145765760},
{44916922.6994371786713600, 0.0009726601149986, 0.0008698923255955},
{51415286.7948531806468964, 0.0010026934820892, 0.0009010571143175} },
{ {608404.9245936087099835, 0.0002467352739782, 0.0002276072682246},
{2245389.2136498652398586, 0.0004191743472801, 0.0003005892583709},
{4475178.0526012461632490, 0.0004842471842924, 0.0004333183955767},
{7235892.3796056518331170, 0.0005828173059701, 0.0004982524005532},
{10514934.5312844626605511, 0.0006396860673210, 0.0005746750856641},
{14240503.7891420684754848, 0.0007086423082976, 0.0006248283338882},
{18430229.4252950139343739, 0.0007556705176716, 0.0006800703431482},
{23030718.2259608618915081, 0.0008093308266165, 0.0007203231439211},
{28060507.1763827279210091, 0.0008488971843600, 0.0007639304291349},
{33478834.2766341976821423, 0.0008929577978404, 0.0007975095950125},
{39302051.1268508955836296, 0.0009269726523013, 0.0008336437415699},
{45497840.4050130695104599, 0.0009643689248365, 0.0008624736185322},
{52080244.6256179213523865, 0.0009941467038120, 0.0008933728423141} },
{ {616239.5306517754215747, 0.0002446396156465, 0.0002256740530062},
{2274305.3052286119200289, 0.0004156142770098, 0.0002980356218658},
{4532809.2461994113400578, 0.0004801346120638, 0.0004296378376961},
{7329076.5698850462213159, 0.0005778679821058, 0.0004940202584841},
{10650346.2124647237360477, 0.0006342541007771, 0.0005697941073384},
{14423893.2877577114850283, 0.0007026252186213, 0.0006195214266856},
{18667573.4341013170778751, 0.0007492544614303, 0.0006742944887056},
{23327306.0252104252576828, 0.0008024596187807, 0.0007142055017231},
{28421866.3788703233003616, 0.0008416904428708, 0.0007574426616983},
{33909967.5893013104796410, 0.0008853774697873, 0.0007907367529529},
{39808171.6217749565839767, 0.0009191039754345, 0.0008265642497578},
{46083744.9349664822220802, 0.0009561832803482, 0.0008551494101474},
{52750910.6961213275790215, 0.0009857087176997, 0.0008857864480681} },
{ {624141.1033032496925443, 0.0002425705447943, 0.0002237653646906},
{2303468.5662634456530213, 0.0004120993695840, 0.0002955143926889},
{4590933.0621476387605071, 0.0004760742089371, 0.0004260039772447},
{7423057.2833678377792239, 0.0005729814354544, 0.0004898418128249},
{10786915.3778609819710255, 0.0006288910283799, 0.0005649750527364},
{14608850.3651895783841610, 0.0006966844370691, 0.0006142818460968},
{18906946.2109092250466347, 0.0007429197671181, 0.0006685919061055},
{23626429.0025561489164829, 0.0007956755370096, 0.0007081654658783},
{28786314.3956082873046398, 0.0008345750755033, 0.0007510371920537},
{34344786.1587266549468040, 0.0008778932446939, 0.0007840498231818},
{40318618.2986876964569092, 0.0009113350504193, 0.0008195745563717},
{46674657.6014235913753510, 0.0009481013979901, 0.0008479181023297},
{53427309.3999217748641968, 0.0009773776854391, 0.0008782962763623} },
{ {632109.9280535249272361, 0.0002405276135443, 0.0002218807901651},
{2332880.0485521801747382, 0.0004086288643170, 0.0002930250247254},
{4649551.5921853948384523, 0.0004720650962616, 0.0004224160275451},
{7517837.9096159692853689, 0.0005681566087970, 0.0004857161589905},
{10924646.9413017984479666, 0.0006235956900308, 0.0005602168787342},
{14795381.6950632426887751, 0.0006908186788576, 0.0006091084580098},
{19148356.3966667801141739, 0.0007366650649714, 0.0006629613611339},
{23928097.9350350722670555, 0.0007889771146370, 0.0007022017292334},
{29153864.3748219609260559, 0.0008275495442500, 0.0007447126341234},
{34783305.6020072102546692, 0.0008705035051135, 0.0007774473588047},
{40833409.5630259737372398, 0.0009036641985377, 0.0008126731491025},
{47270599.7128190621733665, 0.0009401215317041, 0.0008407781305539},
{54109465.1307785436511040, 0.0009691518074354, 0.0008709007067934} },
{ {640146.2893685329472646, 0.0002385103834024, 0.0002200199248581},
{2362540.8038497660309076, 0.0004052020164666, 0.0002905669833035},
{4708666.9370634462684393, 0.0004681064137660, 0.0004188732184101},
{7613421.8387281717732549, 0.0005633924671024, 0.0004816424113182},
{11063545.8319192007184029, 0.0006183669499131, 0.0005555185641204},
{14983493.9478362817317247, 0.0006850266861143, 0.0006040001521121},
{19391812.6190370135009289, 0.0007304890138530, 0.0006574016456013},
{24232323.6165447942912579, 0.0007823629157504, 0.0006963130120681},
{29524529.4551996700465679, 0.0008206123432838, 0.0007384676309123},
{35225541.6380422934889793, 0.0008632066674648, 0.0007709279432059},
{41352563.8334122523665428, 0.0008960897762171, 0.0008058585473406},
{47871592.5862829983234406, 0.0009322419720003, 0.0008337279631433},
{54797402.2767619341611862, 0.0009610293217512, 0.0008635981530303} },
{ {648250.4732224381295964, 0.0002365184250588, 0.0002181823727434},
{2392451.8839485393837094, 0.0004018180967795, 0.0002881397449816},
{4768281.1903005680069327, 0.0004641973191922, 0.0004153747957427},
{7709812.4598138714209199, 0.0005586879969443, 0.0004776197026781},
{11203616.9754030276089907, 0.0006132036959696, 0.0005508791089770},
{15173193.7942325212061405, 0.0006793072272357, 0.0005989558412241},
{19637323.5168368592858315, 0.0007243903007578, 0.0006519115763603},
{24539116.8388828299939632, 0.0007758315344013, 0.0006904980613602},
{29898322.7886153459548950, 0.0008137619982121, 0.0007323008537056},
{35671509.9388526007533073, 0.0008560011812060, 0.0007644901893845},
{41876099.5061185136437416, 0.0008886101741506, 0.0007991293014389},
{48477657.5225728452205658, 0.0009244610450562, 0.0008267661003703},
{55491145.2678978964686394, 0.0009530085032031, 0.0008563870617587} },
{ {656422.7629549243720248, 0.0002345513180840, 0.0002163677460250},
{2422614.3406009920872748, 0.0003984763912473, 0.0002857427971387},
{4828396.4489466547966003, 0.0004603369877592, 0.0004119200211225},
{7807013.1611887747421861, 0.0005540422059616, 0.0004736471839541},
{11344865.2966991420835257, 0.0006081048392344, 0.0005462975342010},
{15364487.9047178737819195, 0.0006736590961660, 0.0005939744608527},
{19884897.7147292904555798, 0.0007183676398691, 0.0006464899949956},
{24848488.3845207057893276, 0.0007693815938180, 0.0006847556501508},
{30275257.5146117880940437, 0.0008069970652966, 0.0007262110014521},
{36121226.1919276490807533, 0.0008488855279901, 0.0007581327391778},
{42404034.9944516196846962, 0.0008812238165051, 0.0007924839918196},
{49088815.8566533848643303, 0.0009167771117982, 0.0008198910736685},
{56190718.4324695318937302, 0.0009450876624315, 0.0008492659119852} },
{ {664663.4459882599767298, 0.0002326086508177, 0.0002145756648430},
{2453029.2256340296007693, 0.0003951762005845, 0.0002833756377991},
{4889014.8095732238143682, 0.0004565246117900, 0.0004085081713963},
{7905027.3335707644000649, 0.0005494541223228, 0.0004697240236273},
{11487295.7217567432671785, 0.0006030693132434, 0.0005417728809341},
{15557382.9497688952833414, 0.0006680811119699, 0.0005890549684008},
{20134543.8435394838452339, 0.0007124197719982, 0.0006411357670476},
{25160449.0316452793776989, 0.0007630117457507, 0.0006790845768790},
{30655346.7720760218799114, 0.0008003161306398, 0.0007201968000187},
{36574706.0759609863162041, 0.0008418582209139, 0.0007518542625116},
{42936388.7086666822433472, 0.0008739291600018, 0.0007859212283640},
{49705088.8667452111840248, 0.0009091885670552, 0.0008131014449183},
{56896146.2116920277476311, 0.0009372651450118, 0.0008422332142012} },
{ {672972.8044459408847615, 0.0002306900200401, 0.0002128057572989},
{2483697.5908638946712017, 0.0003919168399691, 0.0002810377753184},
{4950138.3683288050815463, 0.0004527594002344, 0.0004051385383873},
{8003858.3646587477996945, 0.0005449227942608, 0.0004658494073667},
{11630913.1754738315939903, 0.0005980960735921, 0.0005373042101245},
{15751885.5990490801632404, 0.0006625721175721, 0.0005841963432042},
{20386270.5551056936383247, 0.0007065454639084, 0.0006358477814011},
{25475009.5788272917270660, 0.0007567206697047, 0.0006734836647596},
{31038603.7134096994996071, 0.0007937178094681, 0.0007142570015392},
{37031965.2571687921881676, 0.0008349178036584, 0.0007456534567603},
{43473179.0556014180183411, 0.0008667246931590, 0.0007794396495554},
{50326497.8827067613601685, 0.0009016938387044, 0.0008063958056023},
{57607453.0070898905396461, 0.0009295393305908, 0.0008352875095842} },
{ {681351.1251116257626563, 0.0002287950308330, 0.0002110576590461},
{2514620.4878556905314326, 0.0003886976385883, 0.0002787287282589},
{5011769.2212594756856561, 0.0004490405783018, 0.0004018104284047},
{8103509.6473146267235279, 0.0005404472895150, 0.0004620225374636},
{11775722.5842030178755522, 0.0005931840972587, 0.0005328906019897},
{15948002.5231188982725143, 0.0006571309803460, 0.0005793975849167},
{20640086.4648746252059937, 0.0007007435076862, 0.0006306249498230},
{25792180.8219854421913624, 0.0007505070723157, 0.0006679517611899},
{31425041.4790950156748295, 0.0007872007454765, 0.0007083903837750},
{37493019.4430044814944267, 0.0008280628498315, 0.0007395290460305},
{44014424.4517693817615509, 0.0008596089355338, 0.0007730379218374},
{50953064.1948416903614998, 0.0008942913869213, 0.0007997727761577},
{58324663.1949404478073120, 0.0009219086320562, 0.0008284273692868} },
{ {689798.6924234863836318, 0.0002269232963573, 0.0002093310132152},
{2545798.9693347164429724, 0.0003855179394123, 0.0002764480249021},
{5073909.4647375224158168, 0.0004453673870316, 0.0003985231619937},
{8203984.5703106895089149, 0.0005360266949211, 0.0004582426326306},
{11921728.8724058289080858, 0.0005883323821537, 0.0005285311555395},
{16145740.4004211146384478, 0.0006517565903360, 0.0005746577140836},
{20896000.2116183079779148, 0.0006950127200882, 0.0006254662063050},
{26111973.5397600121796131, 0.0007443696866456, 0.0006624877371038},
{31814673.2114514969289303, 0.0007807636100146, 0.0007025957495303},
{37957884.2869433984160423, 0.0008212919621780, 0.0007334797805213},
{44560143.2933066785335541, 0.0008525804369091, 0.0007667147389617},
{51584809.1558341234922409, 0.0008869797032911, 0.0007932310051983},
{59047801.1551263481378555, 0.0009143714947075, 0.0008216513936944} },
{ {698315.7916444932343438, 0.0002250744376461, 0.0002076254702133},
{2577234.0849070665426552, 0.0003823770987613, 0.0002741952031779},
{5136561.1953980950638652, 0.0004417390829476, 0.0003952760735030},
{8305286.5192012637853622, 0.0005316601158825, 0.0004545089274612},
{12068936.9660246502608061, 0.0005835399465943, 0.0005242249882113},
{16345105.8938493337482214, 0.0006464478603497, 0.0005699757710915},
{21154020.4257718361914158, 0.0006893519419944, 0.0006203705065608},
{26434398.5192441903054714, 0.0007383072715363, 0.0006570904864706},
{32207512.0580698698759079, 0.0007744051015211, 0.0006968719258947},
{38426575.5093960538506508, 0.0008146037718424, 0.0007275044359037},
{45110354.0015514642000198, 0.0008456377765682, 0.0007604688212524},
{52221754.0487721636891365, 0.0008797573101082, 0.0007867691688764},
{59776891.2932402640581131, 0.0009069263954941, 0.0008149582116344} },
{ {706902.7066129709128290, 0.0002232480834311, 0.0002059406875077},
{2608926.8887099106796086, 0.0003792744860075, 0.0002719698104473},
{5199726.5086770467460155, 0.0004381549376358, 0.0003920685108084},
{8407418.8871709201484919, 0.0005273466759577, 0.0004508206720684},
{12217351.7904563434422016, 0.0005788058288350, 0.0005199712352336},
{16546105.6641615983098745, 0.0006412037251674, 0.0005653508157906},
{21414155.7506476975977421, 0.0006837600377229, 0.0006153368274892},
{26759466.5498562939465046, 0.0007323186109715, 0.0006517589256950},
{32603571.1534770131111145, 0.0007681239447857, 0.0006912177638387},
{38899108.7670296728610992, 0.0008079969377520, 0.0007216018126593},
{45665074.9871469363570213, 0.0008387795626180, 0.0007542989149741},
{52863920.2112730666995049, 0.0008726227596055, 0.0007803859701807},
{60511958.0167149528861046, 0.0008995718422205, 0.0008083464797937} },
{ {715559.7241816779132932, 0.0002214438699289, 0.0002042763295735},
{2640878.4320793999359012, 0.0003762094832732, 0.0002697714031735},
{5263407.5025149919092655, 0.0004346142373934, 0.0003888998349524},
{8510385.0624654348939657, 0.0005230855164034, 0.0004471771317144},
{12366978.2707421239465475, 0.0005741290864689, 0.0005157690494527},
{16748746.3994342628866434, 0.0006360231411734, 0.0005607819270594},
{21676414.7969718314707279, 0.0006782358945785, 0.0006103641666464},
{27087188.4202522337436676, 0.0007264025135100, 0.0006464919930236},
{33002863.6559749580919743, 0.0007619188903292, 0.0006856321375099},
{39375499.7319057360291481, 0.0008014701459082, 0.0007157707355311},
{46224324.6463025808334351, 0.0008320044312163, 0.0007482037917480},
{53511328.9008292257785797, 0.0008655746332589, 0.0007740801383295},
{61253025.6999124139547348, 0.0008923063727887, 0.0008018148819671} },
{ {724287.1278274311916903, 0.0002196614407003, 0.0002026320675344},
{2673089.7667099148966372, 0.0003731814851072, 0.0002675995467766},
{5327606.2715041041374207, 0.0004311162828769, 0.0003857694198535},
{8614188.4360294230282307, 0.0005188757957761, 0.0004435775864778},
{12517821.3328844942152500, 0.0005695087961348, 0.0005116176006190},
{16953034.7628224492073059, 0.0006309050856640, 0.0005562682022638},
{21940806.2241983935236931, 0.0006727784222365, 0.0006054515417739},
{27417574.9137091524899006, 0.0007205578116978, 0.0006412886481170},
{33405402.6933243162930012, 0.0007557887138900, 0.0006801139437028},
{39855764.1074378341436386, 0.0007950221086992, 0.0007100100528542},
{46788121.3961909487843513, 0.0008253110459829, 0.0007421822479071},
{54164001.4716186672449112, 0.0008586115410215, 0.0007678504280900},
{62000118.7657863944768906, 0.0008851285544852, 0.0007953621284409} },
{ {733085.2031269196886569, 0.0002179004464207, 0.0002010075791530},
{2705561.9422734510153532, 0.0003701898982184, 0.0002654538153592},
{5392324.9076409386470914, 0.0004276603887831, 0.0003826766519704},
{8718832.3933656010776758, 0.0005147166894725, 0.0004400213308587},
{12669885.9017477035522461, 0.0005649440529709, 0.0005075160752003},
{17158977.4255612455308437, 0.0006258485564312, 0.0005518087569172},
{22207338.6505091078579426, 0.0006673865522009, 0.0006005979902869},
{27750636.8311606906354427, 0.0007147833614036, 0.0006361478714929},
{33811201.4312300905585289, 0.0007497322156058, 0.0006746621013090},
{40339917.5579945445060730, 0.0007886515643248, 0.0007043186361138},
{47356483.6587688550353050, 0.0008186980972672, 0.0007362331038929},
{54821959.2220453619956970, 0.0008517321206903, 0.0007616956191647},
{62753261.5777419880032539, 0.0008780369833137, 0.0007889869553390} },
{ {741954.2344169778516516, 0.0002161605447769, 0.0001994025486079},
{2738296.0130600612610579, 0.0003672341411298, 0.0002633337915634},
{5457565.5189381781965494, 0.0004242458834657, 0.0003796209300037},
{8824320.3301945943385363, 0.0005106073893482, 0.0004365076733992},
{12823176.9029940795153379, 0.0005604339701012, 0.0005034636759092},
{17366581.0575849600136280, 0.0006208525713437, 0.0005474027241181},
{22476020.7073518708348274, 0.0006620592373154, 0.0005958025688330},
{28086384.9476013220846653, 0.0007090780413734, 0.0006310686639630},
{34220272.9881830364465714, 0.0007437482196322, 0.0006692755508587},
{40827975.7959663048386574, 0.0007823572761314, 0.0006986953792671},
{47929429.8105266541242599, 0.0008121643015609, 0.0007303552036735},
{55485223.4525013118982315, 0.0008449350372495, 0.0007556145156200},
{63512478.5453210994601250, 0.0008710302832451, 0.0007826881240102} },
{ {750894.5080168675631285, 0.0002144414002182, 0.0001978166664431},
{2771293.0301093822345138, 0.0003643136439337, 0.0002612390663201},
{5523330.1897782962769270, 0.0004208721086404, 0.0003766016646322},
{8930655.6292349323630333, 0.0005065471033206, 0.0004330359364836},
{12977699.2617083806544542, 0.0005559776783488, 0.0004994596212894},
{17575852.3302370756864548, 0.0006159161676849, 0.0005430492542852},
{22746861.0411599092185497, 0.0006567954511846, 0.0005910643528100},
{28424830.0556947775185108, 0.0007034407526049, 0.0006260500462727},
{34632630.5230924189090729, 0.0007378355734467, 0.0006639532538632},
{41319954.4487107023596764, 0.0007761380320536, 0.0006931391982799},
{48506978.2901017367839813, 0.0008057084008323, 0.0007245474142404},
{56153815.5128803774714470, 0.0008382189821711, 0.0007496059453185},
{64277794.0633587688207626, 0.0008641071055865, 0.0007764644204285} },
{ {759906.3078512359643355, 0.0002127426839038, 0.0001962496291894},
{2804554.0451275897212327, 0.0003614278480163, 0.0002591692386489},
{5589621.0231968350708485, 0.0004175384190090, 0.0003736182782433},
{9037841.6816425081342459, 0.0005025350550062, 0.0004296054558204},
{13133457.9035750087350607, 0.0005515743257015, 0.0004955031453606},
{17786797.9131879694759846, 0.0006110384018639, 0.0005387475145832},
{23019868.2718055583536625, 0.0006515941877867, 0.0005863824359063},
{28765982.9428277797996998, 0.0006978704178468, 0.0006210910585058},
{35048287.1734517887234688, 0.0007319931473296, 0.0006586941924583},
{41815869.2416603416204453, 0.0007699926439962, 0.0006876490305665},
{49089147.4903124570846558, 0.0007993291619450, 0.0007188086249797},
{56827756.6898068115115166, 0.0008315826728307, 0.0007436687593166},
{65049232.4962088167667389, 0.0008572661283280, 0.0007703146545811} },
{ {768989.9189146589487791, 0.0002110640734045, 0.0001947011394515},
{2838080.1100291395559907, 0.0003585762057653, 0.0002571239155074},
{5656440.1124755544587970, 0.0004142441821813, 0.0003706702044126},
{9145881.8800551667809486, 0.0004985704833137, 0.0004262155802975},
{13290457.7564158048480749, 0.0005472230769741, 0.0004915934972641},
{17999424.4776867255568504, 0.0006062183488660, 0.0005344966886649},
{23295051.0389421246945858, 0.0006464544609031, 0.0005817559297352},
{29109854.3974587358534336, 0.0006923659810968, 0.0006161907596877},
{35467256.0915134251117706, 0.0007262198338574, 0.0006534973688372},
{42315735.8607710301876068, 0.0007639199472912, 0.0006822238345208},
{49675955.8378265127539635, 0.0007930253760544, 0.0007131377471950},
{57507068.2595111355185509, 0.0008250248518756, 0.0007378018313126},
{65826818.3110776394605637, 0.0008505060554798, 0.0007642376599687} },
{ {778145.6263305360917002, 0.0002094052526659, 0.0001931709056891},
{2871872.2764387908391654, 0.0003557581803636, 0.0002551027115521},
{5723789.5576083976775408, 0.0004109887779339, 0.0003677568881637},
{9254779.6139589082449675, 0.0004946526421200, 0.0004228656715505},
{13448703.7433195747435093, 0.0005429231133645, 0.0004877299409001},
{18213738.6950016468763351, 0.0006014551018588, 0.0005302959761980},
{23572417.9667546562850475, 0.0006413753037001, 0.0005771839632923},
{29456455.2074270322918892, 0.0006869264070725, 0.0006113482273993},
{35889550.4166189655661583, 0.0007205145473457, 0.0006483618047936},
{42819569.9611127004027367, 0.0007579188001037, 0.0006768625890023},
{50267421.7227149009704590, 0.0007867958580921, 0.0007075337135804},
{58191771.5882538333535194, 0.0008185442867132, 0.0007320040570692},
{66610575.8133242428302765, 0.0008438256165351, 0.0007582322929505} },
{ {787373.7150356513448060, 0.0002077659118388, 0.0001916586420425},
{2905931.5962981404736638, 0.0003529732454559, 0.0002531052490104},
{5791671.4503395473584533, 0.0004077715983296, 0.0003648777851719},
{9364538.2633441817015409, 0.0004907807999003, 0.0004195551037290},
{13608200.7883209194988012, 0.0005386736321244, 0.0004839117545783},
{18429747.2326757982373238, 0.0005967477717376, 0.0005261445925444},
{23851977.6984016858041286, 0.0006363557682165, 0.0005726656827097},
{29805796.1598936095833778, 0.0006815506808555, 0.0006065625570904},
{36315183.2929013371467590, 0.0007148762233186, 0.0006432865412823},
{43327387.2350521311163902, 0.0007519880829841, 0.0006715642928546},
{50863563.5708588808774948, 0.0007806394460986, 0.0007019954777216},
{58881887.9739190265536308, 0.0008121397684804, 0.0007262743542851},
{67400529.4543447643518448, 0.0008372235657922, 0.0007522974322906} },
{ {796674.4700644515687600, 0.0002061457470811, 0.0001901640681672},
{2940259.1212907950393856, 0.0003502208849840, 0.0002511311574889},
{5860087.8887523068115115, 0.0004045920472822, 0.0003620323617851},
{9475161.2368991095572710, 0.0004869542393443, 0.0004162832632134},
{13768953.8213133551180363, 0.0005344738461056, 0.0004801382306404},
{18647456.7670348770916462, 0.0005920954867322, 0.0005220417683065},
{24133738.8698538094758987, 0.0006313949250201, 0.0005682002506853},
{30157888.0450420752167702, 0.0006762378069052, 0.0006018328622420},
{36744167.8624945282936096, 0.0007093038180583, 0.0006382706379137},
{43839203.3678787797689438, 0.0007461266982701, 0.0006663279643636},
{51464399.7919588983058929, 0.0007745550007897, 0.0006965220135852},
{59577438.7138122022151947, 0.0008058101124508, 0.0007206116612319},
{68196703.6143547147512436, 0.0008306986818182, 0.0007464319786253} }
};

}

#endif
