﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Stereo.GpGpuLib;
using VideoLib.Parameters;

namespace VideoLib.Stereo.GpGpu.StereoNodes.PreProcessors
{
	/// <summary>
	/// Residual pre-processing subtracts the local average from each pixel, to reduce radiometric differences
	/// between input images and to emphasise details over constant image areas.
	/// </summary>
	class BlurPreProcessor : PreProcessorStereoNode, IDisposable
	{
		#region Parameters

		/// <summary>
		/// Gets or sets the standard deviation of the Gaussian blur.
		/// </summary>
		[Parameter(Minimum = 0.0, Maximum = 5.0, Default = 0.7)]
		public float Sigma { get; set; }

		#endregion

		/// <summary>
		/// Initializes a new instance of the <see cref="BlurPreProcessor"/> class.
		/// </summary>
		public BlurPreProcessor() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="BlurPreProcessor"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public BlurPreProcessor(ExecutionStream stream) : base(stream) { }

		private InputImage temp = new InputImage();

		/// <summary>
		/// Pre-processes the given stereo image pair.
		/// </summary>
		public override void PreProcessImages(InputImage leftImage, InputImage rightImage)
		{
			ManagedPreProcessors.RunBlurKernel(leftImage, temp, leftImage, Sigma);
			ManagedPreProcessors.RunBlurKernel(rightImage, temp, rightImage, Sigma);
		}

		/// <summary>
		/// Frees temporary image data
		/// </summary>
		public void Dispose()
		{
			temp.Dispose();
		}
	}

	/// <summary>
	/// Factory that produces a <see cref="BlurPreProcessor"/>.
	/// </summary>
	class BlurPreProcessorFactory : StereoNodeFactory<PreProcessorStereoNode>
	{
		/// <summary>
		/// Creates a <see cref="PreProcessorStereoNode"/> to run on the specified execution stream.
		/// </summary>
		/// <param name="stream">The execution stream.</param>
		/// <returns>
		/// An instance of <see cref="PreProcessorStereoNode"/>.
		/// </returns>
		public override PreProcessorStereoNode Create(ExecutionStream stream)
		{
			return new BlurPreProcessor(stream);
		}

		/// <summary>
		/// Checks if the <see cref="ResidualPreProcessor"/> is compatible with the given datatypes.
		/// </summary>
		public override void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType)
		{
			isValid = (imageType == InputImageType.Xrgb_uint);
		}

		/// <summary>
		/// Gets the name of this node.
		/// </summary>
		public override string Name { get { return "Blur pre-processing"; } }
	}
}
